/**
 * EVAS Virtual Keyboard
 *
 * @package Everyone_Accessibility_Suite
 * @version 1.0.0
 */

(function($) {
    'use strict';

    const EVAS_Keyboard = {
        layouts: window.evasKeyboard?.layouts || {},
        specialKeys: window.evasKeyboard?.specialKeys || {},
        settings: window.evasKeyboard?.settings || {},
        
        currentLayout: null,
        shiftActive: false,
        capsLock: false,
        activeInput: null,
        isVisible: false,
        isEnabled: false,

        init() {
            this.currentLayout = this.settings.defaultLayout || 'qwerty';
            this.loadState();
            this.bindEvents();
            
            // Update button state when panel opens
            $(document).on('evas-panel-opened', () => {
                this.updateButton();
            });
        },

        loadState() {
            this.isEnabled = localStorage.getItem('evas_virtual_keyboard_enabled') === 'true';
        },

        saveState() {
            localStorage.setItem('evas_virtual_keyboard_enabled', this.isEnabled);
        },

        bindEvents() {
            // Toggle enable/disable
            $(document).on('click', '#evas-action-virtual-keyboard', () => {
                this.isEnabled = !this.isEnabled;
                this.saveState();
                this.updateButton();
                
                if (!this.isEnabled) {
                    this.hide();
                }
            });

            $(document).on('click', '#evas-keyboard-close', () => this.hide());
            $(document).on('change', '#evas-keyboard-layout', (e) => this.changeLayout(e.target.value));
            $(document).on('click', '.evas-key', (e) => this.handleKeyPress(e));

            // Show keyboard on input focus only if enabled
            $(document).on('focus', 'input[type="text"], input[type="search"], input[type="email"], input[type="password"], textarea', (e) => {
                if (this.isEnabled) {
                    this.activeInput = e.target;
                    this.show();
                }
            });

            $(document).on('keydown', (e) => {
                if (e.key === 'Escape' && this.isVisible) {
                    this.hide();
                }
            });
        },

        updateButton() {
            const $button = $('#evas-action-virtual-keyboard');
            if (this.isEnabled) {
                $button.addClass('evas-active').attr('aria-pressed', 'true');
            } else {
                $button.removeClass('evas-active').attr('aria-pressed', 'false');
            }
        },


        show() {
            this.renderKeyboard();
            $('#evas-keyboard-container').slideDown(200);
            this.isVisible = true;
        },

        hide() {
            $('#evas-keyboard-container').slideUp(200);
            this.isVisible = false;
        },

        changeLayout(layoutId) {
            this.currentLayout = layoutId;
            this.renderKeyboard();
        },

        renderKeyboard() {
            const layout = this.layouts[this.currentLayout];
            if (!layout) return;

            const $container = $('#evas-keyboard-keys');
            $container.empty();

            const rows = (this.shiftActive || this.capsLock) && layout.shift ? layout.shift : layout.rows;

            rows.forEach(row => {
                const $row = $('<div class="evas-keyboard-row">');
                
                row.forEach(key => {
                    const $key = this.createKey(key);
                    $row.append($key);
                });
                
                $container.append($row);
            });

            $('#evas-keyboard-layout').val(this.currentLayout);
        },

        createKey(key) {
            const specialKey = this.specialKeys[key];
            
            if (specialKey) {
                return $('<button>')
                    .addClass('evas-key evas-key-special')
                    .addClass(specialKey.class || '')
                    .addClass(key === '{shift}' && this.shiftActive ? 'evas-active' : '')
                    .attr('data-action', specialKey.action)
                    .attr('data-key', key)
                    .attr('type', 'button')
                    .text(specialKey.label);
            }

            return $('<button>')
                .addClass('evas-key')
                .attr('data-char', key)
                .attr('type', 'button')
                .text(key);
        },

        handleKeyPress(e) {
            const $key = $(e.currentTarget);
            const char = $key.data('char');
            const action = $key.data('action');

            if (this.settings.soundEnabled) {
                this.playClickSound();
            }

            $key.addClass('evas-pressed');
            setTimeout(() => $key.removeClass('evas-pressed'), 100);

            if (action) {
                this.handleAction(action);
            } else if (char) {
                this.insertCharacter(char);
            }
        },

        handleAction(action) {
            switch (action) {
                case 'shift':
                    this.shiftActive = !this.shiftActive;
                    this.renderKeyboard();
                    break;
                case 'capslock':
                    this.capsLock = !this.capsLock;
                    this.renderKeyboard();
                    break;
                case 'enter':
                    this.insertCharacter('\n');
                    if (this.activeInput && this.activeInput.form) {
                        // Submit form on enter for single-line inputs
                        if (this.activeInput.tagName === 'INPUT') {
                            $(this.activeInput.form).submit();
                        }
                    }
                    break;
                case 'backspace':
                    this.deleteCharacter();
                    break;
                case 'space':
                    this.insertCharacter(' ');
                    break;
                case 'tab':
                    this.insertCharacter('\t');
                    break;
                case 'close':
                    this.hide();
                    break;
            }
        },

        insertCharacter(char) {
            if (!this.activeInput) {
                this.activeInput = document.activeElement;
            }

            if (!this.activeInput || !this.isInputElement(this.activeInput)) {
                return;
            }

            const input = this.activeInput;
            const start = input.selectionStart || 0;
            const end = input.selectionEnd || 0;
            const value = input.value;

            input.value = value.slice(0, start) + char + value.slice(end);
            input.selectionStart = input.selectionEnd = start + char.length;

            // Trigger input event
            $(input).trigger('input');

            // Reset shift after character input
            if (this.shiftActive && !this.capsLock) {
                this.shiftActive = false;
                this.renderKeyboard();
            }
        },

        deleteCharacter() {
            if (!this.activeInput) return;

            const input = this.activeInput;
            const start = input.selectionStart || 0;
            const end = input.selectionEnd || 0;
            const value = input.value;

            if (start !== end) {
                input.value = value.slice(0, start) + value.slice(end);
                input.selectionStart = input.selectionEnd = start;
            } else if (start > 0) {
                input.value = value.slice(0, start - 1) + value.slice(end);
                input.selectionStart = input.selectionEnd = start - 1;
            }

            $(input).trigger('input');
        },

        isInputElement(el) {
            const tagName = el.tagName.toLowerCase();
            return tagName === 'input' || tagName === 'textarea';
        },

        playClickSound() {
            try {
                const ctx = new (window.AudioContext || window.webkitAudioContext)();
                const oscillator = ctx.createOscillator();
                const gain = ctx.createGain();

                oscillator.connect(gain);
                gain.connect(ctx.destination);

                oscillator.type = 'square';
                oscillator.frequency.value = 800;
                gain.gain.setValueAtTime(0.05, ctx.currentTime);
                gain.gain.exponentialRampToValueAtTime(0.01, ctx.currentTime + 0.05);

                oscillator.start(ctx.currentTime);
                oscillator.stop(ctx.currentTime + 0.05);
            } catch (e) {
                // Audio not available
            }
        }
    };

    $(document).ready(() => EVAS_Keyboard.init());
    window.EVAS_Keyboard = EVAS_Keyboard;

})(jQuery);

