<?php
/**
 * Usage Analytics Storage Class
 *
 * Handles database operations for storing and retrieving analytics data.
 *
 * @package Everyone_Accessibility_Suite
 * @since 1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Usage Analytics Storage class
 */
class EVAS_Usage_Analytics_Storage {

	/**
	 * Singleton instance
	 *
	 * @var EVAS_Usage_Analytics_Storage
	 */
	private static $instance;

	/**
	 * Table name (without prefix)
	 *
	 * @var string
	 */
	public static $table = 'evas_usage_analytics';

	/**
	 * Store usage analytics to the database (AJAX handler)
	 *
	 * @return void
	 */
	public function store_usage_analytics() {
		// Check security.
		if ( ! check_ajax_referer( 'evas_analytics', 'nonce', false ) ) {
			wp_send_json_error(
				esc_html__( 'Invalid security token.', 'everyone-accessibility-suite' )
			);
			wp_die();
		}

		// Get the body of the request.
		$request = file_get_contents( 'php://input' );
		$request = json_decode( $request, true );

		// Create table if needed.
		if ( ! self::create_table() ) {
			wp_send_json_error(
				esc_html__( 'Error creating analytics table.', 'everyone-accessibility-suite' )
			);
			wp_die();
		}

		// Store analytics.
		$status = self::put_event( $request );

		wp_send_json(
			array(
				'success' => $status,
				'data'    => ! $status ? esc_html__( 'Error saving analytics.', 'everyone-accessibility-suite' ) : '',
			)
		);
		wp_die();
	}

	/**
	 * Get usage analytics (AJAX handler)
	 *
	 * @return void
	 */
	public function get_usage_analytics() {
		// Check security.
		if ( ! check_ajax_referer( 'evas_analytics_admin', 'nonce', false ) ) {
			wp_send_json_error(
				esc_html__( 'Invalid security token.', 'everyone-accessibility-suite' )
			);
			wp_die();
		}

		// Check permissions.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error(
				esc_html__( 'Permission denied.', 'everyone-accessibility-suite' )
			);
			wp_die();
		}

		$key = isset( $_GET['key'] ) ? sanitize_text_field( wp_unslash( $_GET['key'] ) ) : '';

		if ( '' === $key ) {
			wp_send_json_error(
				esc_html__( 'Invalid data format.', 'everyone-accessibility-suite' )
			);
			wp_die();
		}

		// Create table if needed.
		if ( ! self::create_table() ) {
			wp_send_json_error(
				esc_html__( 'Error creating analytics table.', 'everyone-accessibility-suite' )
			);
			wp_die();
		}

		// Get request data.
		$period = ! empty( $_GET['period'] ) ? sanitize_text_field( wp_unslash( $_GET['period'] ) ) : 604800;
		if ( 'all' === $period ) {
			$period = time();
		}
		if ( ! is_numeric( $period ) ) {
			$period = 604800;
		}

		$sessions = self::get_rows_count( (int) $period );

		if ( ! $sessions ) {
			wp_send_json(
				array(
					'success' => true,
					'data'    => 0,
				)
			);
			wp_die();
		}

		$response = $this->process_analytics_request( $key, (int) $period, (int) $sessions );

		wp_send_json(
			array(
				'success' => true,
				'data'    => $response,
			)
		);
		wp_die();
	}

	/**
	 * Process analytics request based on key
	 *
	 * @param string $key     Analytics key.
	 * @param int    $period  Period in seconds.
	 * @param int    $sessions Total sessions count.
	 * @return mixed Response data.
	 */
	private function process_analytics_request( string $key, int $period, int $sessions ) {
		switch ( $key ) {
			case 'sessions':
				return self::digit_with_suffix( $sessions );

			case 'open':
				$open = self::get_total( 'open', $period );
				return number_format( $open / max( $sessions, 1 ), 2 );

			case 'open-timer':
				$open_timer = intval( self::get_total( 'open_timer', $period ) / 1000 );
				return self::seconds_to_time( $open_timer / max( $sessions, 1 ) );

			case 'usage':
				$used = self::get_rows_by_condition(
					array(
						array(
							'column'   => 'open',
							'value'    => '1',
							'operator' => '>=',
						),
						array(
							'column'   => 'load_saved',
							'value'    => '1',
							'operator' => '>=',
						),
					),
					'OR',
					$period
				);
				return number_format( ( $used / max( $sessions, 1 ) ) * 100, 2 ) . '%';

			case 'modes-chart':
				return self::get_columns_rating( $period );

			case 'profiles-chart':
				return self::get_profiles_rating( $period );

			case 'hourly-usage-chart':
				return $this->get_hourly_usage_data( $period );

			case 'mobile-chart':
				return $this->get_mobile_chart_data( $period );

			case 'country-code-chart':
				return $this->get_country_chart_data( $period );

			default:
				return false;
		}
	}

	/**
	 * Get hourly usage data for chart
	 *
	 * @param int $period Period in seconds.
	 * @return array Hourly usage data.
	 */
	private function get_hourly_usage_data( int $period ): array {
		$opened = self::get_column_values(
			'open',
			array(
				array(
					'column'   => 'open',
					'value'    => '1',
					'operator' => '>=',
				),
			),
			'AND',
			$period
		);

		$load_saved = self::get_column_values(
			'load_saved',
			array(
				array(
					'column'   => 'load_saved',
					'value'    => '1',
					'operator' => '>=',
				),
			),
			'AND',
			$period
		);

		return array(
			'hourlyOpen'      => array_reverse( self::get_hourly_rows_by_condition( $opened, 'open' ) ),
			'hourlyLoadSaved' => array_reverse( self::get_hourly_rows_by_condition( $load_saved, 'load_saved' ) ),
		);
	}

	/**
	 * Get mobile chart data
	 *
	 * @param int $period Period in seconds.
	 * @return array Mobile chart data.
	 */
	private function get_mobile_chart_data( int $period ): array {
		$raw_data = self::get_strings(
			'is_mobile',
			array(
				array(
					'column'   => 'open',
					'value'    => '1',
					'operator' => '>=',
				),
				array(
					'column'   => 'load_saved',
					'value'    => '1',
					'operator' => '>=',
				),
			),
			'AND',
			$period
		);

		$response = array();
		foreach ( $raw_data as $row ) {
			if ( '' === $row['is_mobile'] ) {
				$label = esc_html__( 'Unknown', 'everyone-accessibility-suite' );
			} elseif ( '1' === $row['is_mobile'] ) {
				$label = esc_html__( 'Mobile', 'everyone-accessibility-suite' );
			} else {
				$label = esc_html__( 'Desktop', 'everyone-accessibility-suite' );
			}

			$response[] = array(
				'label' => $label,
				'value' => $row['count'],
			);
		}

		return $response;
	}

	/**
	 * Get country chart data
	 *
	 * @param int $period Period in seconds.
	 * @return array Country chart data.
	 */
	private function get_country_chart_data( int $period ): array {
		$raw_data = self::get_strings(
			'country_code',
			array(
				array(
					'column'   => 'open',
					'value'    => '1',
					'operator' => '>=',
				),
				array(
					'column'   => 'load_saved',
					'value'    => '1',
					'operator' => '>=',
				),
			),
			'AND',
			$period
		);

		$response = array();
		foreach ( $raw_data as $row ) {
			$response[] = array(
				'label' => '' !== $row['country_code'] ? $row['country_code'] : esc_html__( 'Unknown', 'everyone-accessibility-suite' ),
				'value' => $row['count'],
			);
		}

		return $response;
	}

	/**
	 * Reset usage analytics (AJAX handler)
	 *
	 * @return void
	 */
	public function reset_usage_analytics() {
		// Check security.
		if ( ! check_ajax_referer( 'evas_analytics_admin', 'nonce', false ) ) {
			wp_send_json_error(
				esc_html__( 'Invalid security token.', 'everyone-accessibility-suite' )
			);
			wp_die();
		}

		// Check permissions.
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_send_json_error(
				esc_html__( 'Permission denied.', 'everyone-accessibility-suite' )
			);
			wp_die();
		}

		$status = EVAS_SQL_Helper::remove_table( self::$table );

		wp_send_json(
			array(
				'success' => $status,
				'data'    => $status
					? esc_html__( 'Analytics data cleared.', 'everyone-accessibility-suite' )
					: esc_html__( 'Error clearing analytics data.', 'everyone-accessibility-suite' ),
			)
		);
		wp_die();
	}

	/**
	 * Create the analytics table
	 *
	 * @return bool True on success.
	 */
	public static function create_table(): bool {
		if ( EVAS_SQL_Helper::is_table( self::$table ) ) {
			return true;
		}

		global $wpdb;

		$charset_collate = $wpdb->get_charset_collate();
		$table_name      = $wpdb->prefix . self::$table;

		$sql = "CREATE TABLE {$table_name} (
			id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
			uid varchar(255) NOT NULL,
			post_id bigint(20) unsigned NOT NULL,
			post_type varchar(255) NOT NULL,
			page_lang varchar(255) NOT NULL,
			country_code varchar(255) NOT NULL,
			is_mobile varchar(255) NOT NULL,
			open int(11) NOT NULL DEFAULT 0,
			open_timer int(11) NOT NULL DEFAULT 0,
			created int(11) NOT NULL,
			load_saved int(11) NOT NULL DEFAULT 0,
			PRIMARY KEY (id)
		) {$charset_collate};";

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery, WordPress.DB.PreparedSQL.NotPrepared
		$wpdb->query( $sql );

		return EVAS_SQL_Helper::is_table( self::$table );
	}

	/**
	 * Put (add or update) an event
	 *
	 * @param array $request Request data.
	 * @return bool|int Result.
	 */
	public static function put_event( array $request ) {
		global $wpdb;

		$uid = $request['uid'] ?? false;
		if ( ! $uid ) {
			return false;
		}

		$table_name = $wpdb->prefix . self::$table;

		// Check if record exists.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery
		$exists = $wpdb->get_var(
			$wpdb->prepare(
				"SELECT id FROM {$table_name} WHERE uid = %s", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
				$uid
			)
		);

		if ( ! $exists ) {
			return self::add_event( $uid, $request );
		} else {
			return self::update_event( $uid, $request );
		}
	}

	/**
	 * Add a new event
	 *
	 * @param string $uid     Unique identifier.
	 * @param array  $request Request data.
	 * @return bool|int Result.
	 */
	private static function add_event( string $uid, array $request ) {
		$post_id = $request['post_id'] ?? false;
		if ( ! $post_id ) {
			return false;
		}

		global $wpdb;

		$row_data = array(
			'uid'          => $uid,
			'post_id'      => intval( $request['post_id'] ),
			'post_type'    => $request['post_type'] ?? '',
			'page_lang'    => $request['page_lang'] ?? '',
			'country_code' => $request['country_code'] ?? '',
			'is_mobile'    => ! empty( $request['is_mobile'] ) ? 1 : 0,
			'open_timer'   => intval( $request['open_timer'] ?? 0 ),
			'created'      => time(),
		);

		// Process events.
		if ( isset( $request['events'] ) && is_array( $request['events'] ) ) {
			$row_data = self::events_processing( $request, $row_data );
		}

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery
		return $wpdb->insert( $wpdb->prefix . self::$table, $row_data );
	}

	/**
	 * Update an existing event
	 *
	 * @param string $uid          Unique identifier.
	 * @param array  $request_data Request data.
	 * @return bool|int Result.
	 */
	private static function update_event( string $uid, array $request_data ) {
		global $wpdb;

		$row_data = array(
			'open_timer' => $request_data['open_timer'] ?? 0,
		);

		if ( isset( $request_data['events'] ) && is_array( $request_data['events'] ) ) {
			$row_data = self::events_processing( $request_data, $row_data );
			$row_data = self::merge_with_db( $uid, $row_data );
		}

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery
		return $wpdb->update(
			$wpdb->prefix . self::$table,
			$row_data,
			array( 'uid' => $uid )
		);
	}

	/**
	 * Process events from request
	 *
	 * @param array $request  Request data.
	 * @param array $row_data Row data to update.
	 * @return array Updated row data.
	 */
	private static function events_processing( array $request, array $row_data ): array {
		$column_names = EVAS_SQL_Helper::column_names( self::$table );

		foreach ( $request['events'] as $event ) {
			$column_name  = $event['id'];
			$column_value = $event['value'];
			
			// Sanitize column name: replace special characters with underscores
			$safe_column_name = preg_replace( '/[^a-zA-Z0-9_]/', '_', $column_name );

			if ( ! in_array( $safe_column_name, $column_names, true ) && ! EVAS_SQL_Helper::is_column( self::$table, $safe_column_name ) ) {
				if ( EVAS_SQL_Helper::add_column( self::$table, $safe_column_name ) ) {
					$column_names[]                 = $safe_column_name;
					$row_data[ $safe_column_name ] = $column_value;
				}
			} else {
				if ( ! isset( $row_data[ $safe_column_name ] ) ) {
					$row_data[ $safe_column_name ] = 0;
				}
				$row_data[ $safe_column_name ] += $column_value;
			}
		}

		return $row_data;
	}

	/**
	 * Merge row data with existing database values
	 *
	 * @param string $uid      Unique identifier.
	 * @param array  $row_data Row data.
	 * @return array Merged data.
	 */
	private static function merge_with_db( string $uid, array $row_data ): array {
		$row = self::get_row( $uid );

		foreach ( $row_data as $column_name => $column_value ) {
			// Sum all values including open_timer.
			$row_data[ $column_name ] += $row[ $column_name ] ?? 0;
		}

		return $row_data;
	}

	/**
	 * Get a row by UID
	 *
	 * @param string $uid Unique identifier.
	 * @return array|null Row data.
	 */
	private static function get_row( string $uid ) {
		global $wpdb;

		$table_name = $wpdb->prefix . self::$table;

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery
		return $wpdb->get_row(
			$wpdb->prepare(
				"SELECT * FROM {$table_name} WHERE uid = %s", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
				$uid
			),
			ARRAY_A
		);
	}

	/**
	 * Get total for a column
	 *
	 * @param string $column_name Column name.
	 * @param int    $period      Period in seconds.
	 * @return string|null Total value.
	 */
	public static function get_total( string $column_name, int $period = 604800 ) {
		global $wpdb;

		$table_name  = esc_sql( $wpdb->prefix . self::$table );
		$timeframe  = time() - $period;
		$column_name = sanitize_key( $column_name );

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery
		return $wpdb->get_var(
			$wpdb->prepare(
				"SELECT SUM(`{$column_name}`) FROM `{$table_name}` WHERE created > %d", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
				$timeframe
			)
		);
	}

	/**
	 * Get strings for a column with count
	 *
	 * @param string $column_name Column name.
	 * @param array  $conditions  Conditions.
	 * @param string $operator    Operator.
	 * @param int    $period      Period in seconds.
	 * @return array Results.
	 */
	public static function get_strings( string $column_name, array $conditions = array(), string $operator = 'AND', int $period = 604800 ): array {
		global $wpdb;

		$table_name     = esc_sql( $wpdb->prefix . self::$table );
		$timeframe      = time() - $period;
		$column_name    = sanitize_key( $column_name );
		$prepared_cond  = EVAS_SQL_Helper::sql_conditions_prepared( $conditions, $operator );
		$conditions_sql = $prepared_cond['sql'];

		// Build prepared query.
		$sql  = "SELECT `{$column_name}`, COUNT(`{$column_name}`) as count FROM `{$table_name}` WHERE ";
		$args = array();

		if ( ! empty( $conditions_sql ) ) {
			$sql .= $conditions_sql . ' AND ';
			$args = array_merge( $args, $prepared_cond['args'] );
		}

		$sql .= 'created > %d GROUP BY `' . $column_name . '`';
		$args[] = $timeframe;

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery
		return $wpdb->get_results( $wpdb->prepare( $sql, $args ), ARRAY_A );
	}

	/**
	 * Get rows count by condition
	 *
	 * @param array  $conditions Conditions.
	 * @param string $operator   Operator.
	 * @param int    $period     Period in seconds.
	 * @return string|null Count.
	 */
	public static function get_rows_by_condition( array $conditions, string $operator = 'AND', int $period = 604800 ) {
		global $wpdb;

		$table_name     = esc_sql( $wpdb->prefix . self::$table );
		$timeframe      = time() - $period;
		$prepared_cond  = EVAS_SQL_Helper::sql_conditions_prepared( $conditions, $operator );
		$conditions_sql = $prepared_cond['sql'];

		$sql  = "SELECT COUNT(*) FROM `{$table_name}` WHERE created > %d";
		$args = array( $timeframe );

		if ( $conditions_sql ) {
			$sql  .= ' AND ' . $conditions_sql;
			$args = array_merge( $args, $prepared_cond['args'] );
		}

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery
		return $wpdb->get_var( $wpdb->prepare( $sql, $args ) );
	}

	/**
	 * Get total rows count
	 *
	 * @param int $period Period in seconds.
	 * @return string|null Count.
	 */
	public static function get_rows_count( int $period = 604800 ) {
		global $wpdb;

		$table_name = esc_sql( $wpdb->prefix . self::$table );
		$timeframe  = time() - $period;

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery
		return $wpdb->get_var(
			$wpdb->prepare(
				"SELECT COUNT(*) FROM `{$table_name}` WHERE created > %d", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
				$timeframe
			)
		);
	}

	/**
	 * Get columns rating (modes)
	 *
	 * @param int $period Period in seconds.
	 * @return array Columns rating.
	 */
	public static function get_columns_rating( int $period = 604800 ): array {
		$columns = EVAS_SQL_Helper::column_names( self::$table );

		if ( ! is_array( $columns ) ) {
			return array();
		}

		$skip_columns   = array( 'id', 'uid', 'post_id', 'post_type', 'page_lang', 'country_code', 'is_mobile', 'open_timer', 'created', 'open', 'load_saved' );
		$columns_rating = array();

		foreach ( $columns as $column_name ) {
			if ( in_array( $column_name, $skip_columns, true ) ) {
				continue;
			}
			if ( strpos( $column_name, 'profile_' ) !== false ) {
				continue;
			}
			$columns_rating[ $column_name ] = self::get_total( $column_name, $period );
		}

		asort( $columns_rating, SORT_NUMERIC );

		$top         = array();
		$translation = self::get_mode_translations();

		foreach ( array_reverse( $columns_rating ) as $slug => $count ) {
			$top[] = array(
				'slug'  => $slug,
				'value' => $count,
				'label' => $translation[ $slug ] ?? $slug,
			);
		}

		return $top;
	}

	/**
	 * Get profiles rating
	 *
	 * @param int $period Period in seconds.
	 * @return array Profiles rating.
	 */
	public static function get_profiles_rating( int $period = 604800 ): array {
		$columns = EVAS_SQL_Helper::column_names( self::$table );

		if ( ! is_array( $columns ) ) {
			return array();
		}

		$columns_rating = array();

		foreach ( $columns as $column_name ) {
			if ( strpos( $column_name, 'profile_' ) === false ) {
				continue;
			}
			$columns_rating[ $column_name ] = self::get_total( $column_name, $period );
		}

		asort( $columns_rating, SORT_NUMERIC );

		$top         = array();
		$translation = self::get_mode_translations();

		foreach ( array_reverse( $columns_rating ) as $slug => $count ) {
			$top[] = array(
				'slug'  => $slug,
				'value' => $count,
				'label' => $translation[ $slug ] ?? $slug,
			);
		}

		return $top;
	}

	/**
	 * Get column values by condition
	 *
	 * @param string $column_name Column name.
	 * @param array  $conditions  Conditions.
	 * @param string $operator    Operator.
	 * @param int    $period      Period in seconds.
	 * @return array Results.
	 */
	public static function get_column_values( string $column_name, array $conditions, string $operator = 'AND', int $period = 604800 ): array {
		global $wpdb;

		$table_name     = esc_sql( $wpdb->prefix . self::$table );
		$column_name    = sanitize_key( $column_name );
		$prepared_cond  = EVAS_SQL_Helper::sql_conditions_prepared( $conditions, $operator );
		$conditions_sql = $prepared_cond['sql'];
		$timeframe      = time() - $period;

		$sql  = "SELECT `{$column_name}`, created FROM `{$table_name}` WHERE ";
		$args = array();

		if ( ! empty( $conditions_sql ) ) {
			$sql .= $conditions_sql . ' AND ';
			$args = array_merge( $args, $prepared_cond['args'] );
		}

		$sql   .= 'created > %d';
		$args[] = $timeframe;

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery
		return $wpdb->get_results( $wpdb->prepare( $sql, $args ), ARRAY_A );
	}

	/**
	 * Get hourly rows by condition
	 *
	 * @param array  $row         Row data.
	 * @param string $column_name Column name.
	 * @param int    $split       Time split in seconds.
	 * @return array Hourly data.
	 */
	public static function get_hourly_rows_by_condition( array $row, string $column_name, int $split = HOUR_IN_SECONDS ): array {
		$current_time = time();
		$hourly       = array_fill( 0, 25, 0 );

		foreach ( $row as $columns ) {
			$hour = intval( ( $current_time - $columns['created'] ) / $split );
			if ( $hour > 24 ) {
				continue;
			}
			$hourly[ $hour ] += $columns[ $column_name ];
		}

		return $hourly;
	}

	/**
	 * Convert seconds to time format
	 *
	 * @param int|float $sec Seconds.
	 * @return string Formatted time.
	 */
	public static function seconds_to_time( $sec ): string {
		$sec     = intval( $sec );
		$hours   = floor( $sec / 3600 );
		$minutes = floor( ( $sec - ( $hours * 3600 ) ) / 60 );
		$secs    = $sec - ( $hours * 3600 ) - ( $minutes * 60 );

		if ( $hours > 0 ) {
			return sprintf( '%02d:%02d:%02d', $hours, $minutes, $secs );
		} elseif ( $minutes > 0 ) {
			return sprintf( '%02d:%02d', $minutes, $secs );
		} else {
			/* translators: %d: seconds */
			return sprintf( __( '%d s.', 'everyone-accessibility-suite' ), $secs );
		}
	}

	/**
	 * Convert digit to human readable format with suffix
	 *
	 * @param int|float $int Number.
	 * @return string Formatted number.
	 */
	public static function digit_with_suffix( $int ): string {
		$suffix = '';

		if ( $int >= 1000000000 ) {
			$int    = round( $int / 1000000000, 2 );
			$suffix = 'B';
		} elseif ( $int >= 1000000 ) {
			$int    = round( $int / 1000000, 2 );
			$suffix = 'M';
		} elseif ( $int >= 10000 ) {
			$int    = round( $int / 1000, 2 );
			$suffix = 'k';
		} else {
			$int = round( $int );
		}

		return $int . $suffix;
	}

	/**
	 * Get mode translations
	 *
	 * @return array Mode translations.
	 */
	private static function get_mode_translations(): array {
		return array(
			'profile_epilepsy'             => __( 'Epilepsy Safe Mode', 'everyone-accessibility-suite' ),
			'profile_visually_impaired'    => __( 'Visually Impaired Mode', 'everyone-accessibility-suite' ),
			'profile_cognitive_disability' => __( 'Cognitive Disability Mode', 'everyone-accessibility-suite' ),
			'profile_adhd_friendly'        => __( 'ADHD Friendly Mode', 'everyone-accessibility-suite' ),
			'profile_blind_users'          => __( 'Blindness Mode', 'everyone-accessibility-suite' ),
			'content_scaling'              => __( 'Content Scaling', 'everyone-accessibility-suite' ),
			'readable_font'                => __( 'Readable Font', 'everyone-accessibility-suite' ),
			'dyslexia_font'                => __( 'Dyslexia Friendly', 'everyone-accessibility-suite' ),
			'highlight_titles'             => __( 'Highlight Titles', 'everyone-accessibility-suite' ),
			'highlight_links'              => __( 'Highlight Links', 'everyone-accessibility-suite' ),
			'font_sizing'                  => __( 'Font Sizing', 'everyone-accessibility-suite' ),
			'line_height'                  => __( 'Line Height', 'everyone-accessibility-suite' ),
			'letter_spacing'               => __( 'Letter Spacing', 'everyone-accessibility-suite' ),
			'text_magnifier'               => __( 'Text Magnifier', 'everyone-accessibility-suite' ),
			'align_center'                 => __( 'Align Center', 'everyone-accessibility-suite' ),
			'align_left'                   => __( 'Align Left', 'everyone-accessibility-suite' ),
			'align_right'                  => __( 'Align Right', 'everyone-accessibility-suite' ),
			'dark_contrast'                => __( 'Dark Contrast', 'everyone-accessibility-suite' ),
			'light_contrast'               => __( 'Light Contrast', 'everyone-accessibility-suite' ),
			'monochrome'                   => __( 'Monochrome', 'everyone-accessibility-suite' ),
			'high_saturation'              => __( 'High Saturation', 'everyone-accessibility-suite' ),
			'high_contrast'                => __( 'High Contrast', 'everyone-accessibility-suite' ),
			'low_saturation'               => __( 'Low Saturation', 'everyone-accessibility-suite' ),
			'mute_sounds'                  => __( 'Mute Sounds', 'everyone-accessibility-suite' ),
			'hide_images'                  => __( 'Hide Images', 'everyone-accessibility-suite' ),
			'hide_emoji'                   => __( 'Hide Emoji', 'everyone-accessibility-suite' ),
			'virtual_keyboard'             => __( 'Virtual Keyboard', 'everyone-accessibility-suite' ),
			'reading_guide'                => __( 'Reading Guide', 'everyone-accessibility-suite' ),
			'cognitive_reading'            => __( 'Cognitive Reading', 'everyone-accessibility-suite' ),
			'useful_links'                 => __( 'Useful Links', 'everyone-accessibility-suite' ),
			'stop_animations'              => __( 'Stop Animations', 'everyone-accessibility-suite' ),
			'reading_mask'                 => __( 'Reading Mask', 'everyone-accessibility-suite' ),
			'highlight_hover'              => __( 'Highlight Hover', 'everyone-accessibility-suite' ),
			'highlight_focus'              => __( 'Highlight Focus', 'everyone-accessibility-suite' ),
			'big_black_cursor'             => __( 'Big Dark Cursor', 'everyone-accessibility-suite' ),
			'big_white_cursor'             => __( 'Big Light Cursor', 'everyone-accessibility-suite' ),
			'text_to_speech'               => __( 'Text to Speech', 'everyone-accessibility-suite' ),
			'voice_navigation'             => __( 'Voice Navigation', 'everyone-accessibility-suite' ),
			'keyboard_navigation'          => __( 'Keyboard Navigation', 'everyone-accessibility-suite' ),
		);
	}

	/**
	 * Get singleton instance
	 *
	 * @return EVAS_Usage_Analytics_Storage
	 */
	public static function get_instance(): EVAS_Usage_Analytics_Storage {
		if ( ! isset( self::$instance ) || ! ( self::$instance instanceof self ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}
}

