<?php
/**
 * Usage Analytics Module Class
 *
 * Main module class for handling usage analytics functionality.
 *
 * @package Everyone_Accessibility_Suite
 * @since 1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Usage Analytics Module class
 */
class EVAS_Usage_Analytics_Module {

	/**
	 * Singleton instance
	 *
	 * @var EVAS_Usage_Analytics_Module
	 */
	private static $instance;

	/**
	 * REST Controller instance
	 *
	 * @var EVAS_Usage_Analytics_REST_Controller
	 */
	private $rest_controller;

	/**
	 * Module settings
	 *
	 * @var array
	 */
	private $settings;

	/**
	 * Constructor
	 */
	private function __construct() {
		$this->settings = $this->get_settings();
		$this->init_hooks();
	}

	/**
	 * Get module settings
	 *
	 * @return array Settings.
	 */
	private function get_settings(): array {
		$settings = get_option( 'evas_usage_analytics_settings', array() );

		return wp_parse_args(
			$settings,
			array(
				'enabled'          => false,
				'send_interval'    => 2,
				'dashboard_widget' => true,
				'gdpr_strict'      => false,
			)
		);
	}

	/**
	 * Initialize hooks
	 *
	 * @return void
	 */
	private function init_hooks() {
		// Register REST API routes.
		add_action( 'rest_api_init', array( $this, 'register_rest_routes' ) );

		// Register admin route for navigation.
		add_filter( 'evas_admin_routes', array( $this, 'register_admin_route' ) );

		// Only proceed if analytics is enabled.
		if ( ! $this->settings['enabled'] ) {
			return;
		}

		// Enqueue frontend tracker script.
		add_action( 'wp_enqueue_scripts', array( $this, 'enqueue_frontend_scripts' ) );

		// Add dashboard widget.
		if ( $this->settings['dashboard_widget'] ) {
			add_action( 'wp_dashboard_setup', array( $this, 'add_dashboard_widget' ) );
		}

		// Add admin scripts for dashboard widget.
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_scripts' ) );
	}

	/**
	 * Register admin route for navigation
	 *
	 * @param array $routes Existing routes.
	 * @return array Modified routes.
	 */
	public function register_admin_route( array $routes ): array {
		$routes[] = array(
			'path'        => '/usage-analytics',
			'title'       => __( 'Usage Analytics', 'everyone-accessibility-suite' ),
			'menu_title'  => __( 'Usage Analytics', 'everyone-accessibility-suite' ),
			'icon'        => 'dashicons-chart-area',
			'description' => __( 'Track feature usage and user statistics', 'everyone-accessibility-suite' ),
			'module'      => 'usage_analytics',
		);
		return $routes;
	}

	/**
	 * Register REST API routes
	 *
	 * @return void
	 */
	public function register_rest_routes() {
		$this->rest_controller = new EVAS_Usage_Analytics_REST_Controller();
		$this->rest_controller->register_routes();
	}

	/**
	 * Enqueue frontend tracker script
	 *
	 * @return void
	 */
	public function enqueue_frontend_scripts() {
		// Check if panel is enabled.
		$panel_settings = get_option( 'evas_settings', array() );
		if ( empty( $panel_settings['enabled'] ) || 'on' !== $panel_settings['enabled'] ) {
			return;
		}

		wp_enqueue_script(
			'evas-analytics-tracker',
			EVAS_UA_URL . 'assets/js/analytics-tracker.js',
			array( 'evas-panel-features' ),
			EVAS_UA_VERSION,
			true
		);

		// Localize script with settings.
		wp_localize_script(
			'evas-analytics-tracker',
			'evasAnalyticsOptions',
			array(
				'ajaxurl'      => admin_url( 'admin-ajax.php' ),
				'restUrl'      => rest_url( 'evas/v1/usage-analytics/' ),
				'nonce'        => wp_create_nonce( 'wp_rest' ),
				'sendInterval' => $this->settings['send_interval'],
				'gdprStrict'   => $this->settings['gdpr_strict'],
				'postId'       => get_the_ID(),
				'postType'     => get_post_type(),
				'pageLang'     => get_locale(),
			)
		);
	}

	/**
	 * Enqueue admin scripts for dashboard widget
	 *
	 * @param string $hook Current admin page hook.
	 * @return void
	 */
	public function enqueue_admin_scripts( $hook ) {
		// Only on dashboard or our admin page.
		if ( 'index.php' !== $hook && strpos( $hook, 'evas' ) === false ) {
			return;
		}

		wp_enqueue_style(
			'evas-analytics-admin',
			EVAS_UA_URL . 'assets/css/analytics.css',
			array(),
			EVAS_UA_VERSION
		);

		wp_enqueue_script(
			'evas-analytics-admin',
			EVAS_UA_URL . 'assets/js/analytics-admin.js',
			array( 'jquery' ),
			EVAS_UA_VERSION,
			true
		);

		wp_localize_script(
			'evas-analytics-admin',
			'evasAnalyticsAdmin',
			array(
				'restUrl' => rest_url( 'evas/v1/usage-analytics/' ),
				'nonce'   => wp_create_nonce( 'wp_rest' ),
				'i18n'    => array(
					'loading'       => __( 'Loading...', 'everyone-accessibility-suite' ),
					'noData'        => __( 'No data available', 'everyone-accessibility-suite' ),
					'sessions'      => __( 'Sessions', 'everyone-accessibility-suite' ),
					'usage'         => __( 'Usage percentage', 'everyone-accessibility-suite' ),
					'openAvg'       => __( 'Open on average', 'everyone-accessibility-suite' ),
					'avgDuration'   => __( 'Average duration', 'everyone-accessibility-suite' ),
					'hour'          => __( 'Hour', 'everyone-accessibility-suite' ),
					'day'           => __( 'Day', 'everyone-accessibility-suite' ),
					'days7'         => __( '7 days', 'everyone-accessibility-suite' ),
					'days30'        => __( '30 days', 'everyone-accessibility-suite' ),
					'allTime'       => __( 'All time', 'everyone-accessibility-suite' ),
					'modes'         => __( 'Modes', 'everyone-accessibility-suite' ),
					'profiles'      => __( 'Profiles', 'everyone-accessibility-suite' ),
					'devices'       => __( 'Devices', 'everyone-accessibility-suite' ),
					'countries'     => __( 'Countries', 'everyone-accessibility-suite' ),
					'loadedSaved'   => __( 'Loaded with settings', 'everyone-accessibility-suite' ),
					'popupOpen'     => __( 'Popup is open', 'everyone-accessibility-suite' ),
				),
			)
		);
	}

	/**
	 * Add WordPress dashboard widget
	 *
	 * @return void
	 */
	public function add_dashboard_widget() {
		wp_add_dashboard_widget(
			'evas_usage_analytics',
			__( 'Accessibility Panel Usage', 'everyone-accessibility-suite' ),
			array( $this, 'render_dashboard_widget' )
		);
	}

	/**
	 * Render dashboard widget content
	 *
	 * @return void
	 */
	public function render_dashboard_widget() {
		?>
		<div class="evas-analytics-widget" id="evas-analytics-widget">
			<div class="evas-analytics-widget__header">
				<div class="evas-analytics-widget__sessions">
					<span class="dashicons dashicons-groups"></span>
					<span>
						<?php esc_html_e( 'Sessions:', 'everyone-accessibility-suite' ); ?>
						<strong data-evas-stat="sessions">--</strong>
					</span>
				</div>
				<div class="evas-analytics-widget__period">
					<select id="evas-analytics-period">
						<option value="3600"><?php esc_html_e( 'Hour', 'everyone-accessibility-suite' ); ?></option>
						<option value="86400"><?php esc_html_e( 'Day', 'everyone-accessibility-suite' ); ?></option>
						<option value="604800" selected><?php esc_html_e( '7 days', 'everyone-accessibility-suite' ); ?></option>
						<option value="2592000"><?php esc_html_e( '30 days', 'everyone-accessibility-suite' ); ?></option>
						<option value="all"><?php esc_html_e( 'All time', 'everyone-accessibility-suite' ); ?></option>
					</select>
				</div>
			</div>

			<div class="evas-analytics-widget__metrics">
				<div class="evas-analytics-widget__metric">
					<span class="evas-analytics-widget__metric-value" data-evas-stat="usage">--</span>
					<span class="evas-analytics-widget__metric-label"><?php esc_html_e( 'Usage percentage', 'everyone-accessibility-suite' ); ?></span>
				</div>
				<div class="evas-analytics-widget__metric">
					<span class="evas-analytics-widget__metric-value" data-evas-stat="open">--</span>
					<span class="evas-analytics-widget__metric-label"><?php esc_html_e( 'Open on average', 'everyone-accessibility-suite' ); ?></span>
				</div>
				<div class="evas-analytics-widget__metric">
					<span class="evas-analytics-widget__metric-value" data-evas-stat="openTimer">--</span>
					<span class="evas-analytics-widget__metric-label"><?php esc_html_e( 'Average duration', 'everyone-accessibility-suite' ); ?></span>
				</div>
			</div>

			<div class="evas-analytics-widget__chart">
				<div class="evas-analytics-widget__chart-legend">
					<span class="evas-analytics-widget__legend-item evas-analytics-widget__legend-item--loaded">
						<?php esc_html_e( 'Loaded with settings', 'everyone-accessibility-suite' ); ?>
					</span>
					<span class="evas-analytics-widget__legend-item evas-analytics-widget__legend-item--open">
						<?php esc_html_e( 'Popup is open', 'everyone-accessibility-suite' ); ?>
					</span>
				</div>
				<canvas id="evas-hourly-chart" width="400" height="150"></canvas>
			</div>

			<div class="evas-analytics-widget__tabs">
				<div class="evas-analytics-widget__tabs-header">
					<button class="evas-analytics-widget__tab is-active" data-tab="modes"><?php esc_html_e( 'Modes', 'everyone-accessibility-suite' ); ?></button>
					<button class="evas-analytics-widget__tab" data-tab="profiles"><?php esc_html_e( 'Profiles', 'everyone-accessibility-suite' ); ?></button>
					<button class="evas-analytics-widget__tab" data-tab="devices"><?php esc_html_e( 'Devices', 'everyone-accessibility-suite' ); ?></button>
					<?php if ( ! $this->settings['gdpr_strict'] ) : ?>
						<button class="evas-analytics-widget__tab" data-tab="countries"><?php esc_html_e( 'Countries', 'everyone-accessibility-suite' ); ?></button>
					<?php endif; ?>
				</div>
				<div class="evas-analytics-widget__tabs-content">
					<div class="evas-analytics-widget__tab-pane is-active" data-tab-content="modes">
						<div class="evas-analytics-widget__chart-container" id="evas-modes-chart"></div>
					</div>
					<div class="evas-analytics-widget__tab-pane" data-tab-content="profiles">
						<div class="evas-analytics-widget__chart-container" id="evas-profiles-chart"></div>
					</div>
					<div class="evas-analytics-widget__tab-pane" data-tab-content="devices">
						<div class="evas-analytics-widget__chart-container" id="evas-devices-chart"></div>
					</div>
					<?php if ( ! $this->settings['gdpr_strict'] ) : ?>
						<div class="evas-analytics-widget__tab-pane" data-tab-content="countries">
							<div class="evas-analytics-widget__chart-container" id="evas-countries-chart"></div>
						</div>
					<?php endif; ?>
				</div>
			</div>
		</div>
		<?php
	}

	/**
	 * Check if module is enabled
	 *
	 * @return bool
	 */
	public function is_enabled(): bool {
		return $this->settings['enabled'];
	}

	/**
	 * Get singleton instance
	 *
	 * @return EVAS_Usage_Analytics_Module
	 */
	public static function get_instance(): EVAS_Usage_Analytics_Module {
		if ( ! isset( self::$instance ) || ! ( self::$instance instanceof self ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}
}

