<?php
/**
 * SQL Helper Class
 *
 * Utility class for SQL operations on custom tables.
 *
 * @package Everyone_Accessibility_Suite
 * @since 1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * SQL Helper class for database operations
 */
class EVAS_SQL_Helper {

	/**
	 * Prepare conditions for SQL query
	 *
	 * @param array  $conditions Array of conditions.
	 * @param string $operator   Logical operator (AND/OR).
	 * @return string SQL conditions string.
	 */
	public static function sql_conditions( array $conditions, string $operator = 'AND' ): string {
		$conditions_sql = array();

		foreach ( $conditions as $condition ) {
			if ( ! is_array( $condition ) ) {
				continue;
			}

			$column_name     = $condition['column'] ?? '';
			$column_value    = $condition['value'] ?? '';
			$column_operator = $condition['operator'] ?? '=';

			if ( '' === $column_name || '' === $column_value ) {
				continue;
			}

			$conditions_sql[] = "{$column_name} {$column_operator} {$column_value}";
		}

		if ( count( $conditions_sql ) > 1 ) {
			return '(' . implode( " {$operator} ", $conditions_sql ) . ')';
		}

		return implode( " {$operator} ", $conditions_sql );
	}

	/**
	 * Prepare conditions for SQL query with placeholders.
	 *
	 * Returns an array with SQL fragment and args for $wpdb->prepare().
	 *
	 * @param array  $conditions Array of conditions.
	 * @param string $operator   Logical operator (AND/OR).
	 * @return array{sql:string,args:array} Prepared conditions.
	 */
	public static function sql_conditions_prepared( array $conditions, string $operator = 'AND' ): array {
		$operator = strtoupper( $operator );
		if ( ! in_array( $operator, array( 'AND', 'OR' ), true ) ) {
			$operator = 'AND';
		}

		$allowed_ops = array( '=', '!=', '>', '>=', '<', '<=', 'LIKE' );

		$sql_parts = array();
		$args      = array();

		foreach ( $conditions as $condition ) {
			if ( ! is_array( $condition ) ) {
				continue;
			}

			$column_name     = isset( $condition['column'] ) ? sanitize_key( (string) $condition['column'] ) : '';
			$column_operator = isset( $condition['operator'] ) ? strtoupper( (string) $condition['operator'] ) : '=';
			$column_value    = $condition['value'] ?? null;

			if ( '' === $column_name ) {
				continue;
			}

			if ( ! in_array( $column_operator, $allowed_ops, true ) ) {
				$column_operator = '=';
			}

			// Choose placeholder by value type.
			if ( is_int( $column_value ) || ( is_string( $column_value ) && ctype_digit( $column_value ) ) ) {
				$placeholder = '%d';
				$args[]      = (int) $column_value;
			} elseif ( is_float( $column_value ) || is_numeric( $column_value ) ) {
				$placeholder = '%f';
				$args[]      = (float) $column_value;
			} else {
				$placeholder = '%s';
				$args[]      = (string) $column_value;
			}

			// Identifiers cannot be placeholders; sanitize the column name and wrap in backticks.
			$sql_parts[] = "`{$column_name}` {$column_operator} {$placeholder}";
		}

		if ( empty( $sql_parts ) ) {
			return array(
				'sql'  => '',
				'args' => array(),
			);
		}

		$sql = implode( " {$operator} ", $sql_parts );
		if ( count( $sql_parts ) > 1 ) {
			$sql = '(' . $sql . ')';
		}

		return array(
			'sql'  => $sql,
			'args' => $args,
		);
	}

	/**
	 * Check if the table exists in the database
	 *
	 * @param string $table Table name (without prefix).
	 * @return bool True if table exists.
	 */
	public static function is_table( string $table ): bool {
		if ( empty( $table ) ) {
			return false;
		}

		global $wpdb;

		$safe_table = preg_replace( '/[^a-zA-Z0-9_]/', '', $table );
		$table_name = $wpdb->prefix . $safe_table;

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery
		return $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $wpdb->esc_like( $table_name ) ) ) === $table_name;
	}

	/**
	 * Get all columns from a table
	 *
	 * @param string $table Table name (without prefix).
	 * @return array|false Array of column objects or false on failure.
	 */
	public static function all_columns( string $table ) {
		if ( empty( $table ) ) {
			return false;
		}

		global $wpdb;

		$safe_table = preg_replace( '/[^a-zA-Z0-9_]/', '', $table );
		$table_name = esc_sql( $wpdb->prefix . $safe_table );

		// Identifiers cannot be passed via $wpdb->prepare(). We sanitize/escape them above.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		return $wpdb->get_results( "SHOW COLUMNS FROM `{$table_name}`" );
	}

	/**
	 * Get all column names from a table
	 *
	 * @param string $table Table name (without prefix).
	 * @return array|false Array of column names or false on failure.
	 */
	public static function column_names( string $table ) {
		if ( empty( $table ) ) {
			return false;
		}

		$columns = self::all_columns( $table );

		if ( ! $columns || ! is_array( $columns ) ) {
			return array();
		}

		return array_map(
			function ( $column ) {
				return $column->Field;
			},
			$columns
		);
	}

	/**
	 * Check if a column exists in the table
	 *
	 * @param string $table       Table name (without prefix).
	 * @param string $column_name Column name to check.
	 * @return string|null Column name if exists, null otherwise.
	 */
	public static function is_column( string $table, string $column_name ) {
		if ( empty( $table ) ) {
			return null;
		}

		global $wpdb;

		$safe_table  = preg_replace( '/[^a-zA-Z0-9_]/', '', $table );
		$table_name  = esc_sql( $wpdb->prefix . $safe_table );
		$column_name = sanitize_key( $column_name );

		// Identifiers cannot be passed via $wpdb->prepare(). We sanitize/escape them above.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		return $wpdb->get_var( $wpdb->prepare( "SHOW COLUMNS FROM `{$table_name}` LIKE %s", $column_name ) );
	}

	/**
	 * Add a column to the table
	 *
	 * @param string $table       Table name (without prefix).
	 * @param string $column_name Column name to add.
	 * @param string $column_type Column type (default: int(11)).
	 * @param string $default     Default value (default: '0').
	 * @return bool|int|mysqli_result|null Result of the query.
	 */
	public static function add_column( string $table, string $column_name, string $column_type = 'int(11)', string $default = '0' ) {
		if ( empty( $table ) ) {
			return false;
		}

		global $wpdb;

		$safe_table = preg_replace( '/[^a-zA-Z0-9_]/', '', $table );
		$table_name = esc_sql( $wpdb->prefix . $safe_table );
		
		// Sanitize column name: replace special characters with underscores
		$safe_column_name = preg_replace( '/[^a-zA-Z0-9_]/', '_', $column_name );

		// Column type/default are controlled by code (not user input). Identifiers cannot be placeholders.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		return $wpdb->query( "ALTER TABLE `{$table_name}` ADD `{$safe_column_name}` {$column_type} NOT NULL DEFAULT {$default}" );
	}

	/**
	 * Remove a table from the database
	 *
	 * @param string $table Table name (without prefix).
	 * @return bool True on success, false on failure.
	 */
	public static function remove_table( string $table = '' ): bool {
		if ( empty( $table ) ) {
			return false;
		}

		global $wpdb;

		$safe_table = preg_replace( '/[^a-zA-Z0-9_]/', '', $table );
		$table_name = $wpdb->prefix . $safe_table;

		// Check if table exists.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery
		if ( $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $wpdb->esc_like( $table_name ) ) ) === $table_name ) {
			$table_name_escaped = esc_sql( $table_name );
			// Identifiers cannot be placeholders.
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
			$wpdb->query( "DROP TABLE `{$table_name_escaped}`;" );
			return true;
		}

		return false;
	}
}

