<?php
/**
 * Usage Analytics REST Controller
 *
 * REST API endpoints for usage analytics tracking and retrieval.
 *
 * @package Everyone_Accessibility_Suite
 * @since 1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * REST Controller class for Usage Analytics
 */
class EVAS_Usage_Analytics_REST_Controller {

	/**
	 * Namespace for the REST API
	 *
	 * @var string
	 */
	private $namespace = 'evas/v1';

	/**
	 * Storage instance
	 *
	 * @var EVAS_Usage_Analytics_Storage
	 */
	private $storage;

	/**
	 * Constructor
	 */
	public function __construct() {
		$this->storage = EVAS_Usage_Analytics_Storage::get_instance();
	}

	/**
	 * Register REST API routes
	 *
	 * @return void
	 */
	public function register_routes() {
		// Track usage (public endpoint for frontend)
		register_rest_route(
			$this->namespace,
			'/usage-analytics/track',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'track_usage' ),
				'permission_callback' => '__return_true', // Public endpoint.
			)
		);

		// Get statistics (admin only)
		register_rest_route(
			$this->namespace,
			'/usage-analytics/stats',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_stats' ),
				'permission_callback' => array( $this, 'check_admin_permissions' ),
				'args'                => array(
					'key'    => array(
						'required'          => true,
						'type'              => 'string',
						'sanitize_callback' => 'sanitize_text_field',
					),
					'period' => array(
						'required'          => false,
						'type'              => 'string',
						'default'           => '604800',
						'sanitize_callback' => 'sanitize_text_field',
					),
				),
			)
		);

		// Get all stats summary (admin only)
		register_rest_route(
			$this->namespace,
			'/usage-analytics/summary',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_summary' ),
				'permission_callback' => array( $this, 'check_admin_permissions' ),
				'args'                => array(
					'period' => array(
						'required'          => false,
						'type'              => 'string',
						'default'           => '604800',
						'sanitize_callback' => 'sanitize_text_field',
					),
				),
			)
		);

		// Reset analytics (admin only)
		register_rest_route(
			$this->namespace,
			'/usage-analytics/reset',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'reset_analytics' ),
				'permission_callback' => array( $this, 'check_admin_permissions' ),
			)
		);

		// Get/update module settings (admin only)
		register_rest_route(
			$this->namespace,
			'/usage-analytics/settings',
			array(
				array(
					'methods'             => 'GET',
					'callback'            => array( $this, 'get_settings' ),
					'permission_callback' => array( $this, 'check_admin_permissions' ),
				),
				array(
					'methods'             => 'POST',
					'callback'            => array( $this, 'update_settings' ),
					'permission_callback' => array( $this, 'check_admin_permissions' ),
				),
			)
		);
	}

	/**
	 * Check admin permissions
	 *
	 * @return bool
	 */
	public function check_admin_permissions(): bool {
		return current_user_can( 'manage_options' );
	}

	/**
	 * Track usage endpoint
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response
	 */
	public function track_usage( WP_REST_Request $request ) {
		// Verify nonce from header.
		$nonce = $request->get_header( 'X-WP-Nonce' );
		if ( ! $nonce || ! wp_verify_nonce( $nonce, 'wp_rest' ) ) {
			// Also check custom nonce.
			$custom_nonce = $request->get_param( 'nonce' );
			if ( ! $custom_nonce || ! wp_verify_nonce( $custom_nonce, 'evas_analytics' ) ) {
				return new WP_REST_Response(
					array(
						'success' => false,
						'message' => __( 'Invalid security token.', 'everyone-accessibility-suite' ),
					),
					403
				);
			}
		}

		// Get request body.
		$body = $request->get_json_params();
		if ( empty( $body ) ) {
			$body = $request->get_body_params();
		}

		// Ensure table exists.
		if ( ! EVAS_Usage_Analytics_Storage::create_table() ) {
			return new WP_REST_Response(
				array(
					'success' => false,
					'message' => __( 'Error creating analytics table.', 'everyone-accessibility-suite' ),
				),
				500
			);
		}

		// Store event.
		$result = EVAS_Usage_Analytics_Storage::put_event( $body );

		return new WP_REST_Response(
			array(
				'success' => (bool) $result,
				'message' => $result ? '' : __( 'Error saving analytics.', 'everyone-accessibility-suite' ),
			),
			$result ? 200 : 500
		);
	}

	/**
	 * Get statistics endpoint
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response
	 */
	public function get_stats( WP_REST_Request $request ) {
		$key    = $request->get_param( 'key' );
		$period = $request->get_param( 'period' );

		// Convert period.
		if ( 'all' === $period ) {
			$period = time();
		} else {
			$period = absint( $period );
		}

		// Ensure table exists.
		if ( ! EVAS_Usage_Analytics_Storage::create_table() ) {
			return new WP_REST_Response(
				array(
					'success' => false,
					'data'    => null,
					'message' => __( 'Error creating analytics table.', 'everyone-accessibility-suite' ),
				),
				500
			);
		}

		$sessions = EVAS_Usage_Analytics_Storage::get_rows_count( $period );

		if ( ! $sessions ) {
			return new WP_REST_Response(
				array(
					'success' => true,
					'data'    => 0,
				),
				200
			);
		}

		$data = $this->process_stats_request( $key, $period, (int) $sessions );

		return new WP_REST_Response(
			array(
				'success' => true,
				'data'    => $data,
			),
			200
		);
	}

	/**
	 * Process statistics request
	 *
	 * @param string $key      Stats key.
	 * @param int    $period   Period in seconds.
	 * @param int    $sessions Total sessions.
	 * @return mixed Stats data.
	 */
	private function process_stats_request( string $key, int $period, int $sessions ) {
		switch ( $key ) {
			case 'sessions':
				return EVAS_Usage_Analytics_Storage::digit_with_suffix( $sessions );

			case 'open':
				$open = EVAS_Usage_Analytics_Storage::get_total( 'open', $period );
				return number_format( $open / max( $sessions, 1 ), 2 );

			case 'open-timer':
				$open_timer = intval( EVAS_Usage_Analytics_Storage::get_total( 'open_timer', $period ) / 1000 );
				return EVAS_Usage_Analytics_Storage::seconds_to_time( $open_timer / max( $sessions, 1 ) );

			case 'usage':
				$used = EVAS_Usage_Analytics_Storage::get_rows_by_condition(
					array(
						array(
							'column'   => 'open',
							'value'    => '1',
							'operator' => '>=',
						),
						array(
							'column'   => 'load_saved',
							'value'    => '1',
							'operator' => '>=',
						),
					),
					'OR',
					$period
				);
				return number_format( ( $used / max( $sessions, 1 ) ) * 100, 2 ) . '%';

			case 'modes-chart':
				return EVAS_Usage_Analytics_Storage::get_columns_rating( $period );

			case 'profiles-chart':
				return EVAS_Usage_Analytics_Storage::get_profiles_rating( $period );

			case 'hourly-usage-chart':
				return $this->get_hourly_data( $period );

			case 'mobile-chart':
			case 'devices-chart':
				return $this->get_mobile_data( $period );

			case 'country-code-chart':
			case 'countries-chart':
				return $this->get_country_data( $period );

			default:
				return null;
		}
	}

	/**
	 * Get hourly usage data
	 *
	 * @param int $period Period in seconds.
	 * @return array Hourly data.
	 */
	private function get_hourly_data( int $period ): array {
		$opened = EVAS_Usage_Analytics_Storage::get_column_values(
			'open',
			array(
				array(
					'column'   => 'open',
					'value'    => '1',
					'operator' => '>=',
				),
			),
			'AND',
			$period
		);

		$load_saved = EVAS_Usage_Analytics_Storage::get_column_values(
			'load_saved',
			array(
				array(
					'column'   => 'load_saved',
					'value'    => '1',
					'operator' => '>=',
				),
			),
			'AND',
			$period
		);

		return array(
			'hourlyOpen'      => array_reverse( EVAS_Usage_Analytics_Storage::get_hourly_rows_by_condition( $opened, 'open' ) ),
			'hourlyLoadSaved' => array_reverse( EVAS_Usage_Analytics_Storage::get_hourly_rows_by_condition( $load_saved, 'load_saved' ) ),
		);
	}

	/**
	 * Get mobile chart data
	 *
	 * @param int $period Period in seconds.
	 * @return array Mobile data.
	 */
	private function get_mobile_data( int $period ): array {
		$raw_data = EVAS_Usage_Analytics_Storage::get_strings(
			'is_mobile',
			array(
				array(
					'column'   => 'open',
					'value'    => '1',
					'operator' => '>=',
				),
				array(
					'column'   => 'load_saved',
					'value'    => '1',
					'operator' => '>=',
				),
			),
			'OR',
			$period
		);

		$response = array();
		foreach ( $raw_data as $row ) {
			if ( '' === $row['is_mobile'] ) {
				$label = __( 'Unknown', 'everyone-accessibility-suite' );
			} elseif ( '1' === $row['is_mobile'] ) {
				$label = __( 'Mobile', 'everyone-accessibility-suite' );
			} else {
				$label = __( 'Desktop', 'everyone-accessibility-suite' );
			}

			$response[] = array(
				'label' => $label,
				'value' => $row['count'],
			);
		}

		return $response;
	}

	/**
	 * Get country chart data
	 *
	 * @param int $period Period in seconds.
	 * @return array Country data.
	 */
	private function get_country_data( int $period ): array {
		$raw_data = EVAS_Usage_Analytics_Storage::get_strings(
			'country_code',
			array(
				array(
					'column'   => 'open',
					'value'    => '1',
					'operator' => '>=',
				),
				array(
					'column'   => 'load_saved',
					'value'    => '1',
					'operator' => '>=',
				),
			),
			'OR',
			$period
		);

		$response = array();
		foreach ( $raw_data as $row ) {
			$response[] = array(
				'label' => '' !== $row['country_code'] ? $row['country_code'] : __( 'Unknown', 'everyone-accessibility-suite' ),
				'value' => $row['count'],
			);
		}

		return $response;
	}

	/**
	 * Get summary endpoint
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response
	 */
	public function get_summary( WP_REST_Request $request ) {
		$period = $request->get_param( 'period' );

		if ( 'all' === $period ) {
			$period = time();
		} else {
			$period = absint( $period );
		}

		// Ensure table exists.
		if ( ! EVAS_Usage_Analytics_Storage::create_table() ) {
			return new WP_REST_Response(
				array(
					'success' => false,
					'data'    => null,
				),
				500
			);
		}

		$sessions = EVAS_Usage_Analytics_Storage::get_rows_count( $period );

		if ( ! $sessions ) {
			return new WP_REST_Response(
				array(
					'success' => true,
					'data'    => array(
						'sessions'     => '0',
						'usage'        => '0%',
						'open'         => '0',
						'openTimer'    => '0 s.',
						'hasData'      => false,
					),
				),
				200
			);
		}

		$sessions_int = (int) $sessions;

		// Calculate usage percentage.
		$used = EVAS_Usage_Analytics_Storage::get_rows_by_condition(
			array(
				array(
					'column'   => 'open',
					'value'    => '1',
					'operator' => '>=',
				),
				array(
					'column'   => 'load_saved',
					'value'    => '1',
					'operator' => '>=',
				),
			),
			'OR',
			$period
		);

		$open       = EVAS_Usage_Analytics_Storage::get_total( 'open', $period );
		$open_timer = intval( EVAS_Usage_Analytics_Storage::get_total( 'open_timer', $period ) / 1000 );

		return new WP_REST_Response(
			array(
				'success' => true,
				'data'    => array(
					'sessions'     => EVAS_Usage_Analytics_Storage::digit_with_suffix( $sessions_int ),
					'usage'        => number_format( ( $used / max( $sessions_int, 1 ) ) * 100, 2 ) . '%',
					'open'         => number_format( $open / max( $sessions_int, 1 ), 2 ),
					'openTimer'    => EVAS_Usage_Analytics_Storage::seconds_to_time( $open_timer / max( $sessions_int, 1 ) ),
					'hasData'      => true,
				),
			),
			200
		);
	}

	/**
	 * Reset analytics endpoint
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response
	 */
	public function reset_analytics( WP_REST_Request $request ) {
		$result = EVAS_SQL_Helper::remove_table( EVAS_Usage_Analytics_Storage::$table );

		return new WP_REST_Response(
			array(
				'success' => $result,
				'message' => $result
					? __( 'Analytics data cleared successfully.', 'everyone-accessibility-suite' )
					: __( 'Error clearing analytics data.', 'everyone-accessibility-suite' ),
			),
			$result ? 200 : 500
		);
	}

	/**
	 * Get module settings
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response
	 */
	public function get_settings( WP_REST_Request $request ) {
		$settings = get_option( 'evas_usage_analytics_settings', array() );

		$defaults = array(
			'enabled'               => false,
			'send_interval'         => 2,
			'dashboard_widget'      => true,
			'gdpr_strict'           => false,
		);

		$settings = wp_parse_args( $settings, $defaults );

		return new WP_REST_Response(
			array(
				'success'  => true,
				'settings' => $settings,
			),
			200
		);
	}

	/**
	 * Update module settings
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response
	 */
	public function update_settings( WP_REST_Request $request ) {
		$body = $request->get_json_params();

		$settings = array(
			'enabled'          => ! empty( $body['enabled'] ),
			'send_interval'    => isset( $body['send_interval'] ) ? absint( $body['send_interval'] ) : 2,
			'dashboard_widget' => isset( $body['dashboard_widget'] ) ? (bool) $body['dashboard_widget'] : true,
			'gdpr_strict'      => ! empty( $body['gdpr_strict'] ),
		);

		// Validate send_interval.
		if ( $settings['send_interval'] < 1 ) {
			$settings['send_interval'] = 1;
		}
		if ( $settings['send_interval'] > 60 ) {
			$settings['send_interval'] = 60;
		}

		$result = update_option( 'evas_usage_analytics_settings', $settings );

		return new WP_REST_Response(
			array(
				'success'  => true,
				'settings' => $settings,
				'message'  => __( 'Settings saved successfully.', 'everyone-accessibility-suite' ),
			),
			200
		);
	}
}

