/**
 * Usage Analytics Tracker
 *
 * Frontend tracking script for accessibility panel usage.
 *
 * @package Everyone_Accessibility_Suite
 * @since 1.0.0
 */

(function() {
	'use strict';

	// Check if options are available
	if (typeof evasAnalyticsOptions === 'undefined') {
		return;
	}

	const options = evasAnalyticsOptions;

	/**
	 * Analytics Tracker Class
	 */
	class AnalyticsTracker {
		constructor() {
			this.uid = this.generateUID();
			this.cache = {};
			this.openStartTime = null;
			this.openTimer = 0;
			this.lastTimestamp = 0;
			this.isSending = false;

			this.init();
		}

		/**
		 * Generate unique identifier for this session
		 */
		generateUID() {
			const timestamp = Date.now().toString(36);
			const random = Math.random().toString(36).substring(2, 15);
			return `${timestamp}-${random}`;
		}

		/**
		 * Initialize tracker
		 */
		init() {
			// Listen for accessibility panel events
			this.bindEvents();

			// Send initial data
			this.sendInitialData();

			// Start interval for sending cached data
			const intervalMs = parseInt(options.sendInterval, 10) * 1000;
			setInterval(() => {
				this.sendCachedData();
			}, intervalMs);
		}

		/**
		 * Get session data
		 */
		getSessionData() {
			return {
				uid: this.uid,
				post_id: options.postId || 0,
				post_type: options.postType || '',
				page_lang: options.pageLang || 'en_US',
				nonce: options.nonce,
			};
		}

		/**
		 * Send initial data with geolocation (if not GDPR strict)
		 */
		sendInitialData() {
			const data = {
				...this.getSessionData(),
				is_mobile: this.isMobile(),
				country_code: '',
			};

			if (options.gdprStrict) {
				this.send(data);
				return;
			}

			// Get country code from IP
			this.getCountryCode()
				.then(countryCode => {
					data.country_code = countryCode;
					this.send(data);
				})
				.catch(() => {
					this.send(data);
				});
		}

		/**
		 * Check if device is mobile
		 */
		isMobile() {
			return /Android|webOS|iPhone|iPad|iPod|BlackBerry|IEMobile|Opera Mini/i.test(navigator.userAgent);
		}

		/**
		 * Get country code from IP
		 */
		async getCountryCode() {
			try {
				const response = await fetch('https://ipapi.co/json/');
				const data = await response.json();
				return data.country_code || '';
			} catch (error) {
				console.warn('Could not get country code:', error);
				return '';
			}
		}

		/**
		 * Bind events to accessibility panel
		 */
		bindEvents() {
			// Listen for custom analytics events
			window.addEventListener('EvasAnalyticsEvent', (e) => {
				this.handleEvent(e.detail);
			});

			// Listen for panel open/close
			document.addEventListener('click', (e) => {
				const target = e.target.closest('[data-evas-trigger], .evas-trigger-button, #evas-panel-trigger, #evas-panel-toggle');
				if (target) {
					this.handlePanelToggle();
				}
			});

			// Listen for toggle changes in the panel
			document.addEventListener('change', (e) => {
				// Check for any checkbox/toggle in the panel
				const target = e.target.closest('#evas-panel input[type="checkbox"], .evas-toggle-box input, [data-evas-toggle]');
				if (target) {
					this.handleToggleChange(target);
				}
			});

			// Listen for profile selection
			document.addEventListener('click', (e) => {
				const target = e.target.closest('[data-evas-profile]');
				if (target) {
					this.handleProfileSelect(target);
				}
			});

			// Track page visibility for accurate time tracking
			document.addEventListener('visibilitychange', () => {
				if (document.hidden) {
					this.pauseOpenTimer();
				} else {
					this.resumeOpenTimer();
				}
			});

			// Check if panel loads with saved settings
			this.checkSavedSettings();
		}

		/**
		 * Check if panel loads with saved settings from localStorage
		 */
		checkSavedSettings() {
			try {
				const savedSettings = localStorage.getItem('evas_settings') || 
				                     localStorage.getItem('evasAccessibilitySettings');
				if (savedSettings) {
					const settings = JSON.parse(savedSettings);
					if (Object.keys(settings).length > 0) {
						this.addToCache('load_saved', 1);
					}
				}
			} catch (error) {
				// Silent fail
			}
		}

		/**
		 * Handle panel toggle
		 */
		handlePanelToggle() {
			const panel = document.querySelector('#evas-panel, .evas-panel');
			// Check multiple class variants for open state
			const isOpen = panel && (
				panel.classList.contains('is-open') || 
				panel.classList.contains('evas-open') ||
				panel.classList.contains('evas-panel--open')
			);

			if (!isOpen) {
				// Panel is being opened
				this.openStartTime = Date.now();
				this.addToCache('open', 1, true);
			} else {
				// Panel is being closed
				this.updateOpenTimer();
			}
		}

		/**
		 * Update open timer
		 */
		updateOpenTimer() {
			if (this.openStartTime) {
				const now = Date.now();
				const duration = now - this.openStartTime;
				this.openTimer += duration;
				this.openStartTime = null;
			}
		}

		/**
		 * Pause open timer
		 */
		pauseOpenTimer() {
			if (this.openStartTime) {
				this.updateOpenTimer();
				this.lastTimestamp = Date.now();
			}
		}

		/**
		 * Resume open timer
		 */
		resumeOpenTimer() {
			const panel = document.querySelector('#evas-panel.is-open, #evas-panel.evas-open, .evas-panel.evas-panel--open');
			if (panel) {
				this.openStartTime = Date.now();
			}
		}

		/**
		 * Handle toggle change
		 */
		handleToggleChange(target) {
			// Get toggle ID from id, name, or data attribute
			let toggleId = target.id || target.name || target.dataset.evasToggle;
			
			// Clean up the name (remove evas- prefix)
			if (toggleId && toggleId.startsWith('evas-')) {
				toggleId = toggleId.replace('evas-', '');
			}
			
			if (toggleId && target.checked) {
				this.addToCache(toggleId, 1);
			}
		}

		/**
		 * Handle profile selection
		 */
		handleProfileSelect(target) {
			const profileId = target.dataset.evasProfile;
			if (profileId) {
				this.addToCache(`profile_${profileId}`, 1);
			}
		}

		/**
		 * Handle custom event
		 */
		handleEvent(detail) {
			const { id, value = 1, immediate = false } = detail;
			if (id) {
				this.addToCache(id, value, immediate);
			}
		}

		/**
		 * Add event to cache
		 */
		addToCache(id, value = 1, immediate = false) {
			if (!this.cache.events) {
				this.cache.events = [];
			}

			// Check if event already exists
			const existing = this.cache.events.find(e => e.id === id);
			if (existing) {
				existing.value += value;
			} else {
				this.cache.events.push({ id, value });
			}

			// Send immediately if needed
			if (immediate) {
				this.sendCachedData();
			}
		}

		/**
		 * Send cached data to server
		 */
		sendCachedData() {
			if (this.isSending) {
				return;
			}

			// Update open timer if panel is currently open
			if (this.openStartTime) {
				const now = Date.now();
				const duration = now - this.openStartTime;
				this.openTimer += duration;
				this.openStartTime = now;
			}

			// Check if there's anything to send
			if (Object.keys(this.cache).length === 0 && this.openTimer === 0) {
				return;
			}

			const data = {
				...this.getSessionData(),
				open_timer: this.openTimer,
				...this.cache,
			};

			this.send(data);

			// Reset cache but keep open timer running if panel is open
			this.cache = {};
			if (!this.openStartTime) {
				this.openTimer = 0;
			}
		}

		/**
		 * Send data to server
		 */
		send(data) {
			this.isSending = true;

			fetch(options.restUrl + 'track', {
				method: 'POST',
				headers: {
					'Content-Type': 'application/json',
					'X-WP-Nonce': options.nonce,
				},
				body: JSON.stringify(data),
				keepalive: true,
			})
			.then(response => response.json())
			.then(result => {
				if (!result.success) {
					console.warn('Analytics tracking error:', result.message);
				}
			})
			.catch(error => {
				console.warn('Analytics tracking error:', error);
			})
			.finally(() => {
				this.isSending = false;
			});
		}
	}

	// Initialize tracker when DOM is ready
	if (document.readyState === 'loading') {
		document.addEventListener('DOMContentLoaded', () => {
			new AnalyticsTracker();
		});
	} else {
		new AnalyticsTracker();
	}
})();

