/**
 * Usage Analytics Admin Script
 *
 * Admin dashboard widget functionality.
 *
 * @package Everyone_Accessibility_Suite
 * @since 1.0.0
 */

(function($) {
	'use strict';

	if (typeof evasAnalyticsAdmin === 'undefined') {
		return;
	}

	const config = evasAnalyticsAdmin;
	const i18n = config.i18n;

	/**
	 * Analytics Widget Class
	 */
	class AnalyticsWidget {
		constructor() {
			this.widget = document.getElementById('evas-analytics-widget');
			if (!this.widget) {
				return;
			}

			this.periodSelect = document.getElementById('evas-analytics-period');
			this.currentPeriod = '604800'; // Default 7 days

			this.init();
		}

		/**
		 * Initialize widget
		 */
		init() {
			this.bindEvents();
			this.loadData();
		}

		/**
		 * Bind events
		 */
		bindEvents() {
			// Period selector
			if (this.periodSelect) {
				this.periodSelect.addEventListener('change', (e) => {
					this.currentPeriod = e.target.value;
					this.loadData();
				});
			}

			// Tab switching
			const tabs = this.widget.querySelectorAll('.evas-analytics-widget__tab');
			tabs.forEach(tab => {
				tab.addEventListener('click', () => {
					this.switchTab(tab.dataset.tab);
				});
			});
		}

		/**
		 * Switch tab
		 */
		switchTab(tabId) {
			// Update active tab button
			const tabs = this.widget.querySelectorAll('.evas-analytics-widget__tab');
			tabs.forEach(tab => {
				tab.classList.toggle('is-active', tab.dataset.tab === tabId);
			});

			// Update active tab content
			const panes = this.widget.querySelectorAll('.evas-analytics-widget__tab-pane');
			panes.forEach(pane => {
				pane.classList.toggle('is-active', pane.dataset.tabContent === tabId);
			});

			// Load tab data
			this.loadTabData(tabId);
		}

		/**
		 * Load all data
		 */
		async loadData() {
			try {
				// Load summary
				const summary = await this.fetchData('summary', { period: this.currentPeriod });
				if (summary.success && summary.data) {
					this.updateSummary(summary.data);
				}

				// Load hourly chart
				const hourlyData = await this.fetchData('stats', { 
					key: 'hourly-usage-chart', 
					period: this.currentPeriod 
				});
				if (hourlyData.success && hourlyData.data) {
					this.renderHourlyChart(hourlyData.data);
				}

				// Load active tab data
				const activeTab = this.widget.querySelector('.evas-analytics-widget__tab.is-active');
				if (activeTab) {
					this.loadTabData(activeTab.dataset.tab);
				}
			} catch (error) {
				console.error('Error loading analytics data:', error);
			}
		}

		/**
		 * Load tab data
		 */
		async loadTabData(tabId) {
			let chartKey = '';
			let chartContainer = '';

			switch (tabId) {
				case 'modes':
					chartKey = 'modes-chart';
					chartContainer = 'evas-modes-chart';
					break;
				case 'profiles':
					chartKey = 'profiles-chart';
					chartContainer = 'evas-profiles-chart';
					break;
				case 'devices':
					chartKey = 'mobile-chart';
					chartContainer = 'evas-devices-chart';
					break;
				case 'countries':
					chartKey = 'country-code-chart';
					chartContainer = 'evas-countries-chart';
					break;
				default:
					return;
			}

			try {
				const data = await this.fetchData('stats', { 
					key: chartKey, 
					period: this.currentPeriod 
				});
				
				if (data.success && data.data) {
					this.renderBarChart(chartContainer, data.data);
				}
			} catch (error) {
				console.error(`Error loading ${tabId} data:`, error);
			}
		}

		/**
		 * Fetch data from API
		 */
		async fetchData(endpoint, params = {}) {
			const url = new URL(config.restUrl + endpoint);
			Object.keys(params).forEach(key => url.searchParams.append(key, params[key]));

			const response = await fetch(url, {
				headers: {
					'X-WP-Nonce': config.nonce,
				},
			});

			return response.json();
		}

		/**
		 * Update summary stats
		 */
		updateSummary(data) {
			const statElements = this.widget.querySelectorAll('[data-evas-stat]');
			statElements.forEach(el => {
				const key = el.dataset.evasStat;
				if (data[key] !== undefined) {
					el.textContent = data[key];
				}
			});
		}

		/**
		 * Render hourly chart using Canvas
		 */
		renderHourlyChart(data) {
			const canvas = document.getElementById('evas-hourly-chart');
			if (!canvas) return;

			const ctx = canvas.getContext('2d');
			const width = canvas.width;
			const height = canvas.height;

			// Clear canvas
			ctx.clearRect(0, 0, width, height);

			// Get data arrays - take last 24 items
			const rawHourlyOpen = data.hourlyOpen || [];
			const rawHourlyLoadSaved = data.hourlyLoadSaved || [];
			const hourlyOpen = rawHourlyOpen.slice(-24);
			const hourlyLoadSaved = rawHourlyLoadSaved.slice(-24);

			if (hourlyOpen.length === 0) {
				// Show no data message
				ctx.fillStyle = '#999';
				ctx.font = '14px -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif';
				ctx.textAlign = 'center';
				ctx.fillText(i18n.noData, width / 2, height / 2);
				return;
			}

			// Find max value for scaling
			const maxValue = Math.max(
				...hourlyOpen.map(v => parseInt(v) || 0),
				...hourlyLoadSaved.map(v => parseInt(v) || 0),
				1
			);

			// Chart settings
			const padding = { top: 10, right: 10, bottom: 30, left: 40 };
			const chartWidth = width - padding.left - padding.right;
			const chartHeight = height - padding.top - padding.bottom;
			const barCount = hourlyOpen.length;
			const barWidth = chartWidth / barCount - 2;

			// Draw grid lines
			ctx.strokeStyle = '#eee';
			ctx.lineWidth = 1;
			for (let i = 0; i <= 4; i++) {
				const y = padding.top + (chartHeight * i / 4);
				ctx.beginPath();
				ctx.moveTo(padding.left, y);
				ctx.lineTo(width - padding.right, y);
				ctx.stroke();
			}

			// Draw bars
			for (let i = 0; i < barCount; i++) {
				const x = padding.left + (i * (chartWidth / barCount));
				
				// Load saved bar
				const loadedValue = parseInt(hourlyLoadSaved[i]) || 0;
				const loadedHeight = (loadedValue / maxValue) * chartHeight;
				ctx.fillStyle = 'rgba(99, 102, 241, 0.6)';
				ctx.fillRect(
					x + 1,
					padding.top + chartHeight - loadedHeight,
					barWidth / 2 - 1,
					loadedHeight
				);

				// Open bar
				const openValue = parseInt(hourlyOpen[i]) || 0;
				const openHeight = (openValue / maxValue) * chartHeight;
				ctx.fillStyle = 'rgba(34, 197, 94, 0.6)';
				ctx.fillRect(
					x + barWidth / 2,
					padding.top + chartHeight - openHeight,
					barWidth / 2 - 1,
					openHeight
				);
			}

			// Draw axes
			ctx.strokeStyle = '#ddd';
			ctx.lineWidth = 1;
			ctx.beginPath();
			ctx.moveTo(padding.left, padding.top);
			ctx.lineTo(padding.left, height - padding.bottom);
			ctx.lineTo(width - padding.right, height - padding.bottom);
			ctx.stroke();

			// Draw hour labels
			ctx.fillStyle = '#999';
			ctx.font = '10px -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif';
			ctx.textAlign = 'center';
			for (let i = 0; i < barCount; i += 4) {
				const x = padding.left + (i * (chartWidth / barCount)) + barWidth / 2;
				ctx.fillText(`${24 - i}h`, x, height - 10);
			}
		}

		/**
		 * Render bar chart for modes/profiles/devices/countries
		 */
		renderBarChart(containerId, data) {
			const container = document.getElementById(containerId);
			if (!container) return;

			// Clear container
			container.innerHTML = '';

			if (!data || data.length === 0) {
				container.innerHTML = `<div class="evas-analytics-widget__no-data">${i18n.noData}</div>`;
				return;
			}

			// Find max value
			const maxValue = Math.max(...data.map(item => parseInt(item.value) || 0), 1);

			// Create bars
			const fragment = document.createDocumentFragment();
			
			// Limit to top 10
			const items = data.slice(0, 10);
			
			items.forEach(item => {
				const value = parseInt(item.value) || 0;
				const percentage = (value / maxValue) * 100;

				const row = document.createElement('div');
				row.className = 'evas-analytics-widget__bar-row';

				row.innerHTML = `
					<span class="evas-analytics-widget__bar-label">${this.escapeHtml(item.label)}</span>
					<div class="evas-analytics-widget__bar-container">
						<div class="evas-analytics-widget__bar" style="width: ${percentage}%"></div>
					</div>
					<span class="evas-analytics-widget__bar-value">${value}</span>
				`;

				fragment.appendChild(row);
			});

			container.appendChild(fragment);
		}

		/**
		 * Escape HTML
		 */
		escapeHtml(str) {
			const div = document.createElement('div');
			div.textContent = str;
			return div.innerHTML;
		}
	}

	// Initialize on DOM ready
	$(document).ready(function() {
		new AnalyticsWidget();
	});

})(jQuery);

