<?php
/**
 * TTS Shortcodes
 *
 * Shortcodes for controlling text-to-speech behavior.
 *
 * @package     Everyone_Accessibility_Suite
 * @subpackage  Modules/Text_To_Speech
 * @version     1.0.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class EVAS_TTS_Shortcodes
 *
 * Provides shortcodes for TTS control.
 */
class EVAS_TTS_Shortcodes {

    /**
     * Constructor
     */
    public function __construct() {
        $this->register_shortcodes();
    }

    /**
     * Register all shortcodes
     *
     * @return void
     */
    private function register_shortcodes(): void {
        // [evas-mute]text[/evas-mute] - Text will not be read
        add_shortcode( 'evas-mute', [ $this, 'mute_shortcode' ] );
        
        // [evas-break time="500ms"] - Add pause
        add_shortcode( 'evas-break', [ $this, 'break_shortcode' ] );
        
        // [evas-say-as interpret-as="date"]...[/evas-say-as] - Special interpretation
        add_shortcode( 'evas-say-as', [ $this, 'say_as_shortcode' ] );
        
        // [evas-emphasis level="strong"]...[/evas-emphasis] - Emphasis
        add_shortcode( 'evas-emphasis', [ $this, 'emphasis_shortcode' ] );
        
        // [evas-prosody rate="slow" pitch="high"]...[/evas-prosody] - Speech modulation
        add_shortcode( 'evas-prosody', [ $this, 'prosody_shortcode' ] );
        
        // [evas-read-button] - Insert read button
        add_shortcode( 'evas-read-button', [ $this, 'read_button_shortcode' ] );
    }

    /**
     * Mute shortcode - content will not be read by TTS
     *
     * @param array  $atts    Shortcode attributes.
     * @param string $content Shortcode content.
     * @return string
     */
    public function mute_shortcode( $atts, $content = null ): string {
        if ( empty( $content ) ) {
            return '';
        }

        return '<span class="evas-tts-mute" data-evas-mute>' . do_shortcode( $content ) . '</span>';
    }

    /**
     * Break shortcode - add pause in speech
     *
     * @param array $atts Shortcode attributes.
     * @return string
     */
    public function break_shortcode( $atts ): string {
        $atts = shortcode_atts( [
            'time'     => '500ms',
            'strength' => 'medium', // x-weak, weak, medium, strong, x-strong
        ], $atts, 'evas-break' );

        // Validate time format
        $time = preg_match( '/^\d+m?s$/', $atts['time'] ) ? $atts['time'] : '500ms';

        return '<span class="evas-tts-break" data-evas-break="' . esc_attr( $time ) . '" data-strength="' . esc_attr( $atts['strength'] ) . '"></span>';
    }

    /**
     * Say-as shortcode - special text interpretation
     *
     * @param array  $atts    Shortcode attributes.
     * @param string $content Shortcode content.
     * @return string
     */
    public function say_as_shortcode( $atts, $content = null ): string {
        if ( empty( $content ) ) {
            return '';
        }

        $atts = shortcode_atts( [
            'interpret-as' => 'characters', // characters, spell-out, cardinal, ordinal, fraction, unit, date, time, telephone, address
            'format'       => '',
            'detail'       => '',
        ], $atts, 'evas-say-as' );

        $valid_types = [ 'characters', 'spell-out', 'cardinal', 'ordinal', 'fraction', 'unit', 'date', 'time', 'telephone', 'address' ];
        $type = in_array( $atts['interpret-as'], $valid_types, true ) ? $atts['interpret-as'] : 'characters';

        $data_attrs = 'data-evas-say-as="' . esc_attr( $type ) . '"';
        
        if ( ! empty( $atts['format'] ) ) {
            $data_attrs .= ' data-format="' . esc_attr( $atts['format'] ) . '"';
        }
        
        if ( ! empty( $atts['detail'] ) ) {
            $data_attrs .= ' data-detail="' . esc_attr( $atts['detail'] ) . '"';
        }

        return '<span class="evas-tts-say-as" ' . $data_attrs . '>' . do_shortcode( $content ) . '</span>';
    }

    /**
     * Emphasis shortcode - emphasize text
     *
     * @param array  $atts    Shortcode attributes.
     * @param string $content Shortcode content.
     * @return string
     */
    public function emphasis_shortcode( $atts, $content = null ): string {
        if ( empty( $content ) ) {
            return '';
        }

        $atts = shortcode_atts( [
            'level' => 'moderate', // reduced, none, moderate, strong
        ], $atts, 'evas-emphasis' );

        $valid_levels = [ 'reduced', 'none', 'moderate', 'strong' ];
        $level = in_array( $atts['level'], $valid_levels, true ) ? $atts['level'] : 'moderate';

        return '<span class="evas-tts-emphasis" data-evas-emphasis="' . esc_attr( $level ) . '">' . do_shortcode( $content ) . '</span>';
    }

    /**
     * Prosody shortcode - control speech rate, pitch, volume
     *
     * @param array  $atts    Shortcode attributes.
     * @param string $content Shortcode content.
     * @return string
     */
    public function prosody_shortcode( $atts, $content = null ): string {
        if ( empty( $content ) ) {
            return '';
        }

        $atts = shortcode_atts( [
            'rate'   => 'medium', // x-slow, slow, medium, fast, x-fast, or percentage
            'pitch'  => 'medium', // x-low, low, medium, high, x-high, or percentage
            'volume' => 'medium', // silent, x-soft, soft, medium, loud, x-loud, or dB
        ], $atts, 'evas-prosody' );

        $data_attrs = [];
        
        if ( $atts['rate'] !== 'medium' ) {
            $data_attrs[] = 'data-rate="' . esc_attr( $atts['rate'] ) . '"';
        }
        
        if ( $atts['pitch'] !== 'medium' ) {
            $data_attrs[] = 'data-pitch="' . esc_attr( $atts['pitch'] ) . '"';
        }
        
        if ( $atts['volume'] !== 'medium' ) {
            $data_attrs[] = 'data-volume="' . esc_attr( $atts['volume'] ) . '"';
        }

        $data_str = ! empty( $data_attrs ) ? ' ' . implode( ' ', $data_attrs ) : '';

        return '<span class="evas-tts-prosody"' . $data_str . '>' . do_shortcode( $content ) . '</span>';
    }

    /**
     * Read button shortcode - insert a read aloud button
     *
     * @param array $atts Shortcode attributes.
     * @return string
     */
    public function read_button_shortcode( $atts ): string {
        $atts = shortcode_atts( [
            'target'  => '', // CSS selector of element to read
            'text'    => __( 'Read Aloud', 'everyone-accessibility-suite' ),
            'class'   => '',
            'icon'    => 'true',
        ], $atts, 'evas-read-button' );

        $classes = [ 'evas-tts-read-button' ];
        
        if ( ! empty( $atts['class'] ) ) {
            $classes[] = sanitize_html_class( $atts['class'] );
        }

        $icon_html = '';
        if ( $atts['icon'] === 'true' ) {
            $icon_html = '<span class="evas-tts-button-icon" aria-hidden="true">🔊</span> ';
        }

        $target_attr = ! empty( $atts['target'] ) ? ' data-target="' . esc_attr( $atts['target'] ) . '"' : '';

        return '<button type="button" class="' . esc_attr( implode( ' ', $classes ) ) . '"' . $target_attr . ' aria-label="' . esc_attr( $atts['text'] ) . '">' . $icon_html . esc_html( $atts['text'] ) . '</button>';
    }

    /**
     * Get all registered shortcodes info
     *
     * @return array
     */
    public static function get_shortcodes_info(): array {
        return [
            'evas-mute' => [
                'name'        => __( 'Mute', 'everyone-accessibility-suite' ),
                'description' => __( 'Content wrapped in this shortcode will not be read by TTS.', 'everyone-accessibility-suite' ),
                'example'     => '[evas-mute]This text will be skipped[/evas-mute]',
            ],
            'evas-break' => [
                'name'        => __( 'Break/Pause', 'everyone-accessibility-suite' ),
                'description' => __( 'Adds a pause in speech.', 'everyone-accessibility-suite' ),
                'example'     => '[evas-break time="1s"]',
                'params'      => [
                    'time'     => __( 'Pause duration (e.g., 500ms, 1s)', 'everyone-accessibility-suite' ),
                    'strength' => __( 'Break strength (x-weak, weak, medium, strong, x-strong)', 'everyone-accessibility-suite' ),
                ],
            ],
            'evas-emphasis' => [
                'name'        => __( 'Emphasis', 'everyone-accessibility-suite' ),
                'description' => __( 'Emphasizes the wrapped text.', 'everyone-accessibility-suite' ),
                'example'     => '[evas-emphasis level="strong"]Important![/evas-emphasis]',
                'params'      => [
                    'level' => __( 'Emphasis level (reduced, none, moderate, strong)', 'everyone-accessibility-suite' ),
                ],
            ],
            'evas-read-button' => [
                'name'        => __( 'Read Button', 'everyone-accessibility-suite' ),
                'description' => __( 'Inserts a button to read content aloud.', 'everyone-accessibility-suite' ),
                'example'     => '[evas-read-button target=".article-content" text="Listen to Article"]',
                'params'      => [
                    'target' => __( 'CSS selector of element to read', 'everyone-accessibility-suite' ),
                    'text'   => __( 'Button text', 'everyone-accessibility-suite' ),
                    'icon'   => __( 'Show icon (true/false)', 'everyone-accessibility-suite' ),
                ],
            ],
        ];
    }
}

