<?php
/**
 * TTS Languages Configuration
 *
 * Contains all supported languages for Text-to-Speech.
 *
 * @package     Everyone_Accessibility_Suite
 * @subpackage  Modules/Text_To_Speech
 * @version     1.0.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class EVAS_TTS_Languages
 *
 * Provides language configurations for Google Cloud TTS.
 */
class EVAS_TTS_Languages {

    /**
     * Get all supported languages
     *
     * @return array
     */
    public static function get_languages(): array {
        return [
            'ar-XA' => __( 'Arabic (العربية)', 'everyone-accessibility-suite' ),
            'bn-IN' => __( 'Bengali (বাংলা)', 'everyone-accessibility-suite' ),
            'bg-BG' => __( 'Bulgarian (Български)', 'everyone-accessibility-suite' ),
            'ca-ES' => __( 'Catalan (Català)', 'everyone-accessibility-suite' ),
            'cs-CZ' => __( 'Czech (Čeština)', 'everyone-accessibility-suite' ),
            'da-DK' => __( 'Danish (Dansk)', 'everyone-accessibility-suite' ),
            'de-DE' => __( 'German (Deutsch)', 'everyone-accessibility-suite' ),
            'el-GR' => __( 'Greek (Ελληνικά)', 'everyone-accessibility-suite' ),
            'en-AU' => __( 'English (Australian)', 'everyone-accessibility-suite' ),
            'en-GB' => __( 'English (UK)', 'everyone-accessibility-suite' ),
            'en-IN' => __( 'English (India)', 'everyone-accessibility-suite' ),
            'en-US' => __( 'English (US)', 'everyone-accessibility-suite' ),
            'es-ES' => __( 'Spanish (Español)', 'everyone-accessibility-suite' ),
            'es-US' => __( 'Spanish US (Español)', 'everyone-accessibility-suite' ),
            'et-EE' => __( 'Estonian (Eesti)', 'everyone-accessibility-suite' ),
            'fi-FI' => __( 'Finnish (Suomi)', 'everyone-accessibility-suite' ),
            'fil-PH' => __( 'Filipino', 'everyone-accessibility-suite' ),
            'fr-CA' => __( 'French Canada (Français)', 'everyone-accessibility-suite' ),
            'fr-FR' => __( 'French (Français)', 'everyone-accessibility-suite' ),
            'gu-IN' => __( 'Gujarati (ગુજરાતી)', 'everyone-accessibility-suite' ),
            'he-IL' => __( 'Hebrew (עברית)', 'everyone-accessibility-suite' ),
            'hi-IN' => __( 'Hindi (हिन्दी)', 'everyone-accessibility-suite' ),
            'hu-HU' => __( 'Hungarian (Magyar)', 'everyone-accessibility-suite' ),
            'id-ID' => __( 'Indonesian (Bahasa)', 'everyone-accessibility-suite' ),
            'is-IS' => __( 'Icelandic (Íslenska)', 'everyone-accessibility-suite' ),
            'it-IT' => __( 'Italian (Italiano)', 'everyone-accessibility-suite' ),
            'ja-JP' => __( 'Japanese (日本語)', 'everyone-accessibility-suite' ),
            'kn-IN' => __( 'Kannada (ಕನ್ನಡ)', 'everyone-accessibility-suite' ),
            'ko-KR' => __( 'Korean (한국어)', 'everyone-accessibility-suite' ),
            'lv-LV' => __( 'Latvian (Latviešu)', 'everyone-accessibility-suite' ),
            'lt-LT' => __( 'Lithuanian (Lietuvių)', 'everyone-accessibility-suite' ),
            'ml-IN' => __( 'Malayalam (മലയാളം)', 'everyone-accessibility-suite' ),
            'mr-IN' => __( 'Marathi (मराठी)', 'everyone-accessibility-suite' ),
            'ms-MY' => __( 'Malay (Bahasa Melayu)', 'everyone-accessibility-suite' ),
            'nb-NO' => __( 'Norwegian (Norsk)', 'everyone-accessibility-suite' ),
            'nl-BE' => __( 'Dutch Belgium (Nederlands)', 'everyone-accessibility-suite' ),
            'nl-NL' => __( 'Dutch (Nederlands)', 'everyone-accessibility-suite' ),
            'pa-IN' => __( 'Punjabi (ਪੰਜਾਬੀ)', 'everyone-accessibility-suite' ),
            'pl-PL' => __( 'Polish (Polski)', 'everyone-accessibility-suite' ),
            'pt-BR' => __( 'Portuguese Brazil (Português)', 'everyone-accessibility-suite' ),
            'pt-PT' => __( 'Portuguese (Português)', 'everyone-accessibility-suite' ),
            'ro-RO' => __( 'Romanian (Română)', 'everyone-accessibility-suite' ),
            'ru-RU' => __( 'Russian (Русский)', 'everyone-accessibility-suite' ),
            'sk-SK' => __( 'Slovak (Slovenčina)', 'everyone-accessibility-suite' ),
            'sr-RS' => __( 'Serbian (Српски)', 'everyone-accessibility-suite' ),
            'sv-SE' => __( 'Swedish (Svenska)', 'everyone-accessibility-suite' ),
            'ta-IN' => __( 'Tamil (தமிழ்)', 'everyone-accessibility-suite' ),
            'te-IN' => __( 'Telugu (తెలుగు)', 'everyone-accessibility-suite' ),
            'th-TH' => __( 'Thai (ภาษาไทย)', 'everyone-accessibility-suite' ),
            'tr-TR' => __( 'Turkish (Türkçe)', 'everyone-accessibility-suite' ),
            'uk-UA' => __( 'Ukrainian (Українська)', 'everyone-accessibility-suite' ),
            'vi-VN' => __( 'Vietnamese (Tiếng Việt)', 'everyone-accessibility-suite' ),
            'yue-HK' => __( 'Chinese Cantonese (粵語)', 'everyone-accessibility-suite' ),
            'cmn-CN' => __( 'Chinese Mandarin (普通话)', 'everyone-accessibility-suite' ),
            'cmn-TW' => __( 'Chinese Taiwan (國語)', 'everyone-accessibility-suite' ),
        ];
    }

    /**
     * Get voice types
     *
     * @return array
     */
    public static function get_voice_types(): array {
        return [
            'Standard'  => __( 'Standard', 'everyone-accessibility-suite' ),
            'Wavenet'   => __( 'WaveNet (Higher quality)', 'everyone-accessibility-suite' ),
            'Neural2'   => __( 'Neural2 (Most natural)', 'everyone-accessibility-suite' ),
            'Polyglot'  => __( 'Polyglot (Multi-language)', 'everyone-accessibility-suite' ),
            'Studio'    => __( 'Studio (Professional)', 'everyone-accessibility-suite' ),
        ];
    }

    /**
     * Get default voices for languages
     *
     * @return array
     */
    public static function get_default_voices(): array {
        return [
            'en-US' => 'en-US-Wavenet-D',
            'en-GB' => 'en-GB-Wavenet-A',
            'de-DE' => 'de-DE-Wavenet-A',
            'fr-FR' => 'fr-FR-Wavenet-A',
            'es-ES' => 'es-ES-Wavenet-B',
            'it-IT' => 'it-IT-Wavenet-A',
            'ru-RU' => 'ru-RU-Wavenet-A',
            'ja-JP' => 'ja-JP-Wavenet-A',
            'ko-KR' => 'ko-KR-Wavenet-A',
            'zh-CN' => 'cmn-CN-Wavenet-A',
        ];
    }

    /**
     * Get language code from HTML lang attribute
     *
     * @param string $html_lang HTML lang attribute value.
     * @return string|null Matching language code or null.
     */
    public static function get_language_from_html_lang( string $html_lang ): ?string {
        $languages = self::get_languages();
        
        // Direct match
        if ( isset( $languages[ $html_lang ] ) ) {
            return $html_lang;
        }

        // Try to find partial match (e.g., 'en' -> 'en-US')
        $lang_prefix = substr( $html_lang, 0, 2 );
        
        foreach ( array_keys( $languages ) as $lang_code ) {
            if ( strpos( $lang_code, $lang_prefix ) === 0 ) {
                return $lang_code;
            }
        }

        // Default to English US
        return 'en-US';
    }
}

