<?php
/**
 * TTS Cache Handler
 *
 * Handles caching of generated audio files.
 *
 * @package     Everyone_Accessibility_Suite
 * @subpackage  Modules/Text_To_Speech
 * @version     1.0.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class EVAS_TTS_Cache
 *
 * Manages caching of TTS generated audio.
 */
class EVAS_TTS_Cache {

    /**
     * Cache directory path
     *
     * @var string
     */
    private string $cache_dir;

    /**
     * Constructor
     */
    public function __construct() {
        $upload_dir = wp_upload_dir();
        $this->cache_dir = $upload_dir['basedir'] . '/evas-tts-cache/';
        
        $this->ensure_cache_dir();
    }

    /**
     * Ensure cache directory exists
     *
     * @return void
     */
    private function ensure_cache_dir(): void {
        if ( ! file_exists( $this->cache_dir ) ) {
            wp_mkdir_p( $this->cache_dir );
            
            // Add .htaccess for security
            $htaccess_content = "Order deny,allow\nDeny from all\n<FilesMatch \"\.(mp3|wav|ogg)$\">\n    Allow from all\n</FilesMatch>";
            file_put_contents( $this->cache_dir . '.htaccess', $htaccess_content );
            
            // Add index.php
            file_put_contents( $this->cache_dir . 'index.php', '<?php // Silence is golden' );
        }
    }

    /**
     * Get cached audio
     *
     * @param string $key Cache key.
     * @return string|false Base64 audio or false if not cached.
     */
    public function get( string $key ) {
        if ( ! $this->is_enabled() ) {
            return false;
        }

        // Best-effort cleanup to prevent cache bloat if wp-cron is not triggered reliably.
        $this->maybe_cleanup_expired();

        $file = $this->get_cache_file_path( $key );
        
        if ( ! file_exists( $file ) ) {
            return false;
        }

        // Check expiration
        $meta_file = $file . '.meta';
        if ( file_exists( $meta_file ) ) {
            $meta = json_decode( file_get_contents( $meta_file ), true );
            
            if ( isset( $meta['expires'] ) && $meta['expires'] < time() ) {
                $this->delete( $key );
                return false;
            }
        }

        $content = file_get_contents( $file );
        return $content !== false ? base64_encode( $content ) : false;
    }

    /**
     * Store audio in cache
     *
     * @param string $key        Cache key.
     * @param string $audio_data Audio data (binary).
     * @param int    $expiration Expiration time in seconds.
     * @return bool Success status.
     */
    public function set( string $key, string $audio_data, int $expiration = DAY_IN_SECONDS ): bool {
        if ( ! $this->is_enabled() ) {
            return false;
        }

        // Best-effort cleanup to prevent cache bloat if wp-cron is not triggered reliably.
        $this->maybe_cleanup_expired();

        $file = $this->get_cache_file_path( $key );
        
        // Save audio data
        $result = file_put_contents( $file, $audio_data );
        
        if ( $result === false ) {
            return false;
        }

        // Save metadata
        $meta = [
            'created' => time(),
            'expires' => time() + $expiration,
            'size'    => strlen( $audio_data ),
        ];
        
        file_put_contents( $file . '.meta', wp_json_encode( $meta ) );

        return true;
    }

    /**
     * Delete cached item
     *
     * @param string $key Cache key.
     * @return bool Success status.
     */
    public function delete( string $key ): bool {
        $file = $this->get_cache_file_path( $key );
        
        $deleted = false;
        
        if ( file_exists( $file ) ) {
            $deleted = (bool) wp_delete_file( $file );
        }
        
        if ( file_exists( $file . '.meta' ) ) {
            wp_delete_file( $file . '.meta' );
        }

        return $deleted;
    }

    /**
     * Clear all cache
     *
     * @return int Number of files deleted.
     */
    public function clear(): int {
        $files = glob( $this->cache_dir . '*.mp3' );
        $count = 0;

        if ( $files ) {
            foreach ( $files as $file ) {
                if ( is_file( $file ) ) {
                    wp_delete_file( $file );
                    $count++;
                }
                
                // Delete meta file too
                if ( file_exists( $file . '.meta' ) ) {
                    wp_delete_file( $file . '.meta' );
                }
            }
        }

        return $count;
    }

    /**
     * Get cache statistics
     *
     * @return array Cache stats.
     */
    public function get_stats(): array {
        $files = glob( $this->cache_dir . '*.mp3' );
        $total_size = 0;
        $count = 0;

        if ( $files ) {
            foreach ( $files as $file ) {
                if ( is_file( $file ) ) {
                    $total_size += filesize( $file );
                    $count++;
                }
            }
        }

        return [
            'count'      => $count,
            'total_size' => $total_size,
            'size_human' => size_format( $total_size ),
            'enabled'    => $this->is_enabled(),
        ];
    }

    /**
     * Clean expired cache entries
     *
     * @return int Number of expired files deleted.
     */
    public function cleanup_expired(): int {
        $meta_files = glob( $this->cache_dir . '*.meta' );
        $deleted = 0;

        if ( $meta_files ) {
            foreach ( $meta_files as $meta_file ) {
                $meta = json_decode( file_get_contents( $meta_file ), true );
                
                if ( isset( $meta['expires'] ) && $meta['expires'] < time() ) {
                    $audio_file = str_replace( '.meta', '', $meta_file );
                    
                    if ( file_exists( $audio_file ) ) {
                        wp_delete_file( $audio_file );
                        $deleted++;
                    }
                    
                    wp_delete_file( $meta_file );
                }
            }
        }

        return $deleted;
    }

    /**
     * Run cleanup with a small probability to avoid frequent directory scans.
     *
     * This is a safety net for environments where wp-cron is disabled or rarely triggered.
     *
     * @return void
     */
    private function maybe_cleanup_expired(): void {
        // ~1% probability.
        if ( mt_rand( 1, 100 ) !== 1 ) {
            return;
        }

        $this->cleanup_expired();
    }

    /**
     * Get cache file path
     *
     * @param string $key Cache key.
     * @return string File path.
     */
    private function get_cache_file_path( string $key ): string {
        return $this->cache_dir . sanitize_file_name( $key ) . '.mp3';
    }

    /**
     * Generate cache key from text and settings
     *
     * @param string $text     Text to convert.
     * @param string $language Language code.
     * @param string $voice    Voice name.
     * @param float  $rate     Speaking rate.
     * @param float  $pitch    Voice pitch.
     * @return string Cache key.
     */
    public static function generate_key( string $text, string $language, string $voice, float $rate = 1.0, float $pitch = 0.0 ): string {
        $data = $text . $language . $voice . $rate . $pitch;
        return md5( $data );
    }

    /**
     * Check if caching is enabled
     *
     * @return bool
     */
    public function is_enabled(): bool {
        return (bool) get_option( 'evas_tts_cache_enabled', true );
    }

    /**
     * Get cache directory path
     *
     * @return string
     */
    public function get_cache_dir(): string {
        return $this->cache_dir;
    }
}

