<?php
/**
 * Text-to-Speech Module Main Class
 *
 * Core module class that extends EVAS_Module_Base.
 *
 * @package     Everyone_Accessibility_Suite
 * @subpackage  Modules/Text_To_Speech
 * @version     1.0.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class EVAS_Text_To_Speech_Module
 *
 * Main module class for Text-to-Speech.
 */
class EVAS_Text_To_Speech_Module extends EVAS_Module_Base {

    /**
     * Speech generator instance
     *
     * @var EVAS_TTS_Speech_Generator|null
     */
    private $generator = null;

    /**
     * Constructor
     *
     * @param EVAS_Settings $settings_manager Settings manager instance.
     * @param array         $settings Plugin settings.
     */
    public function __construct( $settings_manager = null, $settings = array() ) {
        $this->module_id      = 'text_to_speech';
        $this->module_name    = __( 'Text-to-Speech', 'everyone-accessibility-suite' );
        $this->module_version = defined( 'EVAS_TTS_VERSION' ) ? EVAS_TTS_VERSION : '1.0.0';
        
        parent::__construct( $settings_manager, $settings );
    }

    /**
     * Get module settings
     *
     * @return array Module settings.
     */
    public function get_module_settings() {
        return [
            'api_key'       => get_option( 'evas_tts_api_key', '' ),
            'language'      => get_option( 'evas_tts_language', 'en-US' ),
            'voice_name'    => get_option( 'evas_tts_voice_name', 'en-US-Wavenet-D' ),
            'speaking_rate' => (float) get_option( 'evas_tts_speaking_rate', 1.0 ),
            'pitch'         => (float) get_option( 'evas_tts_pitch', 0.0 ),
            'cache_enabled' => (bool) get_option( 'evas_tts_cache_enabled', true ),
        ];
    }

    /**
     * Initialize the module
     */
    public function init() {
        // Initialize components
        $this->generator = new EVAS_TTS_Speech_Generator();
        new EVAS_TTS_Shortcodes();

        // Register REST routes
        add_action( 'rest_api_init', [ $this, 'register_rest_routes' ] );

        // Enqueue frontend assets
        add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_frontend_assets' ] );

        // Add TTS button to panel
        add_action( 'evas_panel_controls', [ $this, 'render_tts_control' ], 20 );

        // Schedule cache cleanup
        if ( ! wp_next_scheduled( 'evas_tts_cache_cleanup' ) ) {
            // Run hourly to keep cache directory bounded even when traffic is low.
            wp_schedule_event( time(), 'hourly', 'evas_tts_cache_cleanup' );
        }
        add_action( 'evas_tts_cache_cleanup', [ $this, 'cleanup_cache' ] );
    }

    /**
     * Register module settings
     */
    public function register_settings() {
        register_setting( 'evas_tts_settings', 'evas_tts_api_key', [
            'type'              => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default'           => '',
        ]);

        register_setting( 'evas_tts_settings', 'evas_tts_language', [
            'type'              => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default'           => 'en-US',
        ]);

        register_setting( 'evas_tts_settings', 'evas_tts_voice_name', [
            'type'              => 'string',
            'sanitize_callback' => 'sanitize_text_field',
            'default'           => 'en-US-Wavenet-D',
        ]);

        register_setting( 'evas_tts_settings', 'evas_tts_speaking_rate', [
            'type'    => 'number',
            'default' => 1.0,
        ]);

        register_setting( 'evas_tts_settings', 'evas_tts_pitch', [
            'type'    => 'number',
            'default' => 0.0,
        ]);

        register_setting( 'evas_tts_settings', 'evas_tts_cache_enabled', [
            'type'    => 'boolean',
            'default' => true,
        ]);
    }

    /**
     * Register REST API routes
     *
     * @return void
     */
    public function register_rest_routes(): void {
        $controller = new EVAS_TTS_REST_Controller();
        $controller->register_routes();
    }

    /**
     * Register admin pages/routes
     *
     * @return array Admin pages configuration.
     */
    public function register_admin_pages(): array {
        return [
            [
                'path'  => '/text-to-speech',
                'title' => __( 'Text-to-Speech', 'everyone-accessibility-suite' ),
                'icon'  => 'volume-2',
            ],
        ];
    }

    /**
     * Enqueue frontend assets
     *
     * @return void
     */
    public function enqueue_frontend_assets(): void {
        if ( ! $this->is_enabled() ) {
            return;
        }

        // Main JavaScript
        wp_enqueue_script(
            'evas-tts',
            EVAS_TTS_URL . 'assets/js/text-to-speech.js',
            [ 'jquery' ],
            EVAS_TTS_VERSION,
            true
        );

        // Main styles
        wp_enqueue_style(
            'evas-tts',
            EVAS_TTS_URL . 'assets/css/text-to-speech.css',
            [],
            EVAS_TTS_VERSION
        );

        // Localize script
        wp_localize_script( 'evas-tts', 'evasTTS', [
            'ajaxUrl'       => admin_url( 'admin-ajax.php' ),
            'restUrl'       => rest_url( 'evas-tts/v1/' ),
            'nonce'         => wp_create_nonce( 'evas_tts_nonce' ),
            // REST nonce works for guests too (ties request to site context).
            'restNonce'     => wp_create_nonce( 'wp_rest' ),
            // Lightweight guest token to prevent off-site abuse of REST generate.
            'guestToken'    => hash_hmac( 'sha256', gmdate( 'Y-m-d' ), wp_salt( 'evas_tts_guest' ) ),
            'hasApiKey'     => $this->generator->has_api_key(),
            'settings'      => $this->get_frontend_settings(),
            'i18n'          => $this->get_i18n_strings(),
        ] );
    }

    /**
     * Render TTS control in accessibility panel
     *
     * @return void
     */
    public function render_tts_control(): void {
        if ( ! $this->generator->has_api_key() ) {
            return;
        }
        ?>
        <div class="evas-section evas-tts-section">
            <h2 class="evas-section-title">
                <span class="evas-section-icon"></span>
                <?php esc_html_e( 'Read Aloud', 'everyone-accessibility-suite' ); ?>
            </h2>
            <div class="evas-section-content">
                <div id="evas-tts-controls" class="evas-tts-controls">
                    <button type="button" 
                            id="evas-tts-read-selection" 
                            class="evas-action-box evas-toggle-box"
                            aria-label="<?php esc_attr_e( 'Read selected text', 'everyone-accessibility-suite' ); ?>">
                        <div class="evas-action-box-content">
                            <span class="evas-icon evas-icon-read-selection">🔊</span>
                            <span class="evas-title"><?php esc_html_e( 'Read Selection', 'everyone-accessibility-suite' ); ?></span>
                        </div>
                    </button>
                    
                    <button type="button" 
                            id="evas-tts-read-page" 
                            class="evas-action-box evas-toggle-box"
                            aria-label="<?php esc_attr_e( 'Read entire page', 'everyone-accessibility-suite' ); ?>">
                        <div class="evas-action-box-content">
                            <span class="evas-icon evas-icon-read-page">📖</span>
                            <span class="evas-title"><?php esc_html_e( 'Read Page', 'everyone-accessibility-suite' ); ?></span>
                        </div>
                    </button>
                    
                    <button type="button" 
                            id="evas-tts-stop" 
                            class="evas-action-box evas-toggle-box"
                            aria-label="<?php esc_attr_e( 'Stop reading', 'everyone-accessibility-suite' ); ?>"
                            style="display: none;">
                        <div class="evas-action-box-content">
                            <span class="evas-icon evas-icon-stop">⏹️</span>
                            <span class="evas-title"><?php esc_html_e( 'Stop', 'everyone-accessibility-suite' ); ?></span>
                        </div>
                    </button>
                </div>
                
                <div id="evas-tts-status" class="evas-tts-status" aria-live="polite"></div>
            </div>
        </div>
        <?php
    }

    /**
     * Get settings for frontend use
     *
     * @return array
     */
    private function get_frontend_settings(): array {
        return [
            'language'      => get_option( 'evas_tts_language', 'en-US' ),
            'speaking_rate' => (float) get_option( 'evas_tts_speaking_rate', 1.0 ),
        ];
    }

    /**
     * Get internationalization strings
     *
     * @return array
     */
    private function get_i18n_strings(): array {
        return [
            'reading'         => __( 'Reading...', 'everyone-accessibility-suite' ),
            'loading'         => __( 'Loading audio...', 'everyone-accessibility-suite' ),
            'noSelection'     => __( 'Please select some text to read', 'everyone-accessibility-suite' ),
            'error'           => __( 'Error generating speech', 'everyone-accessibility-suite' ),
            'noApiKey'        => __( 'Text-to-Speech is not configured', 'everyone-accessibility-suite' ),
            'stopped'         => __( 'Stopped', 'everyone-accessibility-suite' ),
            'finished'        => __( 'Finished reading', 'everyone-accessibility-suite' ),
        ];
    }

    /**
     * Cleanup expired cache entries
     *
     * @return void
     */
    public function cleanup_cache(): void {
        $cache = new EVAS_TTS_Cache();
        $cache->cleanup_expired();
    }

    /**
     * Get module metadata for REST API
     *
     * @return array
     */
    public function get_metadata(): array {
        return [
            'id'          => $this->get_id(),
            'name'        => $this->get_name(),
            'description' => $this->get_description(),
            'version'     => $this->get_version(),
            'enabled'     => $this->is_enabled(),
            'has_api_key' => $this->generator ? $this->generator->has_api_key() : false,
            'languages'   => count( EVAS_TTS_Languages::get_languages() ),
        ];
    }
}

