<?php
/**
 * Speech Generator
 *
 * Handles text-to-speech generation using Google Cloud TTS API.
 *
 * @package     Everyone_Accessibility_Suite
 * @subpackage  Modules/Text_To_Speech
 * @version     1.0.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class EVAS_TTS_Speech_Generator
 *
 * Generates speech from text using Google Cloud TTS.
 */
class EVAS_TTS_Speech_Generator {

    /**
     * Cache instance
     *
     * @var EVAS_TTS_Cache
     */
    private EVAS_TTS_Cache $cache;

    /**
     * Google Cloud TTS API endpoint
     *
     * @var string
     */
    private const API_ENDPOINT = 'https://texttospeech.googleapis.com/v1/text:synthesize';

    /**
     * Constructor
     */
    public function __construct() {
        $this->cache = new EVAS_TTS_Cache();
        
        // Register AJAX handlers
        add_action( 'wp_ajax_evas_tts_generate', [ $this, 'ajax_generate' ] );
        add_action( 'wp_ajax_nopriv_evas_tts_generate', [ $this, 'ajax_generate' ] );
    }

    /**
     * AJAX handler for speech generation
     *
     * @return void
     */
    public function ajax_generate(): void {
        check_ajax_referer( 'evas_tts_nonce', 'nonce' );

        $text = isset( $_POST['text'] ) ? sanitize_textarea_field( wp_unslash( $_POST['text'] ) ) : '';
        $lang = isset( $_POST['lang'] ) ? sanitize_text_field( wp_unslash( $_POST['lang'] ) ) : '';

        if ( empty( $text ) ) {
            wp_send_json_error( [ 'message' => __( 'No text provided', 'everyone-accessibility-suite' ) ] );
        }

        // Clean the text
        $text = $this->prepare_text( $text );

        if ( empty( $text ) ) {
            wp_send_json_error( [ 'message' => __( 'Text is empty after cleaning', 'everyone-accessibility-suite' ) ] );
        }

        // Get language
        $language = $this->get_language( $lang );

        try {
            $audio = $this->generate_speech( $text, $language );
            wp_send_json_success( [ 'audio' => $audio ] );
        } catch ( Exception $e ) {
            wp_send_json_error( [ 'message' => $e->getMessage() ] );
        }
    }

    /**
     * Generate speech from text
     *
     * @param string      $text     Text to convert.
     * @param string|null $language Language code.
     * @return string Base64 encoded audio.
     * @throws Exception On API error.
     */
    public function generate_speech( string $text, ?string $language = null ): string {
        $settings = $this->get_settings();
        $language = $language ?? $settings['language'];
        $voice    = $settings['voice_name'];
        $rate     = (float) $settings['speaking_rate'];
        $pitch    = (float) $settings['pitch'];

        // Check cache first
        $cache_key = EVAS_TTS_Cache::generate_key( $text, $language, $voice, $rate, $pitch );
        $cached    = $this->cache->get( $cache_key );

        if ( $cached !== false ) {
            return $cached;
        }

        // Generate new audio
        $audio = $this->call_google_api( $text, $language, $voice, $rate, $pitch );

        // Cache the result
        $audio_binary = base64_decode( $audio );
        $this->cache->set( $cache_key, $audio_binary );

        return $audio;
    }

    /**
     * Call Google Cloud TTS API
     *
     * @param string $text     Text to convert.
     * @param string $language Language code.
     * @param string $voice    Voice name.
     * @param float  $rate     Speaking rate.
     * @param float  $pitch    Voice pitch.
     * @return string Base64 encoded audio.
     * @throws Exception On API error.
     */
    private function call_google_api( string $text, string $language, string $voice, float $rate, float $pitch ): string {
        $api_key = $this->get_api_key();

        if ( empty( $api_key ) ) {
            throw new Exception( esc_html__( 'Google Cloud API key not configured', 'everyone-accessibility-suite' ) );
        }

        // Prepare request body
        $body = [
            'input' => [
                'text' => $text,
            ],
            'voice' => [
                'languageCode' => $language,
                'name'         => $voice,
            ],
            'audioConfig' => [
                'audioEncoding' => 'MP3',
                'speakingRate'  => max( 0.25, min( 4.0, $rate ) ),
                'pitch'         => max( -20.0, min( 20.0, $pitch ) ),
            ],
        ];

        // Make API request
        $response = wp_remote_post(
            self::API_ENDPOINT . '?key=' . $api_key,
            [
                'headers' => [
                    'Content-Type' => 'application/json',
                ],
                'body'    => wp_json_encode( $body ),
                'timeout' => 30,
            ]
        );

        if ( is_wp_error( $response ) ) {
            throw new Exception( esc_html( $response->get_error_message() ) );
        }

        $response_code = wp_remote_retrieve_response_code( $response );
        $response_body = wp_remote_retrieve_body( $response );
        $data          = json_decode( $response_body, true );

        if ( $response_code !== 200 ) {
            $error_message = $data['error']['message'] ?? __( 'Unknown API error', 'everyone-accessibility-suite' );
            $error_message = is_string( $error_message ) ? $error_message : '';
            // translators: %s: error message returned by Google Cloud Text-to-Speech API.
            throw new Exception( sprintf( esc_html__( 'Google Cloud API error: %s', 'everyone-accessibility-suite' ), esc_html( $error_message ) ) );
        }

        if ( empty( $data['audioContent'] ) ) {
            throw new Exception( esc_html__( 'No audio content received from API', 'everyone-accessibility-suite' ) );
        }

        return $data['audioContent'];
    }

    /**
     * Prepare text for speech synthesis
     *
     * @param string $html HTML content.
     * @return string Cleaned text.
     */
    private function prepare_text( string $html ): string {
        // Remove muted elements
        $html = $this->remove_muted_elements( $html );

        // Process break tags
        $html = $this->process_break_tags( $html );

        // Strip HTML tags
        $text = wp_strip_all_tags( $html );

        // Decode HTML entities
        $text = html_entity_decode( $text, ENT_QUOTES, 'UTF-8' );

        // Remove extra whitespace
        $text = preg_replace( '/\s+/', ' ', $text );

        // Trim
        $text = trim( $text );

        // Limit length (Google API limit is 5000 characters)
        if ( strlen( $text ) > 4500 ) {
            $text = substr( $text, 0, 4500 );
            // Try to end at a sentence
            $last_period = strrpos( $text, '.' );
            if ( $last_period !== false && $last_period > 4000 ) {
                $text = substr( $text, 0, $last_period + 1 );
            }
        }

        return $text;
    }

    /**
     * Remove muted elements from HTML
     *
     * @param string $html HTML content.
     * @return string HTML without muted elements.
     */
    private function remove_muted_elements( string $html ): string {
        // Remove elements with class evas-tts-mute or data-evas-mute attribute
        $patterns = [
            '/<[^>]*class="[^"]*evas-tts-mute[^"]*"[^>]*>.*?<\/[^>]+>/si',
            '/<[^>]*data-evas-mute[^>]*>.*?<\/[^>]+>/si',
            '/<span[^>]*class="[^"]*evas-mute[^"]*"[^>]*>.*?<\/span>/si',
        ];

        foreach ( $patterns as $pattern ) {
            $html = preg_replace( $pattern, '', $html );
        }

        return $html;
    }

    /**
     * Process break tags
     *
     * @param string $html HTML content.
     * @return string HTML with break tags processed.
     */
    private function process_break_tags( string $html ): string {
        // Replace evas-break spans with pauses (represented as ellipsis for now)
        $html = preg_replace(
            '/<span[^>]*data-evas-break[^>]*>.*?<\/span>/si',
            '... ',
            $html
        );

        return $html;
    }

    /**
     * Get language code
     *
     * @param string $lang Language from request.
     * @return string Language code for API.
     */
    private function get_language( string $lang ): string {
        if ( ! empty( $lang ) ) {
            $matched = EVAS_TTS_Languages::get_language_from_html_lang( $lang );
            if ( $matched ) {
                return $matched;
            }
        }

        return $this->get_settings()['language'] ?? 'en-US';
    }

    /**
     * Get TTS settings
     *
     * @return array
     */
    private function get_settings(): array {
        return [
            'language'      => get_option( 'evas_tts_language', 'en-US' ),
            'voice_name'    => get_option( 'evas_tts_voice_name', 'en-US-Wavenet-D' ),
            'speaking_rate' => get_option( 'evas_tts_speaking_rate', 1.0 ),
            'pitch'         => get_option( 'evas_tts_pitch', 0.0 ),
        ];
    }

    /**
     * Get API key
     *
     * @return string
     */
    private function get_api_key(): string {
        return get_option( 'evas_tts_api_key', '' );
    }

    /**
     * Check if API key is configured
     *
     * @return bool
     */
    public function has_api_key(): bool {
        return ! empty( $this->get_api_key() );
    }

    /**
     * Get available voices from API
     *
     * @param string|null $language_code Filter by language.
     * @return array|WP_Error List of voices or error.
     */
    public function get_voices( ?string $language_code = null ) {
        $api_key = $this->get_api_key();

        if ( empty( $api_key ) ) {
            return new WP_Error( 'no_api_key', __( 'API key not configured', 'everyone-accessibility-suite' ) );
        }

        $url = 'https://texttospeech.googleapis.com/v1/voices?key=' . $api_key;

        if ( $language_code ) {
            $url .= '&languageCode=' . urlencode( $language_code );
        }

        $response = wp_remote_get( $url, [ 'timeout' => 15 ] );

        if ( is_wp_error( $response ) ) {
            return $response;
        }

        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body, true );

        if ( ! isset( $data['voices'] ) ) {
            return new WP_Error( 'api_error', __( 'Failed to fetch voices', 'everyone-accessibility-suite' ) );
        }

        return $data['voices'];
    }
}

