<?php
/**
 * TTS REST Controller
 *
 * REST API endpoints for Text-to-Speech module.
 *
 * @package     Everyone_Accessibility_Suite
 * @subpackage  Modules/Text_To_Speech
 * @version     1.0.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class EVAS_TTS_REST_Controller
 *
 * REST API controller for TTS module.
 */
class EVAS_TTS_REST_Controller extends WP_REST_Controller {

    /**
     * Rate limit: max requests per 10 minutes (per IP).
     *
     * @since 1.1.0
     * @var int
     */
    private const RL_MAX_10M = 60;

    /**
     * Rate limit: window in seconds (10 minutes).
     *
     * @since 1.1.0
     * @var int
     */
    private const RL_WINDOW_10M = 600;

    /**
     * Rate limit: max requests per day (per IP).
     *
     * @since 1.1.0
     * @var int
     */
    private const RL_MAX_DAY = 500;

    /**
     * Max raw text length accepted by REST endpoint (before internal cleaning).
     *
     * @since 1.1.0
     * @var int
     */
    private const MAX_TEXT_LEN_RAW = 6000;

    /**
     * REST API namespace
     *
     * @var string
     */
    protected $namespace = 'evas-tts/v1';

    /**
     * Register REST routes
     *
     * @return void
     */
    public function register_routes(): void {
        // POST /wp-json/evas-tts/v1/generate
        register_rest_route( $this->namespace, '/generate', [
            'methods'             => WP_REST_Server::CREATABLE,
            'callback'            => [ $this, 'generate_speech' ],
            'permission_callback' => '__return_true',
            'args'                => [
                'text' => [
                    'required'          => true,
                    'type'              => 'string',
                    'sanitize_callback' => 'sanitize_textarea_field',
                ],
                'language' => [
                    'type'              => 'string',
                    'default'           => '',
                    'sanitize_callback' => 'sanitize_text_field',
                ],
            ],
        ] );

        // GET /wp-json/evas-tts/v1/voices
        register_rest_route( $this->namespace, '/voices', [
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => [ $this, 'get_voices' ],
            'permission_callback' => [ $this, 'get_permissions_check' ],
            'args'                => [
                'language' => [
                    'type'              => 'string',
                    'sanitize_callback' => 'sanitize_text_field',
                ],
            ],
        ] );

        // GET /wp-json/evas-tts/v1/languages
        register_rest_route( $this->namespace, '/languages', [
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => [ $this, 'get_languages' ],
            'permission_callback' => '__return_true',
        ] );

        // GET/POST /wp-json/evas-tts/v1/settings
        register_rest_route( $this->namespace, '/settings', [
            [
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => [ $this, 'get_settings' ],
                'permission_callback' => [ $this, 'get_permissions_check' ],
            ],
            [
                'methods'             => WP_REST_Server::EDITABLE,
                'callback'            => [ $this, 'update_settings' ],
                'permission_callback' => [ $this, 'update_permissions_check' ],
            ],
        ] );

        // GET /wp-json/evas-tts/v1/cache/stats
        register_rest_route( $this->namespace, '/cache/stats', [
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => [ $this, 'get_cache_stats' ],
            'permission_callback' => [ $this, 'get_permissions_check' ],
        ] );

        // POST /wp-json/evas-tts/v1/cache/clear
        register_rest_route( $this->namespace, '/cache/clear', [
            'methods'             => WP_REST_Server::CREATABLE,
            'callback'            => [ $this, 'clear_cache' ],
            'permission_callback' => [ $this, 'update_permissions_check' ],
        ] );

        // GET /wp-json/evas-tts/v1/shortcodes
        register_rest_route( $this->namespace, '/shortcodes', [
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => [ $this, 'get_shortcodes' ],
            'permission_callback' => '__return_true',
        ] );
    }

    /**
     * Generate speech from text
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response|WP_Error
     */
    public function generate_speech( WP_REST_Request $request ) {
        // Require a lightweight token to prevent off-site abuse of the endpoint.
        $auth = $this->check_guest_token( $request );
        if ( is_wp_error( $auth ) ) {
            return $auth;
        }

        // Rate limit (guest-safe): prevent abuse of Google API key and server resources.
        $rl = $this->check_rate_limit( $request );
        if ( is_wp_error( $rl ) ) {
            return $rl;
        }

        $text     = $request->get_param( 'text' );
        $language = $request->get_param( 'language' );

        if ( empty( $text ) ) {
            return new WP_Error(
                'empty_text',
                __( 'No text provided', 'everyone-accessibility-suite' ),
                [ 'status' => 400 ]
            );
        }

        // Anti-DoS: hard limit on raw incoming text size.
        if ( is_string( $text ) && strlen( $text ) > self::MAX_TEXT_LEN_RAW ) {
            return new WP_Error(
                'text_too_long',
                __( 'Text is too long.', 'everyone-accessibility-suite' ),
                [ 'status' => 400 ]
            );
        }

        try {
            $generator = new EVAS_TTS_Speech_Generator();
            $audio     = $generator->generate_speech( $text, $language ?: null );

            return rest_ensure_response( [
                'success' => true,
                'audio'   => $audio,
            ] );
        } catch ( Exception $e ) {
            return new WP_Error(
                'generation_failed',
                $e->getMessage(),
                [ 'status' => 500 ]
            );
        }
    }

    /**
     * Check and increment rate limiting counters for guest requests.
     *
     * @param WP_REST_Request $request Request object.
     * @return true|WP_Error True if allowed, WP_Error (429) if limited.
     */
    private function check_rate_limit( WP_REST_Request $request ) {
        $ip = $this->get_client_ip();
        if ( empty( $ip ) ) {
            // If IP cannot be determined, allow (do not block legitimate users behind proxies),
            // but keep other protections (token, length limits) in place.
            return true;
        }

        $ip_hash = hash( 'sha256', $ip );

        // 10-minute window counter.
        $key_10m   = 'evas_tts_rl_10m_' . $ip_hash;
        $count_10m = (int) get_transient( $key_10m );
        $count_10m++;
        set_transient( $key_10m, $count_10m, self::RL_WINDOW_10M );

        if ( $count_10m > self::RL_MAX_10M ) {
            $error = new WP_Error(
                'evas_tts_rate_limited',
                __( 'Too many Text-to-Speech requests. Please try again later.', 'everyone-accessibility-suite' ),
                [
                    'status'      => 429,
                    'retry_after' => self::RL_WINDOW_10M,
                ]
            );
            $error->add_data( [ 'headers' => [ 'Retry-After' => (string) self::RL_WINDOW_10M ] ] );
            return $error;
        }

        // Daily counter.
        $key_day   = 'evas_tts_rl_day_' . $ip_hash;
        $count_day = (int) get_transient( $key_day );
        $count_day++;
        set_transient( $key_day, $count_day, DAY_IN_SECONDS );

        if ( $count_day > self::RL_MAX_DAY ) {
            $error = new WP_Error(
                'evas_tts_rate_limited_daily',
                __( 'Daily Text-to-Speech limit reached. Please try again tomorrow.', 'everyone-accessibility-suite' ),
                [
                    'status'      => 429,
                    'retry_after' => DAY_IN_SECONDS,
                ]
            );
            $error->add_data( [ 'headers' => [ 'Retry-After' => (string) DAY_IN_SECONDS ] ] );
            return $error;
        }

        return true;
    }

    /**
     * Get client IP address for rate limiting.
     *
     * @return string Client IP or empty string.
     */
    private function get_client_ip(): string {
        $remote_addr = '';

        if ( isset( $_SERVER['REMOTE_ADDR'] ) ) {
            $remote_addr = sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) );
        }

        // Best-effort: respect first IP from X-Forwarded-For if it is present and valid.
        // Note: This header can be spoofed if not set by a trusted proxy; REMOTE_ADDR remains the primary signal.
        if ( ! empty( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ) {
            $xff   = sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_FORWARDED_FOR'] ) );
            $parts = array_map( 'trim', explode( ',', $xff ) );
            if ( ! empty( $parts[0] ) && filter_var( $parts[0], FILTER_VALIDATE_IP ) ) {
                return $parts[0];
            }
        }

        if ( ! empty( $remote_addr ) && filter_var( $remote_addr, FILTER_VALIDATE_IP ) ) {
            return $remote_addr;
        }

        return '';
    }

    /**
     * Validate guest token / REST nonce for public generate endpoint.
     *
     * Accepts either:
     * - `X-WP-Nonce` header for `wp_rest` action (works for guests too)
     * - `X-EVAS-TTS-Token` header (daily rotating HMAC)
     *
     * @param WP_REST_Request $request Request object.
     * @return true|WP_Error True if allowed, WP_Error (403) otherwise.
     */
    private function check_guest_token( WP_REST_Request $request ) {
        $rest_nonce = $request->get_header( 'X-WP-Nonce' );
        if ( $rest_nonce && wp_verify_nonce( $rest_nonce, 'wp_rest' ) ) {
            return true;
        }

        $token = $request->get_header( 'X-EVAS-TTS-Token' );
        if ( $token ) {
            $expected = hash_hmac( 'sha256', gmdate( 'Y-m-d' ), wp_salt( 'evas_tts_guest' ) );
            if ( hash_equals( $expected, (string) $token ) ) {
                return true;
            }
        }

        return new WP_Error(
            'evas_tts_forbidden',
            __( 'Invalid security token.', 'everyone-accessibility-suite' ),
            [ 'status' => 403 ]
        );
    }

    /**
     * Get available voices
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response|WP_Error
     */
    public function get_voices( WP_REST_Request $request ) {
        $language = $request->get_param( 'language' );

        $generator = new EVAS_TTS_Speech_Generator();
        $voices    = $generator->get_voices( $language ?: null );

        if ( is_wp_error( $voices ) ) {
            return $voices;
        }

        return rest_ensure_response( [
            'success' => true,
            'voices'  => $voices,
            'count'   => count( $voices ),
        ] );
    }

    /**
     * Get supported languages
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response
     */
    public function get_languages( WP_REST_Request $request ): WP_REST_Response {
        return rest_ensure_response( [
            'success'   => true,
            'languages' => EVAS_TTS_Languages::get_languages(),
        ] );
    }

    /**
     * Get TTS settings
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response
     */
    public function get_settings( WP_REST_Request $request ): WP_REST_Response {
        $generator = new EVAS_TTS_Speech_Generator();

        return rest_ensure_response( [
            'success'  => true,
            'settings' => [
                'api_key'       => get_option( 'evas_tts_api_key', '' ) ? '***configured***' : '',
                'has_api_key'   => $generator->has_api_key(),
                'language'      => get_option( 'evas_tts_language', 'en-US' ),
                'voice_name'    => get_option( 'evas_tts_voice_name', 'en-US-Wavenet-D' ),
                'speaking_rate' => (float) get_option( 'evas_tts_speaking_rate', 1.0 ),
                'pitch'         => (float) get_option( 'evas_tts_pitch', 0.0 ),
                'cache_enabled' => (bool) get_option( 'evas_tts_cache_enabled', true ),
            ],
        ] );
    }

    /**
     * Update TTS settings
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response
     */
    public function update_settings( WP_REST_Request $request ): WP_REST_Response {
        $params = $request->get_json_params();

        $allowed_settings = [
            'api_key'       => 'sanitize_text_field',
            'language'      => 'sanitize_text_field',
            'voice_name'    => 'sanitize_text_field',
            'speaking_rate' => 'floatval',
            'pitch'         => 'floatval',
            'cache_enabled' => 'rest_sanitize_boolean',
        ];

        foreach ( $params as $key => $value ) {
            if ( isset( $allowed_settings[ $key ] ) ) {
                $sanitize = $allowed_settings[ $key ];
                $value    = $sanitize( $value );
                update_option( 'evas_tts_' . $key, $value );
            }
        }

        return rest_ensure_response( [
            'success' => true,
            'message' => __( 'Settings saved successfully', 'everyone-accessibility-suite' ),
        ] );
    }

    /**
     * Get cache statistics
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response
     */
    public function get_cache_stats( WP_REST_Request $request ): WP_REST_Response {
        $cache = new EVAS_TTS_Cache();

        return rest_ensure_response( [
            'success' => true,
            'stats'   => $cache->get_stats(),
        ] );
    }

    /**
     * Clear cache
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response
     */
    public function clear_cache( WP_REST_Request $request ): WP_REST_Response {
        $cache   = new EVAS_TTS_Cache();
        $deleted = $cache->clear();

        return rest_ensure_response( [
            'success' => true,
            'message' => sprintf(
                /* translators: %d: Number of deleted files */
                __( 'Cleared %d cached audio files', 'everyone-accessibility-suite' ),
                $deleted
            ),
            'deleted' => $deleted,
        ] );
    }

    /**
     * Get shortcodes info
     *
     * @param WP_REST_Request $request Request object.
     * @return WP_REST_Response
     */
    public function get_shortcodes( WP_REST_Request $request ): WP_REST_Response {
        return rest_ensure_response( [
            'success'    => true,
            'shortcodes' => EVAS_TTS_Shortcodes::get_shortcodes_info(),
        ] );
    }

    /**
     * Check read permissions
     *
     * @param WP_REST_Request $request Request object.
     * @return bool
     */
    public function get_permissions_check( WP_REST_Request $request ): bool {
        return current_user_can( 'manage_options' );
    }

    /**
     * Check update permissions
     *
     * @param WP_REST_Request $request Request object.
     * @return bool
     */
    public function update_permissions_check( WP_REST_Request $request ): bool {
        return current_user_can( 'manage_options' );
    }
}

