/**
 * EVAS Text-to-Speech
 *
 * JavaScript for text-to-speech functionality.
 *
 * @package Everyone_Accessibility_Suite
 * @version 1.0.0
 */

(function($) {
    'use strict';

    /**
     * EVAS Text-to-Speech Controller
     */
    const EVAS_TTS = {
        settings: window.evasTTS?.settings || {},
        i18n: window.evasTTS?.i18n || {},
        ajaxUrl: window.evasTTS?.ajaxUrl || '',
        restUrl: window.evasTTS?.restUrl || '',
        nonce: window.evasTTS?.nonce || '',
        restNonce: window.evasTTS?.restNonce || '',
        guestToken: window.evasTTS?.guestToken || '',
        hasApiKey: window.evasTTS?.hasApiKey || false,

        audio: null,
        isPlaying: false,
        currentSource: null,

        /**
         * Initialize TTS
         */
        init() {
            if (!this.hasApiKey) {
                console.log('EVAS TTS: No API key configured');
                return;
            }

            this.bindEvents();
            this.createAudioElement();
        },

        /**
         * Bind event handlers
         */
        bindEvents() {
            // Read selection button
            $(document).on('click', '#evas-tts-read-selection', () => {
                this.readSelection();
            });

            // Read page button
            $(document).on('click', '#evas-tts-read-page', () => {
                this.readPage();
            });

            // Stop button
            $(document).on('click', '#evas-tts-stop', () => {
                this.stop();
            });

            // Read button shortcode
            $(document).on('click', '.evas-tts-read-button', (e) => {
                const target = $(e.currentTarget).data('target');
                if (target) {
                    this.readElement(target);
                }
            });

            // Selection-based reading (double-click to read)
            $(document).on('dblclick', '[data-evas-tts-dblclick]', (e) => {
                const text = this.getSelectedText() || $(e.currentTarget).text();
                if (text) {
                    this.speak(text);
                }
            });
        },

        /**
         * Create audio element
         */
        createAudioElement() {
            this.audio = new Audio();
            
            this.audio.addEventListener('ended', () => {
                this.onPlaybackEnd();
            });

            this.audio.addEventListener('error', (e) => {
                console.error('EVAS TTS: Audio error', e);
                this.showStatus(this.i18n.error, 'error');
                this.resetUI();
            });

            this.audio.addEventListener('play', () => {
                this.isPlaying = true;
                this.updateUI();
            });

            this.audio.addEventListener('pause', () => {
                this.isPlaying = false;
                this.updateUI();
            });
        },

        /**
         * Read selected text
         */
        readSelection() {
            const text = this.getSelectedText();
            
            if (!text) {
                this.showStatus(this.i18n.noSelection, 'warning');
                return;
            }

            this.speak(text, 'selection');
        },

        /**
         * Read entire page content
         */
        readPage() {
            // Get main content - try common selectors
            const selectors = [
                'article',
                '[role="main"]',
                'main',
                '.entry-content',
                '.post-content',
                '.content',
                '#content'
            ];

            let content = '';
            
            for (const selector of selectors) {
                const $el = $(selector).first();
                if ($el.length) {
                    content = this.extractText($el);
                    break;
                }
            }

            if (!content) {
                content = this.extractText($('body'));
            }

            if (content) {
                this.speak(content, 'page');
            } else {
                this.showStatus(this.i18n.error, 'error');
            }
        },

        /**
         * Read specific element by selector
         *
         * @param {string} selector CSS selector
         */
        readElement(selector) {
            const $el = $(selector);
            
            if (!$el.length) {
                console.warn('EVAS TTS: Element not found:', selector);
                return;
            }

            const text = this.extractText($el);
            
            if (text) {
                this.speak(text, 'element');
            }
        },

        /**
         * Speak text
         *
         * @param {string} text Text to speak
         * @param {string} source Source type
         */
        async speak(text, source = 'manual') {
            if (this.isPlaying) {
                this.stop();
            }

            this.currentSource = source;
            this.showStatus(this.i18n.loading, 'loading');

            try {
                const response = await this.generateSpeech(text);
                
                if (response.success && response.audio) {
                    this.playAudio(response.audio);
                } else {
                    throw new Error(response.message || 'Failed to generate speech');
                }
            } catch (error) {
                console.error('EVAS TTS:', error);
                this.showStatus(this.i18n.error + ': ' + error.message, 'error');
                this.resetUI();
            }
        },

        /**
         * Generate speech via API
         *
         * @param {string} text Text to convert
         * @returns {Promise<Object>}
         */
        async generateSpeech(text) {
            const language = document.documentElement.lang || this.settings.language || 'en-US';

            const response = await fetch(this.restUrl + 'generate', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    ...(this.restNonce ? { 'X-WP-Nonce': this.restNonce } : {}),
                    ...(this.guestToken ? { 'X-EVAS-TTS-Token': this.guestToken } : {}),
                },
                body: JSON.stringify({
                    text: text,
                    language: language
                })
            });

            return await response.json();
        },

        /**
         * Play audio from base64
         *
         * @param {string} base64Audio Base64 encoded audio
         */
        playAudio(base64Audio) {
            const audioSrc = 'data:audio/mp3;base64,' + base64Audio;
            
            this.audio.src = audioSrc;
            this.audio.play().then(() => {
                this.showStatus(this.i18n.reading, 'playing');
            }).catch(error => {
                console.error('EVAS TTS: Playback error', error);
                this.showStatus(this.i18n.error, 'error');
                this.resetUI();
            });
        },

        /**
         * Stop playback
         */
        stop() {
            if (this.audio) {
                this.audio.pause();
                this.audio.currentTime = 0;
            }
            
            this.isPlaying = false;
            this.showStatus(this.i18n.stopped, 'stopped');
            this.resetUI();
        },

        /**
         * Handle playback end
         */
        onPlaybackEnd() {
            this.isPlaying = false;
            this.showStatus(this.i18n.finished, 'finished');
            this.resetUI();

            // Trigger event
            $(document).trigger('evas:tts:ended', { source: this.currentSource });
        },

        /**
         * Get selected text
         *
         * @returns {string}
         */
        getSelectedText() {
            let text = '';
            
            if (window.getSelection) {
                text = window.getSelection().toString();
            } else if (document.selection && document.selection.type !== 'Control') {
                text = document.selection.createRange().text;
            }

            return text.trim();
        },

        /**
         * Extract readable text from element
         *
         * @param {jQuery} $el Element
         * @returns {string}
         */
        extractText($el) {
            // Clone element to work with
            const $clone = $el.clone();

            // Remove muted elements
            $clone.find('.evas-tts-mute, [data-evas-mute]').remove();

            // Remove script, style, hidden elements
            $clone.find('script, style, [hidden], .hidden, [aria-hidden="true"]').remove();

            // Remove navigation, header, footer if present
            $clone.find('nav, .nav, .navigation, .menu, .sidebar, .widget, .comments').remove();

            // Get text content
            let text = $clone.text();

            // Clean up
            text = text.replace(/\s+/g, ' ').trim();

            return text;
        },

        /**
         * Show status message
         *
         * @param {string} message Status message
         * @param {string} type Status type
         */
        showStatus(message, type = 'info') {
            const $status = $('#evas-tts-status');
            
            if ($status.length) {
                $status
                    .text(message)
                    .attr('data-status', type)
                    .show();

                // Auto-hide for non-persistent statuses
                if (['finished', 'stopped', 'warning'].includes(type)) {
                    setTimeout(() => {
                        $status.fadeOut();
                    }, 3000);
                }
            }
        },

        /**
         * Update UI based on playback state
         */
        updateUI() {
            const $readSelection = $('#evas-tts-read-selection');
            const $readPage = $('#evas-tts-read-page');
            const $stop = $('#evas-tts-stop');

            if (this.isPlaying) {
                $readSelection.hide();
                $readPage.hide();
                $stop.show();
            } else {
                $readSelection.show();
                $readPage.show();
                $stop.hide();
            }
        },

        /**
         * Reset UI to initial state
         */
        resetUI() {
            this.updateUI();
        }
    };

    // Initialize when DOM is ready
    $(document).ready(() => EVAS_TTS.init());

    // Expose for external access
    window.EVAS_TTS = EVAS_TTS;

})(jQuery);

