<?php
/**
 * Customizer REST API Controller
 *
 * @package     Everyone_Accessibility_Suite
 * @subpackage  Modules/Customizer
 * @version     1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class EVAS_Customizer_REST_Controller
 */
class EVAS_Customizer_REST_Controller extends WP_REST_Controller {

    /**
     * Namespace
     * 
     * @var string
     */
    protected $namespace = 'evas/v1';

    /**
     * Resource name
     * 
     * @var string
     */
    protected $rest_base = 'customizer';

    /**
     * Register REST routes
     */
    public function register_routes() {
        // Get settings
        register_rest_route( $this->namespace, '/' . $this->rest_base . '/settings', [
            [
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => [ $this, 'get_settings' ],
                'permission_callback' => [ $this, 'get_settings_permissions_check' ],
            ],
        ] );

        // Save settings
        register_rest_route( $this->namespace, '/' . $this->rest_base . '/settings', [
            [
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => [ $this, 'update_settings' ],
                'permission_callback' => [ $this, 'update_settings_permissions_check' ],
                'args'                => $this->get_settings_schema(),
            ],
        ] );

        // Get icons
        register_rest_route( $this->namespace, '/' . $this->rest_base . '/icons', [
            [
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => [ $this, 'get_icons' ],
                'permission_callback' => [ $this, 'get_settings_permissions_check' ],
                'args'                => [
                    'library' => [
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                    'search' => [
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                ],
            ],
        ] );

        // Reset settings to defaults
        register_rest_route( $this->namespace, '/' . $this->rest_base . '/reset', [
            [
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => [ $this, 'reset_settings' ],
                'permission_callback' => [ $this, 'update_settings_permissions_check' ],
            ],
        ] );
    }

    /**
     * Get settings
     * 
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function get_settings( $request ) {
        $settings = [];
        
        // Get all button settings
        $button_options = [
            'show_open_button', 'button_tabindex', 'button_position', 'button_caption', 'button_icon',
            'button_icon_position', 'button_size', 'button_margin', 'button_padding', 'button_border_radius',
            'button_color', 'button_color_hover', 'button_bgcolor', 'button_bgcolor_hover',
            'button_entrance_timeout', 'button_entrance_animation', 'button_hover_animation',
            'button_show_after_scrolling_desktop', 'button_show_after_scrolling_mobile'
        ];
        
        foreach ( $button_options as $option ) {
            $value = get_option( 'evas_cust_' . $option );
            if ( $value !== false ) {
                $settings[ $option ] = $value;
            }
        }
        
        // Get all popup settings
        $popup_options = [
            'popup_position', 'popup_draggable', 'popup_background_color', 'popup_key_color',
            'popup_text_color', 'popup_background_color_dark', 'popup_key_color_dark',
            'popup_text_color_dark', 'popup_border_radius', 'popup_shadow', 'popup_overlay',
            'popup_overlay_color', 'popup_close_anywhere', 'reset_button', 'hide_button'
        ];
        
        foreach ( $popup_options as $option ) {
            $value = get_option( 'evas_cust_' . $option );
            if ( $value !== false ) {
                $settings[ $option ] = $value;
            }
        }
        
        return rest_ensure_response( $settings );
    }

    /**
     * Update settings
     * 
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function update_settings( $request ) {
        $params = $request->get_json_params();
        
        if ( empty( $params ) ) {
            return new WP_Error( 'no_settings', __( 'No settings provided', 'everyone-accessibility-suite' ), [ 'status' => 400 ] );
        }
        
        $updated = [];
        
        foreach ( $params as $key => $value ) {
            $option_name = 'evas_cust_' . $key;
            
            // Sanitize based on type
            if ( is_bool( $value ) ) {
                $value = (bool) $value;
            } elseif ( is_numeric( $value ) ) {
                $value = is_float( $value ) ? (float) $value : (int) $value;
            } else {
                $value = sanitize_text_field( $value );
            }
            
            update_option( $option_name, $value );
            $updated[ $key ] = $value;
        }
        
        return rest_ensure_response( [
            'success' => true,
            'message' => __( 'Settings updated successfully', 'everyone-accessibility-suite' ),
            'data'    => $updated,
        ] );
    }

    /**
     * Get icons
     * 
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function get_icons( $request ) {
        $library = $request->get_param( 'library' );
        $search = $request->get_param( 'search' );
        
        $icon_library = new EVAS_Icon_Library();
        
        if ( $library ) {
            $icons = $icon_library->get_icons_by_library( $library );
        } else {
            $icons = $icon_library->get_all_icons();
        }
        
        if ( $search ) {
            $icons = $icon_library->search_icons( $search, $icons );
        }
        
        return rest_ensure_response( $icons );
    }

    /**
     * Check permissions for getting settings
     * 
     * @param WP_REST_Request $request
     * @return bool
     */
    public function get_settings_permissions_check( $request ) {
        return current_user_can( 'manage_options' );
    }

    /**
     * Check permissions for updating settings
     * 
     * @param WP_REST_Request $request
     * @return bool
     */
    public function update_settings_permissions_check( $request ) {
        return current_user_can( 'manage_options' );
    }

    /**
     * Reset settings to defaults
     * 
     * @param WP_REST_Request $request
     * @return WP_REST_Response|WP_Error
     */
    public function reset_settings( $request ) {
        $defaults = $this->get_default_settings();
        
        foreach ( $defaults as $key => $value ) {
            delete_option( 'evas_cust_' . $key );
        }
        
        return rest_ensure_response( [
            'success' => true,
            'message' => __( 'Settings reset to defaults successfully', 'everyone-accessibility-suite' ),
            'data'    => $defaults,
        ] );
    }

    /**
     * Get default settings
     * 
     * @return array
     */
    private function get_default_settings() {
        return [
            // Open Button defaults
            'show_open_button'                   => true,
            'button_tabindex'                    => 0,
            'button_position'                    => 'bottom-right',
            'button_caption'                     => '',
            'button_icon'                        => 'default',
            'button_icon_position'               => 'before',
            'button_size'                        => 24,
            'button_margin'                      => 10,
            'button_padding'                     => 20,
            'button_border_radius'               => 50,
            'button_color'                       => '#ffffff',
            'button_color_hover'                 => '#216ff3',
            'button_bgcolor'                     => '#216ff3',
            'button_bgcolor_hover'               => '#ffffff',
            'button_entrance_timeout'            => 0,
            'button_entrance_animation'          => 'fade',
            'button_hover_animation'             => 'none',
            'button_show_after_scrolling_desktop' => 0,
            'button_show_after_scrolling_mobile'  => 0,
            
            // Modal Popup defaults
            'popup_position'              => 'right',
            'popup_draggable'             => true,
            'popup_background_color'      => '#ffffff',
            'popup_key_color'             => '#216ff3',
            'popup_text_color'            => '#333333',
            'popup_background_color_dark' => '#16191b',
            'popup_key_color_dark'        => '#216ff3',
            'popup_text_color_dark'       => '#deeffd',
            'popup_border_radius'         => 20,
            'popup_shadow'                => true,
            'popup_overlay'               => false,
            'popup_overlay_color'         => '#0253ee',
            'popup_close_anywhere'        => false,
            'reset_button'                => true,
            'hide_button'                 => true,
        ];
    }

    /**
     * Get settings schema
     * 
     * @return array
     */
    private function get_settings_schema() {
        return [
            'type'       => 'object',
            'properties' => [
                'show_open_button'                   => [ 'type' => 'boolean' ],
                'button_tabindex'                    => [ 'type' => 'integer' ],
                'button_position'                    => [ 'type' => 'string' ],
                'button_caption'                     => [ 'type' => 'string' ],
                'button_icon'                        => [ 'type' => 'string' ],
                'button_icon_position'               => [ 'type' => 'string' ],
                'button_size'                        => [ 'type' => 'integer' ],
                'button_margin'                      => [ 'type' => 'integer' ],
                'button_padding'                     => [ 'type' => 'integer' ],
                'button_border_radius'               => [ 'type' => 'integer' ],
                'button_color'                       => [ 'type' => 'string' ],
                'button_color_hover'                 => [ 'type' => 'string' ],
                'button_bgcolor'                     => [ 'type' => 'string' ],
                'button_bgcolor_hover'               => [ 'type' => 'string' ],
                'button_entrance_timeout'            => [ 'type' => 'integer' ],
                'button_entrance_animation'          => [ 'type' => 'string' ],
                'button_hover_animation'             => [ 'type' => 'string' ],
                'button_show_after_scrolling_desktop' => [ 'type' => 'integer' ],
                'button_show_after_scrolling_mobile'  => [ 'type' => 'integer' ],
                'popup_position'                     => [ 'type' => 'string' ],
                'popup_draggable'                    => [ 'type' => 'boolean' ],
                'popup_background_color'             => [ 'type' => 'string' ],
                'popup_key_color'                    => [ 'type' => 'string' ],
                'popup_text_color'                   => [ 'type' => 'string' ],
                'popup_background_color_dark'        => [ 'type' => 'string' ],
                'popup_key_color_dark'               => [ 'type' => 'string' ],
                'popup_text_color_dark'              => [ 'type' => 'string' ],
                'popup_border_radius'                => [ 'type' => 'integer' ],
                'popup_shadow'                       => [ 'type' => 'boolean' ],
                'popup_overlay'                      => [ 'type' => 'boolean' ],
                'popup_overlay_color'                => [ 'type' => 'string' ],
                'popup_close_anywhere'               => [ 'type' => 'boolean' ],
                'reset_button'                       => [ 'type' => 'boolean' ],
                'hide_button'                        => [ 'type' => 'boolean' ],
            ],
        ];
    }
}

