<?php
/**
 * Icon Library Class
 *
 * @package     Everyone_Accessibility_Suite
 * @subpackage  Modules/Customizer
 * @version     1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class EVAS_Icon_Library
 */
class EVAS_Icon_Library {

    /**
     * Get available icon libraries
     * 
     * @return array
     */
    public function get_icon_libraries() {
        return [
            'custom' => [
                'name'        => __( 'Custom Icons', 'everyone-accessibility-suite' ),
                'description' => __( 'Built-in plugin icons', 'everyone-accessibility-suite' ),
            ],
            'fontawesome' => [
                'name'        => __( 'Font Awesome', 'everyone-accessibility-suite' ),
                'description' => __( 'Popular icon library', 'everyone-accessibility-suite' ),
            ],
            'material' => [
                'name'        => __( 'Material Icons', 'everyone-accessibility-suite' ),
                'description' => __( 'Google Material Design icons', 'everyone-accessibility-suite' ),
            ],
        ];
    }

    /**
     * Get all icons
     * 
     * @return array
     */
    public function get_all_icons() {
        $icons = [];
        
        $libraries = $this->get_icon_libraries();
        foreach ( $libraries as $library_id => $library_data ) {
            $icons[ $library_id ] = $this->get_icons_by_library( $library_id );
        }
        
        return $icons;
    }

    /**
     * Get icons by library
     * 
     * @param string $library
     * @return array
     */
    public function get_icons_by_library( $library ) {
        switch ( $library ) {
            case 'custom':
                return $this->get_custom_icons();
            
            case 'fontawesome':
                return $this->get_fontawesome_icons();
            
            case 'material':
                return $this->get_material_icons();
            
            default:
                return [];
        }
    }

    /**
     * Get custom icons from plugin
     * 
     * @return array
     */
    private function get_custom_icons() {
        $icons_dir = EVAS_PLUGIN_DIR . 'assets/accessibility/images/toggles/';
        $icons_url = EVAS_PLUGIN_URL . 'assets/accessibility/images/toggles/';
        
        $icons = [];
        
        if ( is_dir( $icons_dir ) ) {
            $files = glob( $icons_dir . '*.svg' );
            
            foreach ( $files as $file ) {
                $filename = basename( $file, '.svg' );
                $icons[] = [
                    'id'   => $filename,
                    'name' => ucwords( str_replace( '-', ' ', $filename ) ),
                    'url'  => $icons_url . basename( $file ),
                    'svg'  => file_get_contents( $file ),
                ];
            }
        }
        
        return $icons;
    }

    /**
     * Get Font Awesome icons (common ones)
     * 
     * @return array
     */
    private function get_fontawesome_icons() {
        $icons = [
            'universal-access' => 'Universal Access',
            'wheelchair' => 'Wheelchair',
            'eye' => 'Eye',
            'eye-slash' => 'Eye Slash',
            'deaf' => 'Deaf',
            'blind' => 'Blind',
            'low-vision' => 'Low Vision',
            'assistive-listening-systems' => 'Assistive Listening',
            'sign-language' => 'Sign Language',
            'braille' => 'Braille',
            'audio-description' => 'Audio Description',
            'closed-captioning' => 'Closed Captioning',
            'user-check' => 'User Check',
            'user-cog' => 'User Settings',
            'sliders-h' => 'Sliders',
            'cog' => 'Settings',
            'paint-brush' => 'Paint Brush',
            'palette' => 'Palette',
            'adjust' => 'Adjust',
            'sun' => 'Sun',
            'moon' => 'Moon',
            'text-height' => 'Text Height',
            'text-width' => 'Text Width',
            'font' => 'Font',
            'bold' => 'Bold',
            'italic' => 'Italic',
            'underline' => 'Underline',
            'align-left' => 'Align Left',
            'align-center' => 'Align Center',
            'align-right' => 'Align Right',
            'keyboard' => 'Keyboard',
            'mouse-pointer' => 'Mouse Pointer',
            'microphone' => 'Microphone',
            'volume-up' => 'Volume Up',
            'volume-mute' => 'Volume Mute',
        ];
        
        $formatted_icons = [];
        foreach ( $icons as $id => $name ) {
            $formatted_icons[] = [
                'id'   => 'fa-' . $id,
                'name' => $name,
                'svg'  => '<i class="fas fa-' . $id . '"></i>',
            ];
        }
        
        return $formatted_icons;
    }

    /**
     * Get Material icons (common ones)
     * 
     * @return array
     */
    private function get_material_icons() {
        $icons = [
            'accessibility' => 'Accessibility',
            'accessibility_new' => 'Accessibility New',
            'accessible' => 'Accessible',
            'accessible_forward' => 'Accessible Forward',
            'visibility' => 'Visibility',
            'visibility_off' => 'Visibility Off',
            'hearing' => 'Hearing',
            'hearing_disabled' => 'Hearing Disabled',
            'record_voice_over' => 'Record Voice Over',
            'settings_voice' => 'Settings Voice',
            'settings' => 'Settings',
            'tune' => 'Tune',
            'palette' => 'Palette',
            'color_lens' => 'Color Lens',
            'brightness_medium' => 'Brightness Medium',
            'brightness_high' => 'Brightness High',
            'brightness_low' => 'Brightness Low',
            'dark_mode' => 'Dark Mode',
            'light_mode' => 'Light Mode',
            'format_size' => 'Format Size',
            'text_fields' => 'Text Fields',
            'title' => 'Title',
            'format_bold' => 'Format Bold',
            'format_italic' => 'Format Italic',
            'format_underlined' => 'Format Underlined',
            'format_align_left' => 'Format Align Left',
            'format_align_center' => 'Format Align Center',
            'format_align_right' => 'Format Align Right',
            'keyboard' => 'Keyboard',
            'mouse' => 'Mouse',
            'mic' => 'Mic',
            'volume_up' => 'Volume Up',
            'volume_off' => 'Volume Off',
        ];
        
        $formatted_icons = [];
        foreach ( $icons as $id => $name ) {
            $formatted_icons[] = [
                'id'   => 'material-' . $id,
                'name' => $name,
                'svg'  => '<span class="material-icons">' . $id . '</span>',
            ];
        }
        
        return $formatted_icons;
    }

    /**
     * Search icons
     * 
     * @param string $query
     * @param array $icons
     * @return array
     */
    public function search_icons( $query, $icons = null ) {
        if ( $icons === null ) {
            $icons = $this->get_all_icons();
        }
        
        $query = strtolower( $query );
        $results = [];
        
        foreach ( $icons as $library_id => $library_icons ) {
            $filtered = array_filter( $library_icons, function( $icon ) use ( $query ) {
                return strpos( strtolower( $icon['name'] ), $query ) !== false ||
                       strpos( strtolower( $icon['id'] ), $query ) !== false;
            } );
            
            if ( ! empty( $filtered ) ) {
                $results[ $library_id ] = array_values( $filtered );
            }
        }
        
        return $results;
    }

    /**
     * Render icon
     * 
     * @param string $icon
     * @param string $library
     * @return string
     */
    public function render_icon( $icon, $library = 'custom' ) {
        $icons = $this->get_icons_by_library( $library );
        
        foreach ( $icons as $icon_data ) {
            if ( $icon_data['id'] === $icon ) {
                return $icon_data['svg'];
            }
        }
        
        return '';
    }
}

