<?php
/**
 * Customizer Module Main Class
 *
 * @package     Everyone_Accessibility_Suite
 * @subpackage  Modules/Customizer
 * @version     1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class EVAS_Customizer_Module
 */
class EVAS_Customizer_Module extends EVAS_Module_Base {

    public function __construct( $settings_manager = null, $settings = array() ) {
        $this->module_id      = 'customizer';
        $this->module_name    = __( 'Customizer', 'everyone-accessibility-suite' );
        $this->module_version = defined( 'EVAS_CUSTOMIZER_VERSION' ) ? EVAS_CUSTOMIZER_VERSION : '1.0.0';
        
        parent::__construct( $settings_manager, $settings );
    }

    /**
     * Get module settings
     * 
     * @return array
     */
    public function get_module_settings() {
        return array_merge(
            $this->get_button_settings(),
            $this->get_popup_settings()
        );
    }

    /**
     * Get Open Button settings
     * 
     * @return array
     */
    private function get_button_settings() {
        return [
            // Основные
            'show_open_button'                   => (bool) get_option( 'evas_cust_show_open_button', true ),
            'button_tabindex'                    => (int) get_option( 'evas_cust_button_tabindex', 0 ),
            'button_position'                    => get_option( 'evas_cust_button_position', 'bottom-right' ),
            'button_caption'                     => get_option( 'evas_cust_button_caption', '' ),
            'button_icon'                        => get_option( 'evas_cust_button_icon', 'default' ),
            'button_icon_position'               => get_option( 'evas_cust_button_icon_position', 'before' ),
            
            // Размеры
            'button_size'                        => (int) get_option( 'evas_cust_button_size', 24 ),
            'button_margin'                      => (int) get_option( 'evas_cust_button_margin', 10 ),
            'button_padding'                     => (int) get_option( 'evas_cust_button_padding', 20 ),
            'button_border_radius'               => (int) get_option( 'evas_cust_button_border_radius', 50 ),
            
            // Цвета
            'button_color'                       => get_option( 'evas_cust_button_color', '#ffffff' ),
            'button_color_hover'                 => get_option( 'evas_cust_button_color_hover', 'rgba(33, 111, 243, 1)' ),
            'button_bgcolor'                     => get_option( 'evas_cust_button_bgcolor', 'rgba(33, 111, 243, 1)' ),
            'button_bgcolor_hover'               => get_option( 'evas_cust_button_bgcolor_hover', '#ffffff' ),
            
            // Анимация
            'button_entrance_timeout'            => (int) get_option( 'evas_cust_button_entrance_timeout', 0 ),
            'button_entrance_animation'          => get_option( 'evas_cust_button_entrance_animation', 'fade' ),
            'button_hover_animation'             => get_option( 'evas_cust_button_hover_animation', 'none' ),
            
            // Показ после прокрутки
            'button_show_after_scrolling_desktop' => (int) get_option( 'evas_cust_button_show_after_scrolling_desktop', 0 ),
            'button_show_after_scrolling_mobile'  => (int) get_option( 'evas_cust_button_show_after_scrolling_mobile', 0 ),
        ];
    }

    /**
     * Get Modal Popup settings
     * 
     * @return array
     */
    private function get_popup_settings() {
        return [
            // Основные
            'popup_position'              => get_option( 'evas_cust_popup_position', 'right' ),
            'popup_draggable'             => (bool) get_option( 'evas_cust_popup_draggable', true ),
            
            // Светлая тема
            'popup_background_color'      => get_option( 'evas_cust_popup_background_color', '#ffffff' ),
            'popup_key_color'             => get_option( 'evas_cust_popup_key_color', 'rgba(33, 111, 243, 1)' ),
            'popup_text_color'            => get_option( 'evas_cust_popup_text_color', '#333' ),
            
            // Темная тема
            'popup_background_color_dark' => get_option( 'evas_cust_popup_background_color_dark', '#16191b' ),
            'popup_key_color_dark'        => get_option( 'evas_cust_popup_key_color_dark', 'rgba(33, 111, 243, 1)' ),
            'popup_text_color_dark'       => get_option( 'evas_cust_popup_text_color_dark', '#deeffd' ),
            
            // Внешний вид
            'popup_border_radius'         => (int) get_option( 'evas_cust_popup_border_radius', 20 ),
            'popup_shadow'                => (bool) get_option( 'evas_cust_popup_shadow', true ),
            'popup_overlay'               => (bool) get_option( 'evas_cust_popup_overlay', false ),
            'popup_overlay_color'         => get_option( 'evas_cust_popup_overlay_color', '#0253ee' ),
            
            // Поведение
            'popup_close_anywhere'        => (bool) get_option( 'evas_cust_popup_close_anywhere', false ),
            'reset_button'                => (bool) get_option( 'evas_cust_reset_button', true ),
            'hide_button'                 => (bool) get_option( 'evas_cust_hide_button', true ),
        ];
    }

    /**
     * Register module settings
     */
    public function register_settings() {
        // Register all button options
        $button_options = [
            'show_open_button', 'button_tabindex', 'button_position', 'button_caption', 'button_icon',
            'button_icon_position', 'button_size', 'button_margin', 'button_padding', 'button_border_radius',
            'button_color', 'button_color_hover', 'button_bgcolor', 'button_bgcolor_hover',
            'button_entrance_timeout', 'button_entrance_animation', 'button_hover_animation',
            'button_show_after_scrolling_desktop', 'button_show_after_scrolling_mobile'
        ];
        
        foreach ( $button_options as $option ) {
            register_setting(
                'evas_customizer_settings',
                'evas_cust_' . $option,
                array(
                    'sanitize_callback' => 'sanitize_text_field',
                )
            );
        }
        
        // Register all popup options
        $popup_options = [
            'popup_position', 'popup_draggable', 'popup_background_color', 'popup_key_color',
            'popup_text_color', 'popup_background_color_dark', 'popup_key_color_dark',
            'popup_text_color_dark', 'popup_border_radius', 'popup_shadow', 'popup_overlay',
            'popup_overlay_color', 'popup_close_anywhere', 'reset_button', 'hide_button'
        ];
        
        foreach ( $popup_options as $option ) {
            register_setting(
                'evas_customizer_settings',
                'evas_cust_' . $option,
                array(
                    'sanitize_callback' => 'sanitize_text_field',
                )
            );
        }
    }

    /**
     * Initialize module
     */
    public function init() {
        add_action( 'rest_api_init', [ $this, 'register_rest_routes' ] );
        add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_frontend_assets' ] );
        
        // Hooks for customization
        add_action( 'evas_before_panel_render', [ $this, 'inject_custom_styles' ] );
        add_filter( 'evas_panel_classes', [ $this, 'add_panel_classes' ] );
        add_filter( 'evas_button_classes', [ $this, 'add_button_classes' ] );
        add_action( 'wp_footer', [ $this, 'apply_customization' ], 999 );
    }

    /**
     * Register REST routes
     */
    public function register_rest_routes(): void {
        $controller = new EVAS_Customizer_REST_Controller();
        $controller->register_routes();
    }

    /**
     * Enqueue frontend assets
     */
    public function enqueue_frontend_assets(): void {
        if ( ! $this->is_enabled() ) {
            return;
        }

        wp_enqueue_script(
            'evas-customizer',
            EVAS_CUSTOMIZER_URL . 'assets/js/customizer.js',
            [ 'jquery' ],
            EVAS_VERSION,
            true
        );

        wp_enqueue_style(
            'evas-customizer',
            EVAS_CUSTOMIZER_URL . 'assets/css/customizer.css',
            [],
            EVAS_CUSTOMIZER_VERSION
        );

        $settings = $this->get_module_settings();
        wp_localize_script( 'evas-customizer', 'evasCustomizer', [
            'settings' => $settings,
            'ajaxurl'  => admin_url( 'admin-ajax.php' ),
            'nonce'    => wp_create_nonce( 'evas-customizer' ),
        ] );
    }

    /**
     * Inject custom styles
     */
    public function inject_custom_styles() {
        if ( ! $this->is_enabled() ) {
            return;
        }
        
        $button_styles = $this->get_button_styles();
        $popup_styles = $this->get_popup_styles();
        
        echo '<style id="evas-customizer-inline-styles">';
        echo wp_kses( $button_styles, array() );
        echo wp_kses( $popup_styles, array() );
        echo '</style>';
    }

    /**
     * Add panel classes
     * 
     * @param array $classes
     * @return array
     */
    public function add_panel_classes( $classes ) {
        if ( ! $this->is_enabled() ) {
            return $classes;
        }
        
        $settings = $this->get_popup_settings();
        
        $classes[] = 'evas-panel-position-' . $settings['popup_position'];
        
        if ( $settings['popup_draggable'] ) {
            $classes[] = 'evas-panel-draggable';
        }
        
        if ( $settings['popup_shadow'] ) {
            $classes[] = 'evas-panel-with-shadow';
        }
        
        if ( $settings['popup_overlay'] ) {
            $classes[] = 'evas-panel-with-overlay';
        }
        
        return $classes;
    }

    /**
     * Add button classes
     * 
     * @param array $classes
     * @return array
     */
    public function add_button_classes( $classes ) {
        if ( ! $this->is_enabled() ) {
            return $classes;
        }
        
        $settings = $this->get_button_settings();
        
        $classes[] = 'evas-button-position-' . str_replace( '_', '-', $settings['button_position'] );
        
        if ( $settings['button_entrance_animation'] !== 'none' ) {
            $classes[] = 'evas-entrance-' . $settings['button_entrance_animation'];
        }
        
        if ( $settings['button_hover_animation'] !== 'none' ) {
            $classes[] = 'evas-hover-' . $settings['button_hover_animation'];
        }
        
        return $classes;
    }

    /**
     * Generate button inline styles
     * 
     * @return string
     */
    private function get_button_styles() {
        $settings = $this->get_button_settings();
        
        $styles = ':root {';
        $styles .= '--evas-button-size: ' . $settings['button_size'] . 'px;';
        $styles .= '--evas-button-margin: ' . $settings['button_margin'] . 'px;';
        $styles .= '--evas-button-padding: ' . $settings['button_padding'] . 'px;';
        $styles .= '--evas-button-border-radius: ' . $settings['button_border_radius'] . 'px;';
        $styles .= '--evas-button-color: ' . $settings['button_color'] . ';';
        $styles .= '--evas-button-color-hover: ' . $settings['button_color_hover'] . ';';
        $styles .= '--evas-button-bgcolor: ' . $settings['button_bgcolor'] . ';';
        $styles .= '--evas-button-bgcolor-hover: ' . $settings['button_bgcolor_hover'] . ';';
        $styles .= '--evas-button-entrance-timeout: ' . $settings['button_entrance_timeout'] . 's;';
        $styles .= '}';
        
        // Additional button styles with higher specificity
        $styles .= '#evas-panel-toggle.evas-panel-toggle {';
        $styles .= 'font-size: ' . $settings['button_size'] . 'px !important;';
        $styles .= 'padding: ' . $settings['button_padding'] . 'px !important;';
        $styles .= 'border-radius: ' . $settings['button_border_radius'] . 'px !important;';
        $styles .= 'color: ' . $settings['button_color'] . ' !important;';
        $styles .= 'background-color: ' . $settings['button_bgcolor'] . ' !important;';
        $styles .= 'animation-delay: ' . $settings['button_entrance_timeout'] . 's;';
        $styles .= '}';
        
        $styles .= '#evas-panel-toggle.evas-panel-toggle:hover {';
        $styles .= 'color: ' . $settings['button_color_hover'] . ' !important;';
        $styles .= 'background-color: ' . $settings['button_bgcolor_hover'] . ' !important;';
        $styles .= '}';
        
        // Icon size
        $styles .= '#evas-panel-toggle .evas-panel-toggle-icon {';
        $styles .= 'width: ' . $settings['button_size'] . 'px !important;';
        $styles .= 'height: ' . $settings['button_size'] . 'px !important;';
        $styles .= '}';
        
        return $styles;
    }

    /**
     * Generate popup inline styles
     * 
     * @return string
     */
    private function get_popup_styles() {
        $settings = $this->get_popup_settings();
        
        $styles = ':root {';
        $styles .= '--evas-popup-bg: ' . $settings['popup_background_color'] . ';';
        $styles .= '--evas-popup-key-color: ' . $settings['popup_key_color'] . ';';
        $styles .= '--evas-popup-text-color: ' . $settings['popup_text_color'] . ';';
        $styles .= '--evas-popup-bg-dark: ' . $settings['popup_background_color_dark'] . ';';
        $styles .= '--evas-popup-key-color-dark: ' . $settings['popup_key_color_dark'] . ';';
        $styles .= '--evas-popup-text-color-dark: ' . $settings['popup_text_color_dark'] . ';';
        $styles .= '--evas-popup-border-radius: ' . $settings['popup_border_radius'] . 'px;';
        $styles .= '--evas-popup-overlay-color: ' . $settings['popup_overlay_color'] . ';';
        $styles .= '}';
        
        // Popup styles with high specificity
        $styles .= '#evas-panel.evas-panel {';
        $styles .= 'background-color: ' . $settings['popup_background_color'] . ' !important;';
        $styles .= 'color: ' . $settings['popup_text_color'] . ' !important;';
        $styles .= 'border-radius: ' . $settings['popup_border_radius'] . 'px !important;';
        $styles .= '}';
        
        // Panel content
        $styles .= '#evas-panel .evas-panel-content {';
        $styles .= 'background-color: ' . $settings['popup_background_color'] . ' !important;';
        $styles .= 'border-radius: ' . $settings['popup_border_radius'] . 'px !important;';
        $styles .= '}';
        
        // Panel header
        $styles .= '#evas-panel .evas-panel-header {';
        $styles .= 'background-color: ' . $settings['popup_background_color'] . ' !important;';
        $styles .= 'color: ' . $settings['popup_text_color'] . ' !important;';
        $styles .= '}';
        
        // Panel body
        $styles .= '#evas-panel .evas-panel-body {';
        $styles .= 'background-color: ' . $settings['popup_background_color'] . ' !important;';
        $styles .= 'color: ' . $settings['popup_text_color'] . ' !important;';
        $styles .= '}';
        
        // Action buttons - use key color
        $styles .= '#evas-panel .evas-action-box:hover {';
        $styles .= 'background-color: ' . $settings['popup_key_color'] . ' !important;';
        $styles .= 'color: #fff !important;';
        $styles .= '}';
        
        // Dark theme
        $styles .= '#evas-panel.evas-dark-theme {';
        $styles .= 'background-color: ' . $settings['popup_background_color_dark'] . ' !important;';
        $styles .= 'color: ' . $settings['popup_text_color_dark'] . ' !important;';
        $styles .= '}';
        
        // Overlay
        if ( $settings['popup_overlay'] ) {
            $styles .= '.evas-panel-overlay {';
            $styles .= 'display: block !important;';
            $styles .= 'background-color: ' . $settings['popup_overlay_color'] . ' !important;';
            $styles .= 'opacity: 0.5;';
            $styles .= '}';
        }
        
        return $styles;
    }

    /**
     * Apply customization
     */
    public function apply_customization() {
        if ( ! $this->is_enabled() ) {
            return;
        }
        
        $settings = $this->get_module_settings();
        
        // Add data attributes for JavaScript
        ?>
        <script type="text/javascript">
            if (typeof evasCustomizerData === 'undefined') {
                window.evasCustomizerData = <?php echo json_encode( $settings ); ?>;
            }
        </script>
        <?php
    }
}

