/**
 * EVAS Customizer Frontend JavaScript
 *
 * Applies customization to button and panel on the frontend.
 *
 * @package Everyone_Accessibility_Suite
 * @subpackage Modules/Customizer
 */

(function($) {
    'use strict';

    const EVAS_Customizer = {
        settings: {},
        $button: null,
        $panel: null,
        scrollTimeout: null,

        /**
         * Initialize
         */
        init() {
            if (typeof evasCustomizerData === 'undefined') {
                return;
            }

            this.settings = evasCustomizerData;
            this.$button = $('#evas-panel-toggle');
            this.$panel = $('.evas-panel');

            if (this.$button.length === 0 || this.$panel.length === 0) {
                return;
            }

            this.applyButtonCustomization();
            this.applyPopupCustomization();
            this.handleButtonAnimations();
            this.handleScrollBehavior();
            this.makePanelDraggable();
            this.handlePopupPosition();
            this.handleOverlay();
            this.handleCloseAnywhere();
        },

        /**
         * Apply button customization
         */
        applyButtonCustomization() {
            const s = this.settings;

            // Apply position class
            if (s.button_position) {
                this.$button.addClass('evas-button-position-' + s.button_position.replace('_', '-'));
            }

            // Apply inline styles for immediate effect
            if (s.button_size) {
                this.$button.css('font-size', s.button_size + 'px');
                this.$button.find('.evas-panel-toggle-icon').css({
                    'width': s.button_size + 'px',
                    'height': s.button_size + 'px'
                });
            }

            if (s.button_padding) {
                this.$button.css('padding', s.button_padding + 'px');
            }

            if (s.button_border_radius) {
                this.$button.css('border-radius', s.button_border_radius + 'px');
            }

            if (s.button_color) {
                this.$button.css('color', s.button_color);
            }

            if (s.button_bgcolor) {
                this.$button.css('background-color', s.button_bgcolor);
            }

            // Apply caption if set
            if (s.button_caption) {
                const iconPosition = s.button_icon_position || 'before';
                const captionHtml = '<span class="evas-button-caption">' + s.button_caption + '</span>';
                
                if (iconPosition === 'before' || iconPosition === 'above') {
                    this.$button.append(captionHtml);
                } else {
                    this.$button.prepend(captionHtml);
                }
            }

            // Apply tabindex
            if (typeof s.button_tabindex !== 'undefined') {
                this.$button.attr('tabindex', s.button_tabindex);
            }

            // Show/hide button
            if (s.show_open_button === false) {
                this.$button.hide();
            }

            // Apply hover colors
            this.$button.on('mouseenter', function() {
                if (s.button_color_hover) {
                    $(this).css('color', s.button_color_hover);
                }
                if (s.button_bgcolor_hover) {
                    $(this).css('background-color', s.button_bgcolor_hover);
                }
            }).on('mouseleave', function() {
                if (s.button_color) {
                    $(this).css('color', s.button_color);
                }
                if (s.button_bgcolor) {
                    $(this).css('background-color', s.button_bgcolor);
                }
            });
        },

        /**
         * Apply popup customization
         */
        applyPopupCustomization() {
            const s = this.settings;

            // Apply position class
            if (s.popup_position) {
                this.$panel.addClass('evas-panel-position-' + s.popup_position);
            }

            // Apply inline styles for immediate effect
            if (s.popup_background_color) {
                this.$panel.css('background-color', s.popup_background_color);
                this.$panel.find('.evas-panel-content, .evas-panel-header, .evas-panel-body').css('background-color', s.popup_background_color);
            }

            if (s.popup_text_color) {
                this.$panel.css('color', s.popup_text_color);
                this.$panel.find('.evas-panel-header, .evas-panel-body').css('color', s.popup_text_color);
            }

            if (s.popup_border_radius) {
                this.$panel.css('border-radius', s.popup_border_radius + 'px');
                this.$panel.find('.evas-panel-content').css('border-radius', s.popup_border_radius + 'px');
            }

            // Apply draggable class
            if (s.popup_draggable) {
                this.$panel.addClass('evas-panel-draggable');
            }

            // Apply shadow class
            if (s.popup_shadow) {
                this.$panel.addClass('evas-panel-with-shadow');
            }

            // Hide reset button if needed
            if (s.reset_button === false) {
                this.$panel.find('#evas-reset-all, .evas-btn-reset').hide();
            }

            // Hide "hide forever" button if needed
            if (s.hide_button === false) {
                this.$panel.find('.evas-hide-button').hide();
            }
        },

        /**
         * Handle button animations
         */
        handleButtonAnimations() {
            const s = this.settings;

            // Entrance animation
            if (s.button_entrance_animation && s.button_entrance_animation !== 'none') {
                this.$button.addClass('evas-entrance-' + s.button_entrance_animation);
                
                // Add entrance timeout
                if (s.button_entrance_timeout > 0) {
                    this.$button.css({
                        'opacity': '0',
                        'animation-delay': s.button_entrance_timeout + 's'
                    });
                    
                    setTimeout(() => {
                        this.$button.css('opacity', '1');
                    }, s.button_entrance_timeout * 1000);
                }
            }

            // Hover animation
            if (s.button_hover_animation && s.button_hover_animation !== 'none') {
                this.$button.addClass('evas-hover-' + s.button_hover_animation);
            }
        },

        /**
         * Handle scroll behavior
         */
        handleScrollBehavior() {
            const s = this.settings;
            const desktopScroll = s.button_show_after_scrolling_desktop || 0;
            const mobileScroll = s.button_show_after_scrolling_mobile || 0;

            if (desktopScroll === 0 && mobileScroll === 0) {
                return;
            }

            const isMobile = window.innerWidth <= 768;
            const scrollThreshold = isMobile ? mobileScroll : desktopScroll;

            if (scrollThreshold > 0) {
                // Hide button initially
                this.$button.css('display', 'none');

                // Show/hide on scroll
                $(window).on('scroll', () => {
                    clearTimeout(this.scrollTimeout);
                    this.scrollTimeout = setTimeout(() => {
                        const scrollTop = $(window).scrollTop();
                        
                        if (scrollTop >= scrollThreshold) {
                            this.$button.fadeIn(300);
                        } else {
                            this.$button.fadeOut(300);
                        }
                    }, 100);
                });
            }
        },

        /**
         * Make panel draggable
         */
        makePanelDraggable() {
            if (!this.settings.popup_draggable) {
                return;
            }

            let isDragging = false;
            let startX, startY, startLeft, startTop;

            const $header = this.$panel.find('.evas-panel-header');
            
            if ($header.length === 0) {
                return;
            }

            $header.css('cursor', 'move');

            $header.on('mousedown', (e) => {
                isDragging = true;
                startX = e.clientX;
                startY = e.clientY;
                
                const offset = this.$panel.offset();
                startLeft = offset.left;
                startTop = offset.top;
                
                this.$panel.addClass('evas-dragging');
                e.preventDefault();
            });

            $(document).on('mousemove', (e) => {
                if (!isDragging) return;
                
                const deltaX = e.clientX - startX;
                const deltaY = e.clientY - startY;
                
                this.$panel.css({
                    'left': startLeft + deltaX + 'px',
                    'top': startTop + deltaY + 'px',
                    'right': 'auto',
                    'bottom': 'auto'
                });
            });

            $(document).on('mouseup', () => {
                if (isDragging) {
                    isDragging = false;
                    this.$panel.removeClass('evas-dragging');
                }
            });
        },

        /**
         * Handle popup position
         */
        handlePopupPosition() {
            const position = this.settings.popup_position || 'right';
            
            // Remove existing position classes
            this.$panel.removeClass('evas-panel-position-left evas-panel-position-center evas-panel-position-right');
            
            // Add new position class
            this.$panel.addClass('evas-panel-position-' + position);
        },

        /**
         * Handle overlay
         */
        handleOverlay() {
            if (!this.settings.popup_overlay) {
                return;
            }

            // Create overlay if it doesn't exist
            if ($('.evas-panel-overlay').length === 0) {
                $('body').append('<div class="evas-panel-overlay"></div>');
            }

            const $overlay = $('.evas-panel-overlay');

            // Show overlay when panel opens
            this.$button.on('click', () => {
                setTimeout(() => {
                    if (this.$panel.hasClass('evas-open')) {
                        $overlay.fadeIn(200);
                    }
                }, 100);
            });

            // Hide overlay when panel closes
            this.$panel.find('.evas-panel-close, .evas-reset-button').on('click', () => {
                $overlay.fadeOut(200);
            });

            // Close panel when overlay is clicked
            $overlay.on('click', () => {
                this.$panel.removeClass('evas-open');
                $overlay.fadeOut(200);
            });
        },

        /**
         * Handle close anywhere
         */
        handleCloseAnywhere() {
            if (!this.settings.popup_close_anywhere) {
                return;
            }

            $(document).on('click', (e) => {
                if (this.$panel.hasClass('evas-open')) {
                    const $target = $(e.target);
                    
                    // Check if click is outside panel and button
                    if (!$target.closest('.evas-panel').length && 
                        !$target.closest('#evas-panel-toggle').length) {
                        this.$panel.removeClass('evas-open');
                        
                        // Hide overlay if present
                        $('.evas-panel-overlay').fadeOut(200);
                    }
                }
            });
        }
    };

    // Initialize when DOM is ready
    $(document).ready(function() {
        EVAS_Customizer.init();
    });

})(jQuery);

