<?php
/**
 * Usage Analytics
 *
 * Tracks accessibility feature usage.
 *
 * @package     Everyone_Accessibility_Suite
 * @subpackage  Modules/Accessibility_Extras
 * @version     1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class EVAS_Usage_Analytics
 */
class EVAS_Usage_Analytics {

    private const OPTION_KEY = 'evas_usage_analytics';

    /**
     * Track feature usage
     */
    public static function track( string $feature, string $action = 'activate' ): void {
        $analytics = get_option( self::OPTION_KEY, [] );
        
        $date = current_time( 'Y-m-d' );
        
        if ( ! isset( $analytics[ $date ] ) ) {
            $analytics[ $date ] = [];
        }
        
        if ( ! isset( $analytics[ $date ][ $feature ] ) ) {
            $analytics[ $date ][ $feature ] = [
                'activations' => 0,
                'deactivations' => 0,
            ];
        }
        
        if ( $action === 'activate' ) {
            $analytics[ $date ][ $feature ]['activations']++;
        } else {
            $analytics[ $date ][ $feature ]['deactivations']++;
        }
        
        // Keep only last 90 days
        $cutoff = gmdate( 'Y-m-d', strtotime( '-90 days' ) );
        $analytics = array_filter( $analytics, fn( $k ) => $k >= $cutoff, ARRAY_FILTER_USE_KEY );
        
        update_option( self::OPTION_KEY, $analytics );
    }

    /**
     * Get analytics data
     */
    public static function get_data( int $days = 30 ): array {
        $analytics = get_option( self::OPTION_KEY, [] );
        
        $cutoff = gmdate( 'Y-m-d', strtotime( "-{$days} days" ) );
        
        return array_filter( $analytics, fn( $k ) => $k >= $cutoff, ARRAY_FILTER_USE_KEY );
    }

    /**
     * Get feature statistics
     */
    public static function get_feature_stats( int $days = 30 ): array {
        $data = self::get_data( $days );
        $stats = [];

        foreach ( $data as $date => $features ) {
            foreach ( $features as $feature => $counts ) {
                if ( ! isset( $stats[ $feature ] ) ) {
                    $stats[ $feature ] = [
                        'feature'       => $feature,
                        'activations'   => 0,
                        'deactivations' => 0,
                        'total'         => 0,
                    ];
                }
                
                $stats[ $feature ]['activations'] += $counts['activations'];
                $stats[ $feature ]['deactivations'] += $counts['deactivations'];
                $stats[ $feature ]['total'] += $counts['activations'];
            }
        }

        // Sort by total usage
        uasort( $stats, fn( $a, $b ) => $b['total'] <=> $a['total'] );

        return array_values( $stats );
    }

    /**
     * Get daily statistics
     */
    public static function get_daily_stats( int $days = 30 ): array {
        $data = self::get_data( $days );
        $daily = [];

        foreach ( $data as $date => $features ) {
            $total = 0;
            foreach ( $features as $counts ) {
                $total += $counts['activations'];
            }
            $daily[] = [
                'date'  => $date,
                'count' => $total,
            ];
        }

        return $daily;
    }

    /**
     * Get summary
     */
    public static function get_summary(): array {
        $all_time = self::get_feature_stats( 365 );
        $last_30 = self::get_feature_stats( 30 );
        $last_7 = self::get_feature_stats( 7 );

        $total_all = array_sum( array_column( $all_time, 'total' ) );
        $total_30 = array_sum( array_column( $last_30, 'total' ) );
        $total_7 = array_sum( array_column( $last_7, 'total' ) );

        return [
            'total_all_time'    => $total_all,
            'total_last_30_days' => $total_30,
            'total_last_7_days'  => $total_7,
            'top_features'       => array_slice( $all_time, 0, 5 ),
            'feature_count'      => count( $all_time ),
        ];
    }

    /**
     * Clear analytics
     */
    public static function clear(): void {
        delete_option( self::OPTION_KEY );
    }
}

