<?php
/**
 * Accessibility Statement Generator
 *
 * Generates accessibility statements.
 *
 * @package     Everyone_Accessibility_Suite
 * @subpackage  Modules/Accessibility_Extras
 * @version     1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class EVAS_Statement_Generator
 */
class EVAS_Statement_Generator {

    /**
     * Generate accessibility statement
     */
    public static function generate( array $settings = [] ): string {
        $defaults = [
            'organization_name' => get_bloginfo( 'name' ),
            'website_url'       => home_url(),
            'email'             => get_option( 'admin_email' ),
            'phone'             => '',
            'conformance_level' => 'AA',
            'standards'         => 'WCAG 2.1',
            'last_updated'      => current_time( 'F j, Y' ),
        ];

        $settings = wp_parse_args( $settings, $defaults );

        ob_start();
        ?>
<h2><?php esc_html_e( 'Accessibility Statement', 'everyone-accessibility-suite' ); ?></h2>

<p><?php 
printf( 
    /* translators: %s: organization name */
    esc_html__( '%s is committed to ensuring digital accessibility for people with disabilities. We are continually improving the user experience for everyone and applying the relevant accessibility standards.', 'everyone-accessibility-suite' ),
    esc_html( $settings['organization_name'] )
); 
?></p>

<h3><?php esc_html_e( 'Conformance Status', 'everyone-accessibility-suite' ); ?></h3>

<p><?php 
printf(
    /* translators: 1: WCAG version, 2: conformance level */
    esc_html__( 'We strive to conform to %1$s Level %2$s guidelines.', 'everyone-accessibility-suite' ),
    esc_html( $settings['standards'] ),
    esc_html( $settings['conformance_level'] )
);
?></p>

<h3><?php esc_html_e( 'Accessibility Features', 'everyone-accessibility-suite' ); ?></h3>

<p><?php esc_html_e( 'This website includes the following accessibility features:', 'everyone-accessibility-suite' ); ?></p>

<ul>
    <li><?php esc_html_e( 'Adjustable font sizes and content scaling', 'everyone-accessibility-suite' ); ?></li>
    <li><?php esc_html_e( 'High contrast and color adjustment options', 'everyone-accessibility-suite' ); ?></li>
    <li><?php esc_html_e( 'Text-to-speech functionality', 'everyone-accessibility-suite' ); ?></li>
    <li><?php esc_html_e( 'Keyboard navigation support', 'everyone-accessibility-suite' ); ?></li>
    <li><?php esc_html_e( 'Screen reader compatibility', 'everyone-accessibility-suite' ); ?></li>
    <li><?php esc_html_e( 'Reading assistance tools', 'everyone-accessibility-suite' ); ?></li>
</ul>

<h3><?php esc_html_e( 'Feedback', 'everyone-accessibility-suite' ); ?></h3>

<p><?php esc_html_e( 'We welcome your feedback on the accessibility of this website. Please contact us if you encounter any accessibility barriers:', 'everyone-accessibility-suite' ); ?></p>

<ul>
    <?php if ( ! empty( $settings['email'] ) ) : ?>
    <li><?php esc_html_e( 'Email:', 'everyone-accessibility-suite' ); ?> <a href="mailto:<?php echo esc_attr( $settings['email'] ); ?>"><?php echo esc_html( $settings['email'] ); ?></a></li>
    <?php endif; ?>
    <?php if ( ! empty( $settings['phone'] ) ) : ?>
    <li><?php esc_html_e( 'Phone:', 'everyone-accessibility-suite' ); ?> <?php echo esc_html( $settings['phone'] ); ?></li>
    <?php endif; ?>
</ul>

<p><em><?php 
printf(
    /* translators: %s: date */
    esc_html__( 'This statement was last updated on %s.', 'everyone-accessibility-suite' ),
    esc_html( $settings['last_updated'] )
);
?></em></p>
        <?php
        return ob_get_clean();
    }

    /**
     * Get statement settings
     */
    public static function get_settings(): array {
        return get_option( 'evas_statement_settings', [
            'organization_name' => get_bloginfo( 'name' ),
            'email'             => get_option( 'admin_email' ),
            'phone'             => '',
            'conformance_level' => 'AA',
            'standards'         => 'WCAG 2.1',
        ] );
    }

    /**
     * Save statement settings
     */
    public static function save_settings( array $settings ): bool {
        return update_option( 'evas_statement_settings', $settings );
    }

    /**
     * Register shortcode
     */
    public static function register_shortcode(): void {
        add_shortcode( 'evas-accessibility-statement', function( $atts ) {
            $settings = self::get_settings();
            return self::generate( $settings );
        });
    }
}

// Register shortcode
add_action( 'init', [ 'EVAS_Statement_Generator', 'register_shortcode' ] );

