<?php
/**
 * Accessibility Extras REST Controller
 *
 * @package     Everyone_Accessibility_Suite
 * @subpackage  Modules/Accessibility_Extras
 * @version     1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class EVAS_AE_REST_Controller
 */
class EVAS_AE_REST_Controller extends WP_REST_Controller {

    protected $namespace = 'evas-extras/v1';

    public function register_routes(): void {
        // Analytics
        register_rest_route( $this->namespace, '/analytics/track', [
            'methods'             => WP_REST_Server::CREATABLE,
            'callback'            => [ $this, 'track_usage' ],
            'permission_callback' => '__return_true',
        ] );

        register_rest_route( $this->namespace, '/analytics/stats', [
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => [ $this, 'get_stats' ],
            'permission_callback' => [ $this, 'permissions_check' ],
        ] );

        register_rest_route( $this->namespace, '/analytics/summary', [
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => [ $this, 'get_summary' ],
            'permission_callback' => [ $this, 'permissions_check' ],
        ] );

        // Statement
        register_rest_route( $this->namespace, '/statement/generate', [
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => [ $this, 'generate_statement' ],
            'permission_callback' => [ $this, 'permissions_check' ],
        ] );

        register_rest_route( $this->namespace, '/statement/settings', [
            [
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => [ $this, 'get_statement_settings' ],
                'permission_callback' => [ $this, 'permissions_check' ],
            ],
            [
                'methods'             => WP_REST_Server::EDITABLE,
                'callback'            => [ $this, 'save_statement_settings' ],
                'permission_callback' => [ $this, 'permissions_check' ],
            ],
        ] );
    }

    public function track_usage( WP_REST_Request $request ): WP_REST_Response {
        $params = $request->get_json_params();
        
        $feature = sanitize_key( $params['feature'] ?? '' );
        $action = sanitize_key( $params['action'] ?? 'activate' );

        if ( ! empty( $feature ) ) {
            EVAS_Usage_Analytics::track( $feature, $action );
        }

        return rest_ensure_response( [ 'success' => true ] );
    }

    public function get_stats( WP_REST_Request $request ): WP_REST_Response {
        $days = (int) ( $request->get_param( 'days' ) ?? 30 );

        return rest_ensure_response( [
            'success'       => true,
            'feature_stats' => EVAS_Usage_Analytics::get_feature_stats( $days ),
            'daily_stats'   => EVAS_Usage_Analytics::get_daily_stats( $days ),
        ] );
    }

    public function get_summary( WP_REST_Request $request ): WP_REST_Response {
        return rest_ensure_response( [
            'success' => true,
            'summary' => EVAS_Usage_Analytics::get_summary(),
        ] );
    }

    public function generate_statement( WP_REST_Request $request ): WP_REST_Response {
        $settings = EVAS_Statement_Generator::get_settings();
        $html = EVAS_Statement_Generator::generate( $settings );

        return rest_ensure_response( [
            'success'   => true,
            'statement' => $html,
            'shortcode' => '[evas-accessibility-statement]',
        ] );
    }

    public function get_statement_settings( WP_REST_Request $request ): WP_REST_Response {
        return rest_ensure_response( [
            'success'  => true,
            'settings' => EVAS_Statement_Generator::get_settings(),
        ] );
    }

    public function save_statement_settings( WP_REST_Request $request ): WP_REST_Response {
        $params = $request->get_json_params();
        
        $settings = [
            'organization_name' => sanitize_text_field( $params['organization_name'] ?? '' ),
            'email'             => sanitize_email( $params['email'] ?? '' ),
            'phone'             => sanitize_text_field( $params['phone'] ?? '' ),
            'conformance_level' => sanitize_text_field( $params['conformance_level'] ?? 'AA' ),
            'standards'         => sanitize_text_field( $params['standards'] ?? 'WCAG 2.1' ),
        ];

        EVAS_Statement_Generator::save_settings( $settings );

        return rest_ensure_response( [
            'success' => true,
            'message' => __( 'Settings saved', 'everyone-accessibility-suite' ),
        ] );
    }

    public function permissions_check(): bool {
        return current_user_can( 'manage_options' );
    }
}

