<?php
/**
 * WCAG Checker
 *
 * Checks HTML content for WCAG 2.1 compliance.
 *
 * @package     Everyone_Accessibility_Suite
 * @subpackage  Modules/Accessibility_Analyzer
 * @version     1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class EVAS_WCAG_Checker
 */
class EVAS_WCAG_Checker {

    /**
     * Check HTML content for accessibility issues
     *
     * @param string $html HTML content to check.
     * @return array Issues found.
     */
    public function check( string $html ): array {
        $issues = [];
        $doc = new DOMDocument();
        
        // Suppress errors for malformed HTML
        libxml_use_internal_errors( true );
        $doc->loadHTML( mb_convert_encoding( $html, 'HTML-ENTITIES', 'UTF-8' ), LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD );
        libxml_clear_errors();

        $xpath = new DOMXPath( $doc );

        // Check images for alt text
        $issues = array_merge( $issues, $this->check_images( $xpath ) );

        // Check heading structure
        $issues = array_merge( $issues, $this->check_headings( $xpath ) );

        // Check form labels
        $issues = array_merge( $issues, $this->check_forms( $xpath ) );

        // Check link text
        $issues = array_merge( $issues, $this->check_links( $xpath ) );

        // Check color contrast (basic check)
        $issues = array_merge( $issues, $this->check_contrast( $xpath ) );

        // Check language
        $issues = array_merge( $issues, $this->check_language( $doc ) );

        return $issues;
    }

    /**
     * Check images for alt text
     */
    private function check_images( DOMXPath $xpath ): array {
        $issues = [];
        $images = $xpath->query( '//img' );

        foreach ( $images as $img ) {
            $alt = $img->getAttribute( 'alt' );
            $src = $img->getAttribute( 'src' );

            if ( empty( $alt ) ) {
                $issues[] = [
                    'type'        => 'error',
                    'wcag'        => '1.1.1',
                    'criterion'   => 'Non-text Content',
                    'element'     => 'img',
                    'description' => __( 'Image missing alt text', 'everyone-accessibility-suite' ),
                    'context'     => $src,
                    'suggestion'  => __( 'Add descriptive alt text to the image', 'everyone-accessibility-suite' ),
                ];
            } elseif ( $this->is_decorative_alt( $alt ) ) {
                $issues[] = [
                    'type'        => 'warning',
                    'wcag'        => '1.1.1',
                    'criterion'   => 'Non-text Content',
                    'element'     => 'img',
                    'description' => __( 'Image has generic alt text', 'everyone-accessibility-suite' ),
                    'context'     => "alt=\"{$alt}\"",
                    'suggestion'  => __( 'Replace with meaningful description', 'everyone-accessibility-suite' ),
                ];
            }
        }

        return $issues;
    }

    /**
     * Check heading structure
     */
    private function check_headings( DOMXPath $xpath ): array {
        $issues = [];
        $headings = $xpath->query( '//h1|//h2|//h3|//h4|//h5|//h6' );
        
        $levels = [];
        foreach ( $headings as $h ) {
            $level = (int) substr( $h->nodeName, 1 );
            $levels[] = $level;
        }

        // Check for multiple h1
        $h1_count = count( array_filter( $levels, fn( $l ) => $l === 1 ) );
        if ( $h1_count > 1 ) {
            $issues[] = [
                'type'        => 'warning',
                'wcag'        => '1.3.1',
                'criterion'   => 'Info and Relationships',
                'element'     => 'h1',
                // translators: %d: number of H1 elements found on the page.
                'description' => sprintf( __( 'Multiple h1 elements found (%d)', 'everyone-accessibility-suite' ), $h1_count ),
                'suggestion'  => __( 'Use only one h1 element per page', 'everyone-accessibility-suite' ),
            ];
        }

        // Check for skipped levels
        $prev_level = 0;
        foreach ( $levels as $level ) {
            if ( $prev_level > 0 && $level > $prev_level + 1 ) {
                $issues[] = [
                    'type'        => 'error',
                    'wcag'        => '1.3.1',
                    'criterion'   => 'Info and Relationships',
                    'element'     => "h{$level}",
                    // translators: 1: previous heading level number. 2: current heading level number.
                    'description' => sprintf( __( 'Heading level skipped from h%1$d to h%2$d', 'everyone-accessibility-suite' ), $prev_level, $level ),
                    'suggestion'  => __( 'Use proper heading hierarchy without skipping levels', 'everyone-accessibility-suite' ),
                ];
            }
            $prev_level = $level;
        }

        return $issues;
    }

    /**
     * Check form elements for labels
     */
    private function check_forms( DOMXPath $xpath ): array {
        $issues = [];
        $inputs = $xpath->query( '//input[@type!="hidden" and @type!="submit" and @type!="button" and @type!="reset" and @type!="image"]|//select|//textarea' );

        foreach ( $inputs as $input ) {
            $id = $input->getAttribute( 'id' );
            $aria_label = $input->getAttribute( 'aria-label' );
            $aria_labelledby = $input->getAttribute( 'aria-labelledby' );

            if ( empty( $id ) && empty( $aria_label ) && empty( $aria_labelledby ) ) {
                $issues[] = [
                    'type'        => 'error',
                    'wcag'        => '1.3.1',
                    'criterion'   => 'Info and Relationships',
                    'element'     => $input->nodeName,
                    'description' => __( 'Form input has no associated label', 'everyone-accessibility-suite' ),
                    'suggestion'  => __( 'Add a label element or aria-label attribute', 'everyone-accessibility-suite' ),
                ];
                continue;
            }

            if ( ! empty( $id ) ) {
                $label = $xpath->query( "//label[@for='{$id}']" );
                if ( $label->length === 0 && empty( $aria_label ) && empty( $aria_labelledby ) ) {
                    $issues[] = [
                        'type'        => 'error',
                        'wcag'        => '1.3.1',
                        'criterion'   => 'Info and Relationships',
                        'element'     => $input->nodeName,
                        'description' => __( 'No label found for form input', 'everyone-accessibility-suite' ),
                        'context'     => "id=\"{$id}\"",
                        'suggestion'  => __( 'Add a label with for attribute matching the input id', 'everyone-accessibility-suite' ),
                    ];
                }
            }
        }

        return $issues;
    }

    /**
     * Check links for accessible text
     */
    private function check_links( DOMXPath $xpath ): array {
        $issues = [];
        $links = $xpath->query( '//a[@href]' );

        $generic_texts = [ 'click here', 'read more', 'more', 'learn more', 'here', 'link' ];

        foreach ( $links as $link ) {
            $text = strtolower( trim( $link->textContent ) );
            $aria_label = $link->getAttribute( 'aria-label' );

            if ( empty( $text ) && empty( $aria_label ) ) {
                $issues[] = [
                    'type'        => 'error',
                    'wcag'        => '2.4.4',
                    'criterion'   => 'Link Purpose',
                    'element'     => 'a',
                    'description' => __( 'Link has no accessible text', 'everyone-accessibility-suite' ),
                    'context'     => $link->getAttribute( 'href' ),
                    'suggestion'  => __( 'Add descriptive text or aria-label', 'everyone-accessibility-suite' ),
                ];
            } elseif ( in_array( $text, $generic_texts, true ) && empty( $aria_label ) ) {
                $issues[] = [
                    'type'        => 'warning',
                    'wcag'        => '2.4.4',
                    'criterion'   => 'Link Purpose',
                    'element'     => 'a',
                    'description' => __( 'Link has generic text', 'everyone-accessibility-suite' ),
                    'context'     => $text,
                    'suggestion'  => __( 'Use descriptive link text that explains the destination', 'everyone-accessibility-suite' ),
                ];
            }
        }

        return $issues;
    }

    /**
     * Basic contrast check (checks for explicit low contrast)
     */
    private function check_contrast( DOMXPath $xpath ): array {
        // This is a simplified check - full contrast checking requires computed styles
        return [];
    }

    /**
     * Check for language attribute
     */
    private function check_language( DOMDocument $doc ): array {
        $issues = [];
        $html = $doc->getElementsByTagName( 'html' )->item( 0 );

        if ( $html && empty( $html->getAttribute( 'lang' ) ) ) {
            $issues[] = [
                'type'        => 'error',
                'wcag'        => '3.1.1',
                'criterion'   => 'Language of Page',
                'element'     => 'html',
                'description' => __( 'Page language not specified', 'everyone-accessibility-suite' ),
                'suggestion'  => __( 'Add lang attribute to html element', 'everyone-accessibility-suite' ),
            ];
        }

        return $issues;
    }

    /**
     * Check if alt text is decorative/generic
     */
    private function is_decorative_alt( string $alt ): bool {
        $decorative = [ 'image', 'photo', 'picture', 'graphic', 'icon', 'logo', 'banner', 'img', 'pic' ];
        return in_array( strtolower( trim( $alt ) ), $decorative, true );
    }

    /**
     * Get WCAG criteria info
     */
    public static function get_wcag_criteria(): array {
        return [
            '1.1.1' => [ 'level' => 'A', 'name' => 'Non-text Content' ],
            '1.3.1' => [ 'level' => 'A', 'name' => 'Info and Relationships' ],
            '2.4.4' => [ 'level' => 'A', 'name' => 'Link Purpose (In Context)' ],
            '3.1.1' => [ 'level' => 'A', 'name' => 'Language of Page' ],
            '4.1.1' => [ 'level' => 'A', 'name' => 'Parsing' ],
            '4.1.2' => [ 'level' => 'A', 'name' => 'Name, Role, Value' ],
        ];
    }
}

