<?php
/**
 * Accessibility Analyzer REST Controller
 *
 * @package     Everyone_Accessibility_Suite
 * @subpackage  Modules/Accessibility_Analyzer
 * @version     1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class EVAS_AA_REST_Controller
 */
class EVAS_AA_REST_Controller extends WP_REST_Controller {

    protected $namespace = 'evas-analyzer/v1';

    public function register_routes(): void {
        register_rest_route( $this->namespace, '/scan', [
            'methods'             => WP_REST_Server::CREATABLE,
            'callback'            => [ $this, 'scan_url' ],
            'permission_callback' => [ $this, 'permissions_check' ],
            'args'                => [
                'url' => [ 'required' => true, 'type' => 'string' ],
            ],
        ] );

        register_rest_route( $this->namespace, '/reports', [
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => [ $this, 'get_reports' ],
            'permission_callback' => [ $this, 'permissions_check' ],
        ] );

        register_rest_route( $this->namespace, '/reports/(?P<id>[\w_-]+)', [
            [
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => [ $this, 'get_report' ],
                'permission_callback' => [ $this, 'permissions_check' ],
            ],
            [
                'methods'             => WP_REST_Server::DELETABLE,
                'callback'            => [ $this, 'delete_report' ],
                'permission_callback' => [ $this, 'permissions_check' ],
            ],
        ] );

        register_rest_route( $this->namespace, '/summary', [
            'methods'             => WP_REST_Server::READABLE,
            'callback'            => [ $this, 'get_summary' ],
            'permission_callback' => [ $this, 'permissions_check' ],
        ] );
    }

    public function scan_url( WP_REST_Request $request ) {
        $url = $request->get_param( 'url' );

        // Validate URL
        if ( ! filter_var( $url, FILTER_VALIDATE_URL ) ) {
            return new WP_Error( 'invalid_url', __( 'Invalid URL', 'everyone-accessibility-suite' ), [ 'status' => 400 ] );
        }

        // Fetch the page
        $response = wp_remote_get( $url, [ 'timeout' => 30 ] );

        if ( is_wp_error( $response ) ) {
            return new WP_Error( 'fetch_failed', $response->get_error_message(), [ 'status' => 500 ] );
        }

        $html = wp_remote_retrieve_body( $response );

        if ( empty( $html ) ) {
            return new WP_Error( 'empty_response', __( 'No content received', 'everyone-accessibility-suite' ), [ 'status' => 500 ] );
        }

        // Run accessibility check
        $checker = new EVAS_WCAG_Checker();
        $issues = $checker->check( $html );

        // Create report
        $report = [
            'url'        => $url,
            'issues'     => $issues,
            'error_count' => count( array_filter( $issues, fn( $i ) => $i['type'] === 'error' ) ),
            'warning_count' => count( array_filter( $issues, fn( $i ) => $i['type'] === 'warning' ) ),
            'score'      => $this->calculate_score( $issues ),
        ];

        // Save report
        $report_id = EVAS_Analyzer_Reports::save( $report );

        return rest_ensure_response( [
            'success' => true,
            'report'  => array_merge( $report, [ 'id' => $report_id ] ),
        ] );
    }

    public function get_reports( WP_REST_Request $request ): WP_REST_Response {
        $reports = EVAS_Analyzer_Reports::get_all();

        return rest_ensure_response( [
            'success' => true,
            'reports' => array_values( $reports ),
            'count'   => count( $reports ),
        ] );
    }

    public function get_report( WP_REST_Request $request ) {
        $id = $request->get_param( 'id' );
        $report = EVAS_Analyzer_Reports::get( $id );

        if ( ! $report ) {
            return new WP_Error( 'not_found', __( 'Report not found', 'everyone-accessibility-suite' ), [ 'status' => 404 ] );
        }

        return rest_ensure_response( [
            'success' => true,
            'report'  => $report,
        ] );
    }

    public function delete_report( WP_REST_Request $request ): WP_REST_Response {
        $id = $request->get_param( 'id' );
        EVAS_Analyzer_Reports::delete( $id );

        return rest_ensure_response( [
            'success' => true,
            'message' => __( 'Report deleted', 'everyone-accessibility-suite' ),
        ] );
    }

    public function get_summary( WP_REST_Request $request ): WP_REST_Response {
        return rest_ensure_response( [
            'success' => true,
            'summary' => EVAS_Analyzer_Reports::get_summary(),
            'wcag'    => EVAS_WCAG_Checker::get_wcag_criteria(),
        ] );
    }

    private function calculate_score( array $issues ): int {
        $errors = count( array_filter( $issues, fn( $i ) => $i['type'] === 'error' ) );
        $warnings = count( array_filter( $issues, fn( $i ) => $i['type'] === 'warning' ) );
        
        $penalty = ( $errors * 10 ) + ( $warnings * 3 );
        return max( 0, 100 - $penalty );
    }

    public function permissions_check(): bool {
        return current_user_can( 'manage_options' );
    }
}

