<?php
/**
 * Analyzer Reports Storage
 *
 * @package     Everyone_Accessibility_Suite
 * @subpackage  Modules/Accessibility_Analyzer
 * @version     1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class EVAS_Analyzer_Reports
 */
class EVAS_Analyzer_Reports {

    private const OPTION_KEY = 'evas_analyzer_reports';

    /**
     * Save report
     */
    public static function save( array $report ): string {
        $reports = get_option( self::OPTION_KEY, [] );
        
        $id = uniqid( 'report_' );
        $report['id'] = $id;
        $report['created_at'] = current_time( 'mysql' );
        
        $reports[ $id ] = $report;
        
        // Keep only last 50 reports
        if ( count( $reports ) > 50 ) {
            array_shift( $reports );
        }
        
        update_option( self::OPTION_KEY, $reports );
        
        return $id;
    }

    /**
     * Get report by ID
     */
    public static function get( string $id ): ?array {
        $reports = get_option( self::OPTION_KEY, [] );
        return $reports[ $id ] ?? null;
    }

    /**
     * Get all reports
     */
    public static function get_all(): array {
        return get_option( self::OPTION_KEY, [] );
    }

    /**
     * Delete report
     */
    public static function delete( string $id ): bool {
        $reports = get_option( self::OPTION_KEY, [] );
        
        if ( isset( $reports[ $id ] ) ) {
            unset( $reports[ $id ] );
            update_option( self::OPTION_KEY, $reports );
            return true;
        }
        
        return false;
    }

    /**
     * Clear all reports
     */
    public static function clear(): int {
        $reports = get_option( self::OPTION_KEY, [] );
        $count = count( $reports );
        delete_option( self::OPTION_KEY );
        return $count;
    }

    /**
     * Get reports summary
     */
    public static function get_summary(): array {
        $reports = self::get_all();
        
        $total_errors = 0;
        $total_warnings = 0;
        
        foreach ( $reports as $report ) {
            foreach ( $report['issues'] ?? [] as $issue ) {
                if ( $issue['type'] === 'error' ) {
                    $total_errors++;
                } else {
                    $total_warnings++;
                }
            }
        }
        
        return [
            'total_reports'   => count( $reports ),
            'total_errors'    => $total_errors,
            'total_warnings'  => $total_warnings,
            'last_scan'       => ! empty( $reports ) ? end( $reports )['created_at'] : null,
        ];
    }
}

