<?php
/**
 * Settings Class
 *
 * Manages plugin settings
 *
 * @package Everyone_Accessibility_Suite
 * @since 1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Settings management class
 */
class EVAS_Settings {

	/**
	 * Settings option name
	 *
	 * @var string
	 */
	private $option_name = 'evas_settings';

	/**
	 * Get all settings
	 *
	 * @return array All settings.
	 */
	public function get_all_settings() {
		return get_option( $this->option_name, array() );
	}

	/**
	 * Get setting value
	 *
	 * @param string $key Setting key.
	 * @param mixed  $default Default value.
	 * @return mixed Setting value.
	 */
	public function get_setting( $key, $default = null ) {
		$settings = $this->get_all_settings();
		return $settings[ $key ] ?? $default;
	}

	/**
	 * Get module setting
	 *
	 * @param string $module_id Module ID.
	 * @param string $key Setting key.
	 * @param mixed  $default Default value.
	 * @return mixed Module setting value.
	 */
	public function get_module_setting( $module_id, $key, $default = null ) {
		$modules = $this->get_setting( 'modules', array() );
		$module  = $modules[ $module_id ] ?? array();

		if ( is_array( $module ) ) {
			return $module[ $key ] ?? $default;
		}

		return $default;
	}

	/**
	 * Update setting
	 *
	 * @param string $key Setting key.
	 * @param mixed  $value Setting value.
	 * @return bool True on success, false on failure.
	 */
	public function update_setting( $key, $value ) {
		$settings         = $this->get_all_settings();
		$settings[ $key ] = $value;
		return update_option( $this->option_name, $settings );
	}

	/**
	 * Update all settings
	 *
	 * @param array $settings Settings array.
	 * @return bool True on success, false on failure.
	 */
	public function update_all_settings( $settings ) {
		return update_option( $this->option_name, $settings );
	}

	/**
	 * Update module setting
	 *
	 * @param string $module_id Module ID.
	 * @param string $key Setting key.
	 * @param mixed  $value Setting value.
	 * @return bool True on success, false on failure.
	 */
	public function update_module_setting( $module_id, $key, $value ) {
		$settings = $this->get_all_settings();
		$modules  = $settings['modules'] ?? array();

		if ( ! isset( $modules[ $module_id ] ) ) {
			$modules[ $module_id ] = array();
		}

		$modules[ $module_id ][ $key ] = $value;
		$settings['modules']           = $modules;

		return update_option( $this->option_name, $settings );
	}

	/**
	 * Check if module is enabled
	 *
	 * @param string $module_id Module ID.
	 * @return bool True if enabled, false otherwise.
	 */
	public function is_module_enabled( $module_id ) {
		// Usage Analytics is always enabled
		if ( $module_id === 'usage_analytics' ) {
			return true;
		}
		
		$modules = $this->get_setting( 'modules', array() );

		if ( ! isset( $modules[ $module_id ] ) ) {
			// Default: accessibility_panel is enabled
			if ( $module_id === 'accessibility_panel' ) {
				return true;
			}
			return false;
		}

		// If module setting is a boolean, return it
		if ( is_bool( $modules[ $module_id ] ) ) {
			return $modules[ $module_id ];
		}

		// If module setting is an array, check 'enabled' key
		if ( is_array( $modules[ $module_id ] ) ) {
			return isset( $modules[ $module_id ]['enabled'] ) && $modules[ $module_id ]['enabled'];
		}

		return (bool) $modules[ $module_id ];
	}

	/**
	 * Enable module
	 *
	 * @param string $module_id Module ID.
	 * @return bool True on success, false on failure.
	 */
	public function enable_module( $module_id ) {
		$settings = $this->get_all_settings();
		$modules  = $settings['modules'] ?? array();

		if ( ! isset( $modules[ $module_id ] ) || ! is_array( $modules[ $module_id ] ) ) {
			$modules[ $module_id ] = array();
		}

		$modules[ $module_id ]['enabled'] = true;
		$settings['modules']              = $modules;

		return update_option( $this->option_name, $settings );
	}

	/**
	 * Disable module
	 *
	 * @param string $module_id Module ID.
	 * @return bool True on success, false on failure.
	 */
	public function disable_module( $module_id ) {
		// Usage Analytics cannot be disabled
		if ( $module_id === 'usage_analytics' ) {
			return false;
		}
		
		$settings = $this->get_all_settings();
		$modules  = $settings['modules'] ?? array();

		if ( ! isset( $modules[ $module_id ] ) || ! is_array( $modules[ $module_id ] ) ) {
			$modules[ $module_id ] = array();
		}

		$modules[ $module_id ]['enabled'] = false;
		$settings['modules']              = $modules;

		return update_option( $this->option_name, $settings );
	}
}

