<?php
/**
 * Plugin Loader Class
 *
 * Manages plugin initialization and coordinates all components
 *
 * @package Everyone_Accessibility_Suite
 * @since 1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Main plugin loader class
 */
class EVAS_Plugin_Loader {

	/**
	 * Plugin settings
	 *
	 * @var array
	 */
	private $settings;

	/**
	 * Settings instance
	 *
	 * @var EVAS_Settings
	 */
	private $settings_manager;

	/**
	 * Module Manager instance
	 *
	 * @var EVAS_Module_Manager
	 */
	private $module_manager;

	/**
	 * Available modules configuration
	 *
	 * @var array
	 */
	private $available_modules = array();

	/**
	 * Constructor
	 *
	 * @param array $available_modules Available modules configuration.
	 */
	public function __construct( $available_modules = array() ) {
		$this->settings          = get_option( 'evas_settings', array() );
		$this->available_modules = $available_modules;
		$this->settings_manager  = new EVAS_Settings();
	}

	/**
	 * Run the plugin
	 */
	public function run() {
		// Initialize Module Manager
		$this->module_manager = new EVAS_Module_Manager( $this->settings_manager, $this->available_modules );

		// Load enabled modules
		$this->module_manager->load_modules();

		// Initialize REST API (always, even outside admin)
		$this->init_rest_api();

		// Initialize admin components (only in admin)
		if ( is_admin() ) {
			$this->init_admin();
		}
	}

	/**
	 * Get modules to load (filtered by settings)
	 *
	 * @return array Modules to load.
	 */
	public function get_load_plugin_modules() {
		if ( $this->module_manager ) {
			return $this->module_manager->get_load_plugin_modules();
		}
		return array();
	}

	/**
	 * Initialize REST API
	 */
	private function init_rest_api() {
		// Initialize settings page (REST API)
		if ( class_exists( 'EVAS_Settings_Page' ) ) {
			new EVAS_Settings_Page( $this->settings_manager );
		}

		// Initialize components page (REST API)
		if ( class_exists( 'EVAS_Components_Page' ) ) {
			new EVAS_Components_Page( $this->module_manager );
		}
	}

	/**
	 * Initialize admin components
	 */
	private function init_admin() {
		// Initialize admin app (main React SPA)
		if ( class_exists( 'EVAS_Admin_App' ) ) {
			new EVAS_Admin_App( $this->module_manager, $this->settings_manager );
		}
	}

	/**
	 * Check REST API permissions
	 *
	 * @return bool
	 */
	public function check_permissions() {
		return current_user_can( 'manage_options' );
	}

	/**
	 * Get settings manager
	 *
	 * @return EVAS_Settings
	 */
	public function get_settings_manager() {
		return $this->settings_manager;
	}

	/**
	 * Get module manager
	 *
	 * @return EVAS_Module_Manager
	 */
	public function get_module_manager() {
		return $this->module_manager;
	}
}

