<?php
/**
 * Module Manager Class
 *
 * Manages plugin modules
 *
 * @package Everyone_Accessibility_Suite
 * @since 1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Module manager class
 */
class EVAS_Module_Manager {

	/**
	 * Settings instance
	 *
	 * @var EVAS_Settings
	 */
	private $settings;

	/**
	 * Loaded modules
	 *
	 * @var array
	 */
	private $loaded_modules = array();

	/**
	 * Module instances
	 *
	 * @var array
	 */
	private $module_instances = array();

	/**
	 * Available modules configuration
	 *
	 * @var array
	 */
	private $available_modules = array();

	/**
	 * Constructor
	 *
	 * @param EVAS_Settings $settings Settings instance.
	 * @param array         $available_modules Available modules configuration.
	 */
	public function __construct( $settings, $available_modules = array() ) {
		$this->settings          = $settings;
		$this->available_modules = $available_modules;
	}

	/**
	 * Get modules to load (filtered by settings)
	 *
	 * @return array Modules to load.
	 */
	public function get_load_plugin_modules() {
		$load_modules = $this->available_modules;

		// Get disabled modules from settings
		$disabled_modules = $this->settings->get_setting( 'deactive_preinstall_modules', array() );

		if ( ! empty( $disabled_modules ) ) {
			foreach ( (array) $load_modules as $module_id => $module ) {
				if ( in_array( $module_id, $disabled_modules, true ) ) {
					unset( $load_modules[ $module_id ] );
				}
			}
		}

		// Filter modules before loading
		$load_modules = apply_filters( 'evas_register_modules', $load_modules );

		return $load_modules;
	}

	/**
	 * Check if module is enabled
	 *
	 * @param string $module_id Module ID.
	 * @return bool True if enabled, false otherwise.
	 */
	public function is_module_enabled( $module_id ) {
		return $this->settings->is_module_enabled( $module_id );
	}

	/**
	 * Load module
	 *
	 * @param string $module_id Module ID.
	 * @param array  $module_config Module configuration.
	 * @return bool True on success, false on failure.
	 */
	public function load_module( $module_id, $module_config ) {
		if ( ! isset( $module_config['autoload'] ) || ! isset( $module_config['module_prefix'] ) ) {
			return false;
		}

		$prefix = rtrim( $module_config['module_prefix'], '_' ) . '_';

		// Check if module is already loaded
		if ( isset( $this->loaded_modules[ $module_id ] ) ) {
			return true;
		}

		$autoload_file = EVAS_PLUGIN_DIR . $module_config['autoload'];

		if ( ! file_exists( $autoload_file ) ) {
			return false;
		}

		// Fire hook before loading module
		do_action( 'evas_pre_load_module', $module_id, $module_config );

		// Load module
		require_once $autoload_file;

		// Apply filter to get updated module configuration (including 'class' from module file)
		$modules_after_load = apply_filters( 'evas_register_modules', array( $module_id => $module_config ) );
		$updated_config = $modules_after_load[ $module_id ] ?? $module_config;

		// Mark module as loaded
		$this->loaded_modules[ $module_id ] = array(
			'module_id'     => $module_id,
			'module_prefix' => $prefix,
			'module_dir'    => defined( $prefix . 'DIR' ) ? constant( $prefix . 'DIR' ) : '',
			'module_url'    => defined( $prefix . 'URL' ) ? constant( $prefix . 'URL' ) : '',
		);

		// Instantiate module class if it exists
		if ( isset( $updated_config['class'] ) && class_exists( $updated_config['class'] ) ) {
			$this->module_instances[ $module_id ] = new $updated_config['class']( $this->settings, $this->settings->get_all_settings() );
		}

		// Fire hook after module loaded
		do_action( 'evas_module_loaded', $module_id, $this->loaded_modules[ $module_id ] );

		return true;
	}

	/**
	 * Load all enabled modules
	 *
	 * @return array Loaded modules.
	 */
	public function load_modules() {
		$modules_to_load = $this->get_load_plugin_modules();

		foreach ( $modules_to_load as $module_id => $module_config ) {
			if ( $this->is_module_enabled( $module_id ) ) {
				$this->load_module( $module_id, $module_config );
			}
		}

		return $this->loaded_modules;
	}

	/**
	 * Get active modules
	 *
	 * @return array Active modules.
	 */
	public function get_active_modules() {
		return $this->loaded_modules;
	}

	/**
	 * Get loaded module info
	 *
	 * @param string $module_id Module ID.
	 * @return array|null Module info or null.
	 */
	public function get_module_info( $module_id ) {
		return $this->loaded_modules[ $module_id ] ?? null;
	}

	/**
	 * Get available modules info for admin page
	 *
	 * @return array Modules info with name, description, icon, status.
	 */
	public function get_available_modules_info() {
		$modules_info = array();

		foreach ( $this->available_modules as $module_id => $module_config ) {
			$is_enabled = $this->is_module_enabled( $module_id );
			$is_loaded  = isset( $this->loaded_modules[ $module_id ] );

			// Get module metadata from configuration
			// Fall back to filter for backwards compatibility
			$module_metadata = apply_filters( "evas_module_{$module_id}_metadata", array() );

			// Check if module is locked
			$is_locked = false;
			$coming_soon = false;
			$can_disable = true;

			// Usage Analytics is always available and cannot be disabled
			if ( $module_id === 'usage_analytics' ) {
				$can_disable = false;
			} else {
				// In production (DEBUG = false), all modules except usage_analytics are locked
				if ( ! defined( 'EVAS_DEBUG' ) || ! EVAS_DEBUG ) {
					$is_locked = true;
					$coming_soon = true;
				}
			}

			$modules_info[] = array(
				'id'          => $module_id,
				'name'        => $module_config['name'] ?? $module_metadata['name'] ?? ucfirst( str_replace( '_', ' ', $module_id ) ),
				'description' => $module_config['description'] ?? $module_metadata['description'] ?? '',
				'icon'        => $module_config['icon'] ?? $module_metadata['icon'] ?? '',
				'enabled'     => $is_enabled,
				'loaded'      => $is_loaded,
				'type'        => $module_config['type'] ?? $module_metadata['type'] ?? 'internal',
				'build'       => $module_config['build'] ?? $module_metadata['build'] ?? 'free',
				'locked'      => $is_locked,
				'coming_soon' => $coming_soon,
				'can_disable' => $can_disable,
			);
		}

		return $modules_info;
	}
}

