<?php
/**
 * Module Base Class
 *
 * Abstract base class for all modules
 *
 * @package Everyone_Accessibility_Suite
 * @since 1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Abstract module base class
 */
abstract class EVAS_Module_Base {

	/**
	 * Module ID
	 *
	 * @var string
	 */
	protected $module_id;

	/**
	 * Module name
	 *
	 * @var string
	 */
	protected $module_name;

	/**
	 * Module version
	 *
	 * @var string
	 */
	protected $module_version;

	/**
	 * Plugin settings
	 *
	 * @var array
	 */
	protected $settings;

	/**
	 * Settings manager instance
	 *
	 * @var EVAS_Settings
	 */
	protected $settings_manager;

	/**
	 * Constructor
	 *
	 * @param EVAS_Settings $settings_manager Settings manager instance.
	 * @param array         $settings Plugin settings.
	 */
	public function __construct( $settings_manager, $settings = array() ) {
		$this->settings_manager = $settings_manager;
		$this->settings         = $settings;
		$this->init();
	}

	/**
	 * Initialize module
	 * Can be overridden in child class
	 */
	public function init() {
		$this->register_settings();
		$this->register_hooks();
		$this->register_rest_routes();

		if ( is_admin() ) {
			$this->register_admin_pages();
		}
	}

	/**
	 * Register module settings
	 * Override in child class
	 */
	abstract public function register_settings();

	/**
	 * Get module settings
	 * Override in child class
	 *
	 * @return array Module settings.
	 */
	abstract public function get_module_settings();

	/**
	 * Register WordPress hooks
	 * Override in child class if needed
	 */
	protected function register_hooks() {
		// Override in child class
	}

	/**
	 * Register REST API routes
	 * Override in child class if needed
	 */
	protected function register_rest_routes() {
		// Override in child class
	}

	/**
	 * Register admin pages
	 * Override in child class if needed
	 */
	protected function register_admin_pages() {
		// Override in child class
	}

	/**
	 * Get module ID
	 *
	 * @return string Module ID.
	 */
	public function get_module_id() {
		return $this->module_id;
	}

	/**
	 * Get module name
	 *
	 * @return string Module name.
	 */
	public function get_module_name() {
		return $this->module_name;
	}

	/**
	 * Get module version
	 *
	 * @return string Module version.
	 */
	public function get_module_version() {
		return $this->module_version;
	}

	/**
	 * Check if module is enabled
	 *
	 * @return bool True if enabled, false otherwise.
	 */
	public function is_enabled() {
		// If settings manager is not available, enable module by default
		if ( ! $this->settings_manager ) {
			return true;
		}

		return $this->settings_manager->is_module_enabled( $this->module_id );
	}
}

