<?php
/**
 * Accessibility Panel Core Class
 *
 * Main class for rendering and managing the accessibility panel.
 * Core functionality that is always active (cannot be disabled).
 *
 * @package Everyone_Accessibility_Suite
 * @since 1.0.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class EVAS_Accessibility_Panel
 *
 * Core accessibility panel functionality.
 */
class EVAS_Accessibility_Panel {

	/**
	 * Settings manager instance
	 *
	 * @var EVAS_Settings
	 */
	private $settings;

	/**
	 * Constructor
	 *
	 * @param EVAS_Settings $settings Settings manager instance.
	 */
	public function __construct( $settings ) {
		$this->settings = $settings;
	}

	/**
	 * Initialize the panel
	 *
	 * @return void
	 */
	public function init() {
		// Enqueue frontend assets
		add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_frontend_assets' ] );

		// Render accessibility panel on frontend
		add_action( 'wp_footer', [ $this, 'render_panel' ], 1 );
	}

	/**
	 * Enqueue frontend assets
	 *
	 * @return void
	 */
	public function enqueue_frontend_assets() {
		// Check if panel is enabled
		if ( ! $this->is_enabled() ) {
			return;
		}

		// Main JavaScript
		wp_enqueue_script(
			'evas-panel-features',
			EVAS_PLUGIN_URL . 'assets/accessibility/js/features.js',
			[ 'jquery' ],
			EVAS_VERSION,
			true
		);

		// Main styles
		wp_enqueue_style(
			'evas-panel-features',
			EVAS_PLUGIN_URL . 'assets/accessibility/css/features.css',
			[],
			EVAS_VERSION
		);

		// Panel styles
		wp_enqueue_style(
			'evas-panel-styles',
			EVAS_PLUGIN_URL . 'assets/accessibility/css/panel.css',
			[],
			EVAS_VERSION
		);

		// Dyslexia font (if enabled)
		$all_settings = $this->settings->get_all_settings();
		if ( isset( $all_settings['dyslexia_font'] ) && $all_settings['dyslexia_font'] === 'on' ) {
			wp_enqueue_style(
				'evas-opendyslexic',
				EVAS_PLUGIN_URL . 'assets/accessibility/fonts/opendyslexic.css',
				[],
				EVAS_VERSION
			);
		}

		// Localize script with settings
		wp_localize_script(
			'evas-panel-features',
			'evasAF',
			[
				'settings'  => $this->get_frontend_settings(),
				'features'  => EVAS_Accessibility_Config::get_all_features(),
				'profiles'  => EVAS_Accessibility_Config::get_profiles(),
				'nonce'     => wp_create_nonce( 'evas_panel_nonce' ),
				'i18n'      => $this->get_i18n_strings(),
			]
		);
	}

	/**
	 * Render accessibility panel on frontend
	 *
	 * @return void
	 */
	public function render_panel() {
		// Check if panel is enabled
		if ( ! $this->is_enabled() ) {
			return;
		}

		// Get panel position
		$all_settings   = $this->settings->get_all_settings();
		$position       = $all_settings['panel_position'] ?? 'bottom-right';
		$position_class = 'evas-position-' . $position;

		// Allow modules to inject custom styles before panel render
		do_action( 'evas_before_panel_render' );
		
		// Apply filters for button classes
		$button_classes = apply_filters( 'evas_button_classes', [ 'evas-panel-toggle' ] );
		$button_class_string = implode( ' ', $button_classes );

		?>
		<div class="evas-toggle-wrapper <?php echo esc_attr( $position_class ); ?>">
			<button type="button" 
					id="evas-panel-toggle" 
					class="<?php echo esc_attr( $button_class_string ); ?>"
					aria-expanded="false"
					aria-label="<?php esc_attr_e( 'Accessibility Options', 'everyone-accessibility-suite' ); ?>">
				<svg class="evas-panel-toggle-icon" xmlns="http://www.w3.org/2000/svg" width="28" height="28" viewBox="0 0 293.05 349.63">
					<path d="M95.37,51.29a51.23,51.23,0,1,1,51.29,51.16h-.07A51.19,51.19,0,0,1,95.37,51.29ZM293,134.59A25.61,25.61,0,0,0,265.49,111h-.13l-89.64,8c-3.06.28-6.13.42-9.19.42H126.65q-4.59,0-9.16-.41L27.7,111a25.58,25.58,0,0,0-4.23,51l.22,0,72.45,6.56a8.55,8.55,0,0,1,7.77,8.48v19.62a33.82,33.82,0,0,1-2.36,12.45L60.48,313.66a25.61,25.61,0,1,0,46.85,20.71h0l39.14-95.61L186,334.63A25.61,25.61,0,0,0,232.86,314L191.63,209.14a34.14,34.14,0,0,1-2.35-12.44V177.09a8.55,8.55,0,0,1,7.77-8.49l72.33-6.55A25.61,25.61,0,0,0,293,134.59Z"/>
				</svg>
			</button>
		</div>
		
		<?php
		// Apply filters for panel classes
		$panel_classes = apply_filters( 'evas_panel_classes', [ 'evas-panel', $position_class ] );
		$panel_class_string = implode( ' ', $panel_classes );
		?>
		<div id="evas-panel" class="<?php echo esc_attr( $panel_class_string ); ?>">
			<div class="evas-panel-content" aria-hidden="true">
				<div class="evas-panel-header">
					<h2><?php esc_html_e( 'Accessibility', 'everyone-accessibility-suite' ); ?></h2>
					<button type="button" 
							class="evas-panel-close"
							aria-label="<?php esc_attr_e( 'Close', 'everyone-accessibility-suite' ); ?>">
					</button>
				</div>
				
				<div class="evas-panel-body">
					<?php
					$controls = new EVAS_Accessibility_Controls( $all_settings );
					$controls->render_all_controls();
					?>
				</div>

				<div class="evas-panel-footer">
					<button type="button" class="evas-btn evas-btn-reset" id="evas-reset-all">
						<?php esc_html_e( 'Reset All', 'everyone-accessibility-suite' ); ?>
					</button>
				</div>
			</div>
		</div>

		<script>
			(function() {
				const toggle = document.getElementById('evas-panel-toggle');
				const panel = document.getElementById('evas-panel');
				if (!toggle || !panel) return;
				
				// Move wrapper and panel to beginning of <html> to avoid body filter issues
				const wrapper = toggle.parentElement;
				const body = document.body;
				if (wrapper && wrapper.classList.contains('evas-toggle-wrapper')) {
					document.documentElement.insertBefore(wrapper, body);
				}
				document.documentElement.insertBefore(panel, body);
				
				// Also move admin bar to <html> to prevent filter issues
				const adminBar = document.getElementById('wpadminbar');
				if (adminBar) {
					document.documentElement.insertBefore(adminBar, body);
				}
				
				const close = panel.querySelector('.evas-panel-close');
				const content = panel.querySelector('.evas-panel-content');
				const resetBtn = document.getElementById('evas-reset-all');

				// Toggle panel
				toggle.addEventListener('click', function(e) {
					e.stopPropagation();
					const isOpen = panel.classList.toggle('evas-open');
					toggle.setAttribute('aria-expanded', isOpen);
					content.setAttribute('aria-hidden', !isOpen);
					
					// Adjust panel position to stay within viewport
					if (isOpen) {
						adjustPanelPosition();
						setTimeout(function() {
							document.dispatchEvent(new CustomEvent('evas-panel-opened'));
						}, 100);
					}
				});
				
				// Adjust panel position based on viewport boundaries
				function adjustPanelPosition() {
					const rect = content.getBoundingClientRect();
					const buttonRect = toggle.getBoundingClientRect();
					const viewportWidth = window.innerWidth;
					const viewportHeight = window.innerHeight;
					const positionClass = panel.className.match(/evas-position-[\w-]+/)?.[0];
					
					// Don't adjust if position is top-left, top-right, bottom-left, or bottom-right
					// These positions are already correctly positioned by CSS
					if (positionClass && (
						positionClass === 'evas-position-top-left' || 
						positionClass === 'evas-position-top-right' || 
						positionClass === 'evas-position-bottom-left' || 
						positionClass === 'evas-position-bottom-right'
					)) {
						return;
					}
					
				// For right-center position, align panel middle with button and check boundaries
				if (positionClass === 'evas-position-right-center') {
					// Calculate button vertical center
					const buttonCenter = buttonRect.top + (buttonRect.height / 2);
					const panelHeight = rect.height;
					const panelWidth = rect.width;
					
					// Calculate ideal top position to center panel with button
					let idealTop = buttonCenter - (panelHeight / 2);
					
					// Check if panel would go off screen at top
					if (idealTop < 20) {
						idealTop = 20;
					}
					
					// Check if panel would go off screen at bottom
					if (idealTop + panelHeight > viewportHeight - 20) {
						idealTop = viewportHeight - panelHeight - 20;
					}
					
					// Calculate right position to prevent overflow
					// Panel should be to the left of the button
					let rightPos = viewportWidth - buttonRect.left + 20; // 20px gap from button
					
					// Check if panel would go off screen on the left
					const leftEdge = viewportWidth - rightPos - panelWidth;
					if (leftEdge < 20) {
						// Adjust to keep panel on screen
						rightPos = viewportWidth - panelWidth - 20;
					}
					
					// Apply the calculated position
					content.style.right = rightPos + 'px';
					content.style.top = idealTop + 'px';
					content.style.transform = 'none';
					return;
				}
					
					// For left-center position, align panel middle with button and check boundaries
					if (positionClass === 'evas-position-left-center') {
						// Calculate button vertical center
						const buttonCenter = buttonRect.top + (buttonRect.height / 2);
						const panelHeight = rect.height;
						const panelWidth = rect.width;
						
						// Calculate ideal top position to center panel with button
						let idealTop = buttonCenter - (panelHeight / 2);
						
						// Check if panel would go off screen at top
						if (idealTop < 20) {
							idealTop = 20;
						}
						
						// Check if panel would go off screen at bottom
						if (idealTop + panelHeight > viewportHeight - 20) {
							idealTop = viewportHeight - panelHeight - 20;
						}
						
						// Calculate left position to prevent overflow
						// Panel should be to the right of the button
						let leftPos = buttonRect.right + 20; // 20px gap from button
						
						// Check if panel would go off screen on the right
						if (leftPos + panelWidth > viewportWidth - 20) {
							// Adjust to keep panel on screen
							leftPos = viewportWidth - panelWidth - 20;
						}
						
						// Apply the calculated position
						content.style.left = leftPos + 'px';
						content.style.top = idealTop + 'px';
						content.style.transform = 'none';
						return;
					}
				}

				// Close button
				if (close) {
					close.addEventListener('click', function() {
						panel.classList.remove('evas-open');
						toggle.setAttribute('aria-expanded', 'false');
						content.setAttribute('aria-hidden', 'true');
					});
				}

				// Reset button
				if (resetBtn) {
					resetBtn.addEventListener('click', function() {
						if (confirm('<?php echo esc_js( __( 'Are you sure you want to reset all accessibility settings?', 'everyone-accessibility-suite' ) ); ?>')) {
							for (let key in localStorage) {
								if (key.startsWith('evas_')) {
									localStorage.removeItem(key);
								}
							}
							window.location.reload();
						}
					});
				}

				// Close on escape
				document.addEventListener('keydown', function(e) {
					if (e.key === 'Escape' && panel.classList.contains('evas-open')) {
						panel.classList.remove('evas-open');
						toggle.setAttribute('aria-expanded', 'false');
						content.setAttribute('aria-hidden', 'true');
					}
				});

				// Close on click outside (using mousedown to avoid conflicts with jQuery delegated events)
				document.addEventListener('mousedown', function(e) {
					if (panel.classList.contains('evas-open') && 
						!panel.contains(e.target) && 
						!toggle.contains(e.target)) {
						panel.classList.remove('evas-open');
						toggle.setAttribute('aria-expanded', 'false');
						content.setAttribute('aria-hidden', 'true');
					}
				});
			})();
		</script>
		<?php
	}

	/**
	 * Check if panel is enabled
	 *
	 * @return bool
	 */
	private function is_enabled() {
		$all_settings = $this->settings->get_all_settings();

		// Support both boolean true and string 'on'
		$is_enabled = isset( $all_settings['enabled'] ) &&
		              ( $all_settings['enabled'] === true ||
		                $all_settings['enabled'] === 'on' ||
		                $all_settings['enabled'] === '1' ||
		                $all_settings['enabled'] === 1 );

		return $is_enabled;
	}

	/**
	 * Get settings for frontend use
	 *
	 * @return array
	 */
	private function get_frontend_settings() {
		$settings = $this->settings->get_all_settings();
		$frontend = [];

		foreach ( $settings as $key => $value ) {
			// Include all accessibility settings
			if ( $value === 'on' || $value === true ) {
				$frontend[ $key ] = $value;
			}
		}

		return $frontend;
	}

	/**
	 * Get internationalization strings for JavaScript
	 *
	 * @return array
	 */
	private function get_i18n_strings() {
		return [
			'default'         => __( 'Default', 'everyone-accessibility-suite' ),
			'activated'       => __( 'Activated', 'everyone-accessibility-suite' ),
			'deactivated'     => __( 'Deactivated', 'everyone-accessibility-suite' ),
			'reset_confirm'   => __( 'Are you sure you want to reset all accessibility settings?', 'everyone-accessibility-suite' ),
			'settings_saved'  => __( 'Settings saved', 'everyone-accessibility-suite' ),
			'settings_reset'  => __( 'All settings have been reset', 'everyone-accessibility-suite' ),
		];
	}
}

