<?php
/**
 * Accessibility Controls Core Class
 *
 * Renders control elements for the accessibility panel.
 * Based on Readabler Controls.php functionality.
 *
 * @package Everyone_Accessibility_Suite
 * @since 1.0.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class EVAS_Accessibility_Controls
 *
 * Handles rendering of all accessibility panel controls.
 */
class EVAS_Accessibility_Controls {

    /**
     * Module settings
     *
     * @var array
     */
    private array $settings;

    /**
     * Constructor
     *
     * @param array $settings Module settings.
     */
    public function __construct( array $settings = [] ) {
        $this->settings = $settings;
    }

    /**
     * Render input spinner control (for font-sizing, line-height, etc.)
     *
     * @param array $feature Feature configuration.
     * @return void
     */
    public function render_spinner( array $feature ): void {
        $id       = $feature['id'];
        $dash_id  = str_replace( '_', '-', $id );
        $step     = $feature['step'] ?? 5;
        $min      = $feature['min'] ?? 0;
        $max      = $feature['max'] ?? 100;
        $default  = $feature['default'] ?? 0;
        $unit     = $feature['unit'] ?? '%';
        $name     = $feature['name'];
        
        // Check if feature is enabled in admin settings
        $is_enabled = $this->is_feature_enabled( $id );
        if ( ! $is_enabled ) {
            return;
        }
        ?>
        <div id="evas-action-<?php echo esc_attr( $dash_id ); ?>" 
             class="evas-action-box evas-spinner-box"
             data-feature="<?php echo esc_attr( $id ); ?>"
             data-step="<?php echo esc_attr( $step ); ?>"
             data-min="<?php echo esc_attr( $min ); ?>"
             data-max="<?php echo esc_attr( $max ); ?>"
             data-default="<?php echo esc_attr( $default ); ?>"
             data-unit="<?php echo esc_attr( $unit ); ?>">
            <div class="evas-action-box-content">
                <span class="evas-title"><?php echo esc_html( $name ); ?></span>
            </div>
            <div class="evas-input-spinner-box">
                <div class="evas-control">
                    <button type="button" 
                            class="evas-minus" 
                            tabindex="0"
                            aria-label="<?php 
                                /* translators: %s: feature name */
                                echo esc_attr( sprintf( __( 'Decrease %s', 'everyone-accessibility-suite' ), $name ) ); 
                            ?>">
                        <span class="evas-minus-icon">−</span>
                    </button>
                    <div class="evas-value" 
                         data-value="<?php echo esc_attr( $default ); ?>" 
                         aria-live="polite">
                        <?php echo $default === 0 ? esc_html__( 'Default', 'everyone-accessibility-suite' ) : esc_html( $default . $unit ); ?>
                    </div>
                    <button type="button" 
                            class="evas-plus" 
                            tabindex="0"
                            aria-label="<?php 
                                /* translators: %s: feature name */
                                echo esc_attr( sprintf( __( 'Increase %s', 'everyone-accessibility-suite' ), $name ) ); 
                            ?>">
                        <span class="evas-plus-icon">+</span>
                    </button>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Render toggle button control (for all on/off features)
     *
     * @param array $feature Feature configuration.
     * @return void
     */
    public function render_toggle( array $feature ): void {
        $id       = $feature['id'];
        $dash_id  = str_replace( '_', '-', $id );
        $name     = $feature['name'];
        $class    = $feature['class'] ?? '';
        $group    = $feature['group'] ?? '';
        
        // Check if feature is enabled in admin settings
        $is_enabled = $this->is_feature_enabled( $id );
        if ( ! $is_enabled ) {
            return;
        }

        $classes = [ 'evas-action-box', 'evas-toggle-box' ];
        ?>
        <div id="evas-action-<?php echo esc_attr( $dash_id ); ?>"
             class="<?php echo esc_attr( implode( ' ', $classes ) ); ?>"
             data-feature="<?php echo esc_attr( $id ); ?>"
             data-class="<?php echo esc_attr( $class ); ?>"
             data-group="<?php echo esc_attr( $group ); ?>"
             tabindex="0"
             role="button"
             aria-pressed="false"
             aria-label="<?php echo esc_attr( $name ); ?>">
            <span class="evas-action-icon"></span>
            <span class="evas-action-label"><?php echo esc_html( $name ); ?></span>
        </div>
        <?php
    }

    /**
     * Render color palette control
     *
     * @param array $feature Feature configuration.
     * @return void
     */
    public function render_palette( array $feature ): void {
        $id       = $feature['id'];
        $dash_id  = str_replace( '_', '-', $id );
        $title    = $feature['name'];

        // Check if feature is enabled in admin settings
        $is_enabled = $this->is_feature_enabled( $id );
        if ( ! $is_enabled ) {
            return;
        }

        // 15 colors palette
        $colors = [
            [ 'name' => __( 'Maroon', 'everyone-accessibility-suite' ), 'value' => 'maroon' ],
            [ 'name' => __( 'Red', 'everyone-accessibility-suite' ), 'value' => 'red' ],
            [ 'name' => __( 'Orange', 'everyone-accessibility-suite' ), 'value' => 'orange' ],
            [ 'name' => __( 'Yellow', 'everyone-accessibility-suite' ), 'value' => 'yellow' ],
            [ 'name' => __( 'Olive', 'everyone-accessibility-suite' ), 'value' => 'olive' ],
            [ 'name' => __( 'Green', 'everyone-accessibility-suite' ), 'value' => 'green' ],
            [ 'name' => __( 'Purple', 'everyone-accessibility-suite' ), 'value' => 'purple' ],
            [ 'name' => __( 'Fuchsia', 'everyone-accessibility-suite' ), 'value' => 'fuchsia' ],
            [ 'name' => __( 'Lime', 'everyone-accessibility-suite' ), 'value' => 'lime' ],
            [ 'name' => __( 'Teal', 'everyone-accessibility-suite' ), 'value' => 'teal' ],
            [ 'name' => __( 'Aqua', 'everyone-accessibility-suite' ), 'value' => 'aqua' ],
            [ 'name' => __( 'Blue', 'everyone-accessibility-suite' ), 'value' => 'blue' ],
            [ 'name' => __( 'Navy', 'everyone-accessibility-suite' ), 'value' => 'navy' ],
            [ 'name' => __( 'Black', 'everyone-accessibility-suite' ), 'value' => 'black' ],
            [ 'name' => __( 'White', 'everyone-accessibility-suite' ), 'value' => 'white' ],
        ];

        ?>
        <div id="evas-action-<?php echo esc_attr( $dash_id ); ?>" 
             class="evas-action-box evas-palette-box" 
             data-feature="<?php echo esc_attr( $id ); ?>"
             role="group"
             aria-label="<?php echo esc_attr( $title ); ?>">
            <div class="evas-action-box-content">
                <span class="evas-title"><?php echo esc_html( $title ); ?></span>
                <div class="evas-color-box">
                    <?php foreach ( $colors as $color ) : ?>
                        <span data-color="<?php echo esc_attr( $color['value'] ); ?>"
                              class="evas-color <?php echo ( 'white' === $color['value'] ) ? 'evas-white' : ''; ?>"
                              role="button"
                              tabindex="0"
                              <?php // translators: %s: color name. ?>
                              aria-label="<?php echo esc_attr( sprintf( __( 'Change color to %s', 'everyone-accessibility-suite' ), $color['name'] ) ); ?>"
                              style="background-color: <?php echo esc_attr( $color['value'] ); ?> !important;">
                        </span>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Render profile button (Epilepsy, Visually Impaired, etc.)
     *
     * @param array $profile Profile configuration.
     * @return void
     */
    public function render_profile( array $profile ): void {
        $id          = $profile['id'];
        $dash_id     = str_replace( '_', '-', $id );
        $name        = $profile['name'];
        $short       = $profile['short'] ?? '';
        $description = $profile['description'];
        $features    = $profile['features'] ?? [];
        
        // Check if profile is enabled in admin settings
        $is_enabled = $this->is_feature_enabled( 'profile_' . $id );
        if ( ! $is_enabled ) {
            return;
        }
        ?>
        <div id="evas-profile-<?php echo esc_attr( $dash_id ); ?>" 
             class="evas-profile-box"
             data-profile="<?php echo esc_attr( $id ); ?>"
             data-features="<?php echo esc_attr( wp_json_encode( $features ) ); ?>"
             tabindex="0"
             role="button"
             aria-pressed="false">
            <div class="evas-profile-row">
                <div class="evas-switch-box">
                    <label class="evas-switch" tabindex="-1">
                        <input type="checkbox" name="evas-profile-<?php echo esc_attr( $dash_id ); ?>" value="on" tabindex="-1">
                        <span class="evas-slider"></span>
                    </label>
                </div>
                <div class="evas-profile-content">
                    <h3 class="evas-profile-title"><?php echo esc_html( $name ); ?></h3>
                    <?php if ( $short ) : ?>
                        <p class="evas-profile-short"><?php echo esc_html( $short ); ?></p>
                    <?php endif; ?>
                </div>
            </div>
            <div class="evas-profile-description"><?php echo esc_html( $description ); ?></div>
        </div>
        <?php
    }

    /**
     * Render all controls for the panel
     *
     * @return void
     */
    public function render_all_controls(): void {
        $groups = EVAS_Accessibility_Config::get_feature_groups();
        ?>
        <div class="evas-af-controls-wrapper">
            
            <!-- Accessibility Profiles -->
            <div class="evas-section evas-profiles-section">
                <h2 class="evas-profiles-section-title">
                    <?php esc_html_e( 'Accessibility modes', 'everyone-accessibility-suite' ); ?>
                </h2>
                <div class="evas-section-content evas-profiles-grid">
                    <?php
                    foreach ( EVAS_Accessibility_Config::get_profiles() as $profile ) {
                        $this->render_profile( $profile );
                    }
                    ?>
                </div>
            </div>

            <!-- Content & Typography -->
            <div class="evas-section evas-typography-section">
                <h2 class="evas-section-title">
                    <span class="evas-section-icon"></span>
                    <?php esc_html_e( 'Content & Typography', 'everyone-accessibility-suite' ); ?>
                </h2>
                <div class="evas-section-content evas-actions-grid">
                    <?php
                    foreach ( EVAS_Accessibility_Config::get_typography_features() as $feature ) {
                        if ( $feature['type'] === 'spinner' ) {
                            $this->render_spinner( $feature );
                        } else {
                            $this->render_toggle( $feature );
                        }
                    }
                    ?>
                </div>
            </div>

            <!-- Visual Adjustments -->
            <div class="evas-section evas-visual-section">
                <h2 class="evas-section-title">
                    <span class="evas-section-icon"></span>
                    <?php esc_html_e( 'Visual Adjustments', 'everyone-accessibility-suite' ); ?>
                </h2>
                <div class="evas-section-content evas-actions-grid">
                    <?php
                    foreach ( EVAS_Accessibility_Config::get_visual_features() as $feature ) {
                        $this->render_toggle( $feature );
                    }
                    ?>
                </div>
            </div>

            <!-- Color Adjustments -->
            <div class="evas-section evas-colors-section">
                <h2 class="evas-section-title">
                    <span class="evas-section-icon"></span>
                    <?php esc_html_e( 'Color Adjustments', 'everyone-accessibility-suite' ); ?>
                </h2>
                <div class="evas-section-content evas-actions-grid">
                    <?php
                    foreach ( EVAS_Accessibility_Config::get_color_features() as $feature ) {
                        if ( $feature['type'] === 'palette' ) {
                            $this->render_palette( $feature );
                        } else {
                            $this->render_toggle( $feature );
                        }
                    }
                    ?>
                </div>
            </div>

            <!-- Media Control -->
            <div class="evas-section evas-media-section">
                <h2 class="evas-section-title">
                    <span class="evas-section-icon"></span>
                    <?php esc_html_e( 'Media Control', 'everyone-accessibility-suite' ); ?>
                </h2>
                <div class="evas-section-content evas-actions-grid">
                    <?php
                    foreach ( EVAS_Accessibility_Config::get_media_features() as $feature ) {
                        $this->render_toggle( $feature );
                    }
                    ?>
                </div>
            </div>

            <!-- Helpful Tools -->
            <div class="evas-section evas-tools-section">
                <h2 class="evas-section-title">
                    <span class="evas-section-icon"></span>
                    <?php esc_html_e( 'Helpful Tools', 'everyone-accessibility-suite' ); ?>
                </h2>
                <div class="evas-section-content evas-actions-grid">
                    <?php
                    foreach ( EVAS_Accessibility_Config::get_tool_features() as $feature ) {
                        $this->render_toggle( $feature );
                    }
                    
                    // Allow modules to add their own controls in the same grid
                    do_action( 'evas_panel_controls' );
                    ?>
                </div>
            </div>

        </div>
        <?php
    }

    /**
     * Check if a feature is enabled in admin settings
     *
     * @param string $feature_id Feature identifier.
     * @return bool
     */
    private function is_feature_enabled( string $feature_id ): bool {
        // By default, all features are enabled
        if ( empty( $this->settings ) ) {
            return true;
        }

        return $this->settings[ $feature_id ]['enabled'] ?? true;
    }

    /**
     * Render a single section
     *
     * @param string $section_id   Section identifier.
     * @param string $section_name Section display name.
     * @param array  $features     Array of features to render.
     * @return void
     */
    public function render_section( string $section_id, string $section_name, array $features ): void {
        ?>
        <div class="evas-section evas-<?php echo esc_attr( $section_id ); ?>-section">
            <h2 class="evas-section-title">
                <span class="evas-section-icon"></span>
                <?php echo esc_html( $section_name ); ?>
            </h2>
            <div class="evas-section-content evas-controls-grid">
                <?php
                foreach ( $features as $feature ) {
                    if ( isset( $feature['type'] ) ) {
                        if ( $feature['type'] === 'spinner' ) {
                            $this->render_spinner( $feature );
                        } else {
                            $this->render_toggle( $feature );
                        }
                    } elseif ( isset( $feature['features'] ) ) {
                        // It's a profile
                        $this->render_profile( $feature );
                    }
                }
                ?>
            </div>
        </div>
        <?php
    }
}

