<?php
/**
 * Accessibility Configuration Core Class
 *
 * Contains all accessibility modes, profiles, and feature definitions.
 * Based on Readabler Config.php functionality.
 *
 * @package Everyone_Accessibility_Suite
 * @since 1.0.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class EVAS_Accessibility_Config
 *
 * Defines all accessibility features, profiles, and their configurations.
 */
class EVAS_Accessibility_Config {

    /**
     * Get all accessibility profiles
     *
     * @return array
     */
    public static function get_profiles(): array {
        return [
            'epilepsy' => [
                'id'          => 'epilepsy',
                'name'        => __( 'Epilepsy Safe Mode', 'everyone-accessibility-suite' ),
                'short'       => __( 'Dampens color and removes blinks', 'everyone-accessibility-suite' ),
                'description' => __( 'This mode enables people with epilepsy to use the website safely by eliminating the risk of seizures that result from flashing or blinking animations and risky color combinations.', 'everyone-accessibility-suite' ),
                'icon'        => 'epilepsy.svg',
                'features'    => [ 'low_saturation', 'stop_animations' ],
            ],
            'visually_impaired' => [
                'id'          => 'visually_impaired',
                'name'        => __( 'Visually Impaired Mode', 'everyone-accessibility-suite' ),
                'short'       => __( "Improves website's visuals", 'everyone-accessibility-suite' ),
                'description' => __( 'This mode adjusts the website for the convenience of users with visual impairments such as Degrading Eyesight, Tunnel Vision, Cataract, Glaucoma, and others.', 'everyone-accessibility-suite' ),
                'icon'        => 'visually-impaired.svg',
                'features'    => [ 'readable_font', 'high_saturation' ],
            ],
            'cognitive_disability' => [
                'id'          => 'cognitive_disability',
                'name'        => __( 'Cognitive Disability Mode', 'everyone-accessibility-suite' ),
                'short'       => __( 'Helps to focus on specific content', 'everyone-accessibility-suite' ),
                'description' => __( 'This mode provides different assistive options to help users with cognitive impairments such as Dyslexia, Autism, CVA, and others, to focus on the essential elements of the website more easily.', 'everyone-accessibility-suite' ),
                'icon'        => 'cognitive.svg',
                'features'    => [ 'highlight_titles', 'highlight_links', 'stop_animations' ],
            ],
            'adhd_friendly' => [
                'id'          => 'adhd_friendly',
                'name'        => __( 'ADHD Friendly Mode', 'everyone-accessibility-suite' ),
                'short'       => __( 'Reduces distractions and improve focus', 'everyone-accessibility-suite' ),
                'description' => __( 'This mode helps users with ADHD and Neurodevelopmental disorders to read, browse, and focus on the main website elements more easily while significantly reducing distractions.', 'everyone-accessibility-suite' ),
                'icon'        => 'adhd.svg',
                'features'    => [ 'high_saturation', 'stop_animations', 'reading_mask' ],
            ],
            'blind_users' => [
                'id'          => 'blind_users',
                'name'        => __( 'Blindness Mode', 'everyone-accessibility-suite' ),
                'short'       => __( 'Allows using the site with your screen-reader', 'everyone-accessibility-suite' ),
                'description' => __( 'This mode configures the website to be compatible with screen-readers such as JAWS, NVDA, VoiceOver, and TalkBack. A screen-reader is software for blind users that is installed on a computer and smartphone, and websites must be compatible with it.', 'everyone-accessibility-suite' ),
                'icon'        => 'blind.svg',
                'features'    => [ 'keyboard_navigation' ],
            ],
        ];
    }

    /**
     * Get all typography features
     *
     * @return array
     */
    public static function get_typography_features(): array {
        return [
            'content_scaling' => [
                'id'       => 'content_scaling',
                'name'     => __( 'Content Scaling', 'everyone-accessibility-suite' ),
                'type'     => 'spinner',
                'step'     => 10,
                'min'      => -50,
                'max'      => 100,
                'default'  => 0,
                'unit'     => '%',
                'css_var'  => '--evas-content-scale',
            ],
            'readable_font' => [
                'id'      => 'readable_font',
                'name'    => __( 'Readable Font', 'everyone-accessibility-suite' ),
                'type'    => 'toggle',
                'default' => false,
                'class'   => 'evas-readable-font',
            ],
            'dyslexia_font' => [
                'id'      => 'dyslexia_font',
                'name'    => __( 'Dyslexia Friendly', 'everyone-accessibility-suite' ),
                'type'    => 'toggle',
                'default' => false,
                'class'   => 'evas-dyslexia-font',
            ],
            'font_sizing' => [
                'id'       => 'font_sizing',
                'name'     => __( 'Font Sizing', 'everyone-accessibility-suite' ),
                'type'     => 'spinner',
                'step'     => 5,
                'min'      => -20,
                'max'      => 50,
                'default'  => 0,
                'unit'     => '%',
                'css_var'  => '--evas-font-size',
            ],
            'line_height' => [
                'id'       => 'line_height',
                'name'     => __( 'Line Height', 'everyone-accessibility-suite' ),
                'type'     => 'spinner',
                'step'     => 10,
                'min'      => 0,
                'max'      => 100,
                'default'  => 0,
                'unit'     => '%',
                'css_var'  => '--evas-line-height',
            ],
            'letter_spacing' => [
                'id'       => 'letter_spacing',
                'name'     => __( 'Letter Spacing', 'everyone-accessibility-suite' ),
                'type'     => 'spinner',
                'step'     => 1,
                'min'      => 0,
                'max'      => 10,
                'default'  => 0,
                'unit'     => 'px',
                'css_var'  => '--evas-letter-spacing',
            ],
        ];
    }

    /**
     * Get all visual adjustment features
     *
     * @return array
     */
    public static function get_visual_features(): array {
        return [
            'highlight_titles' => [
                'id'      => 'highlight_titles',
                'name'    => __( 'Highlight Titles', 'everyone-accessibility-suite' ),
                'type'    => 'toggle',
                'default' => false,
                'class'   => 'evas-highlight-titles',
            ],
            'highlight_links' => [
                'id'      => 'highlight_links',
                'name'    => __( 'Highlight Links', 'everyone-accessibility-suite' ),
                'type'    => 'toggle',
                'default' => false,
                'class'   => 'evas-highlight-links',
            ],
            'highlight_hover' => [
                'id'      => 'highlight_hover',
                'name'    => __( 'Highlight Hover', 'everyone-accessibility-suite' ),
                'type'    => 'toggle',
                'default' => false,
                'class'   => 'evas-highlight-hover',
            ],
            'highlight_focus' => [
                'id'      => 'highlight_focus',
                'name'    => __( 'Highlight Focus', 'everyone-accessibility-suite' ),
                'type'    => 'toggle',
                'default' => false,
                'class'   => 'evas-highlight-focus',
            ],
            'text_magnifier' => [
                'id'      => 'text_magnifier',
                'name'    => __( 'Text Magnifier', 'everyone-accessibility-suite' ),
                'type'    => 'toggle',
                'default' => false,
                'class'   => 'evas-text-magnifier',
            ],
            'align_left' => [
                'id'      => 'align_left',
                'name'    => __( 'Align Left', 'everyone-accessibility-suite' ),
                'type'    => 'toggle',
                'default' => false,
                'class'   => 'evas-align-left',
                'group'   => 'alignment',
            ],
            'align_center' => [
                'id'      => 'align_center',
                'name'    => __( 'Align Center', 'everyone-accessibility-suite' ),
                'type'    => 'toggle',
                'default' => false,
                'class'   => 'evas-align-center',
                'group'   => 'alignment',
            ],
            'align_right' => [
                'id'      => 'align_right',
                'name'    => __( 'Align Right', 'everyone-accessibility-suite' ),
                'type'    => 'toggle',
                'default' => false,
                'class'   => 'evas-align-right',
                'group'   => 'alignment',
            ],
        ];
    }

    /**
     * Get all color adjustment features
     *
     * @return array
     */
    public static function get_color_features(): array {
        return [
            'dark_contrast' => [
                'id'      => 'dark_contrast',
                'name'    => __( 'Dark Contrast', 'everyone-accessibility-suite' ),
                'type'    => 'toggle',
                'default' => false,
                'class'   => 'evas-dark-contrast',
                'group'   => 'color_mode',
            ],
            'light_contrast' => [
                'id'      => 'light_contrast',
                'name'    => __( 'Light Contrast', 'everyone-accessibility-suite' ),
                'type'    => 'toggle',
                'default' => false,
                'class'   => 'evas-light-contrast',
                'group'   => 'color_mode',
            ],
            'monochrome' => [
                'id'      => 'monochrome',
                'name'    => __( 'Monochrome', 'everyone-accessibility-suite' ),
                'type'    => 'toggle',
                'default' => false,
                'class'   => 'evas-monochrome',
                'group'   => 'color_mode',
            ],
            'high_saturation' => [
                'id'      => 'high_saturation',
                'name'    => __( 'High Saturation', 'everyone-accessibility-suite' ),
                'type'    => 'toggle',
                'default' => false,
                'class'   => 'evas-high-saturation',
                'group'   => 'saturation',
            ],
            'low_saturation' => [
                'id'      => 'low_saturation',
                'name'    => __( 'Low Saturation', 'everyone-accessibility-suite' ),
                'type'    => 'toggle',
                'default' => false,
                'class'   => 'evas-low-saturation',
                'group'   => 'saturation',
            ],
            'high_contrast' => [
                'id'      => 'high_contrast',
                'name'    => __( 'High Contrast', 'everyone-accessibility-suite' ),
                'type'    => 'toggle',
                'default' => false,
                'class'   => 'evas-high-contrast',
                'group'   => 'color_mode',
            ],
            'text_colors' => [
                'id'      => 'text_colors',
                'name'    => __( 'Adjust Text Colors', 'everyone-accessibility-suite' ),
                'type'    => 'palette',
                'default' => false,
                'class'   => 'evas-text-colors',
            ],
            'title_colors' => [
                'id'      => 'title_colors',
                'name'    => __( 'Adjust Title Colors', 'everyone-accessibility-suite' ),
                'type'    => 'palette',
                'default' => false,
                'class'   => 'evas-title-colors',
            ],
            'background_colors' => [
                'id'      => 'background_colors',
                'name'    => __( 'Adjust Background Colors', 'everyone-accessibility-suite' ),
                'type'    => 'palette',
                'default' => false,
                'class'   => 'evas-background-colors',
            ],
        ];
    }

    /**
     * Get all media control features
     *
     * @return array
     */
    public static function get_media_features(): array {
        return [
            'mute_sounds' => [
                'id'      => 'mute_sounds',
                'name'    => __( 'Mute Sounds', 'everyone-accessibility-suite' ),
                'type'    => 'toggle',
                'default' => false,
                'class'   => 'evas-mute-sounds',
            ],
            'hide_images' => [
                'id'      => 'hide_images',
                'name'    => __( 'Hide Images', 'everyone-accessibility-suite' ),
                'type'    => 'toggle',
                'default' => false,
                'class'   => 'evas-hide-images',
            ],
            'hide_emoji' => [
                'id'      => 'hide_emoji',
                'name'    => __( 'Hide Emoji', 'everyone-accessibility-suite' ),
                'type'    => 'toggle',
                'default' => false,
                'class'   => 'evas-hide-emoji',
            ],
            'stop_animations' => [
                'id'      => 'stop_animations',
                'name'    => __( 'Stop Animations', 'everyone-accessibility-suite' ),
                'type'    => 'toggle',
                'default' => false,
                'class'   => 'evas-stop-animations',
            ],
        ];
    }

    /**
     * Get all tool features
     *
     * @return array
     */
    public static function get_tool_features(): array {
        return [
            'reading_guide' => [
                'id'      => 'reading_guide',
                'name'    => __( 'Reading Guide', 'everyone-accessibility-suite' ),
                'type'    => 'toggle',
                'default' => false,
                'class'   => 'evas-reading-guide',
            ],
            'reading_mask' => [
                'id'      => 'reading_mask',
                'name'    => __( 'Reading Mask', 'everyone-accessibility-suite' ),
                'type'    => 'toggle',
                'default' => false,
                'class'   => 'evas-reading-mask',
            ],
            'cognitive_reading' => [
                'id'      => 'cognitive_reading',
                'name'    => __( 'Cognitive Reading', 'everyone-accessibility-suite' ),
                'type'    => 'toggle',
                'default' => false,
                'class'   => 'evas-cognitive-reading',
            ],
            'useful_links' => [
                'id'      => 'useful_links',
                'name'    => __( 'Useful Links', 'everyone-accessibility-suite' ),
                'type'    => 'toggle',
                'default' => false,
                'class'   => 'evas-useful-links',
            ],
            'big_black_cursor' => [
                'id'      => 'big_black_cursor',
                'name'    => __( 'Big Dark Cursor', 'everyone-accessibility-suite' ),
                'type'    => 'toggle',
                'default' => false,
                'class'   => 'evas-big-black-cursor',
                'group'   => 'cursor',
            ],
            'big_white_cursor' => [
                'id'      => 'big_white_cursor',
                'name'    => __( 'Big Light Cursor', 'everyone-accessibility-suite' ),
                'type'    => 'toggle',
                'default' => false,
                'class'   => 'evas-big-white-cursor',
                'group'   => 'cursor',
            ],
            'keyboard_navigation' => [
                'id'      => 'keyboard_navigation',
                'name'    => __( 'Keyboard Navigation', 'everyone-accessibility-suite' ),
                'type'    => 'toggle',
                'default' => false,
                'class'   => 'evas-keyboard-navigation',
            ],
        ];
    }

    /**
     * Get all features combined
     *
     * @return array
     */
    public static function get_all_features(): array {
        return array_merge(
            self::get_typography_features(),
            self::get_visual_features(),
            self::get_color_features(),
            self::get_media_features(),
            self::get_tool_features()
        );
    }

    /**
     * Get feature by ID
     *
     * @param string $feature_id Feature identifier.
     * @return array|null
     */
    public static function get_feature( string $feature_id ): ?array {
        $all_features = self::get_all_features();
        return $all_features[ $feature_id ] ?? null;
    }

    /**
     * Get all feature groups (for panel sections)
     *
     * @return array
     */
    public static function get_feature_groups(): array {
        return [
            'profiles' => [
                'id'       => 'profiles',
                'name'     => __( 'Accessibility Profiles', 'everyone-accessibility-suite' ),
                'features' => self::get_profiles(),
            ],
            'typography' => [
                'id'       => 'typography',
                'name'     => __( 'Content & Typography', 'everyone-accessibility-suite' ),
                'features' => self::get_typography_features(),
            ],
            'visual' => [
                'id'       => 'visual',
                'name'     => __( 'Visual Adjustments', 'everyone-accessibility-suite' ),
                'features' => self::get_visual_features(),
            ],
            'colors' => [
                'id'       => 'colors',
                'name'     => __( 'Color Adjustments', 'everyone-accessibility-suite' ),
                'features' => self::get_color_features(),
            ],
            'media' => [
                'id'       => 'media',
                'name'     => __( 'Media Control', 'everyone-accessibility-suite' ),
                'features' => self::get_media_features(),
            ],
            'tools' => [
                'id'       => 'tools',
                'name'     => __( 'Helpful Tools', 'everyone-accessibility-suite' ),
                'features' => self::get_tool_features(),
            ],
        ];
    }

    /**
     * Get default settings for all features
     *
     * @return array
     */
    public static function get_default_settings(): array {
        $defaults = [];
        
        foreach ( self::get_all_features() as $id => $feature ) {
            $defaults[ $id ] = [
                'enabled'  => true, // Feature is available to users
                'default'  => $feature['default'] ?? false,
            ];
        }

        // Profiles defaults
        foreach ( self::get_profiles() as $id => $profile ) {
            $defaults[ 'profile_' . $id ] = [
                'enabled' => true,
            ];
        }

        return $defaults;
    }
}

