<?php
/**
 * Settings Page Class
 *
 * REST API for settings management
 *
 * @package Everyone_Accessibility_Suite
 * @since 1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Settings page REST API class
 */
class EVAS_Settings_Page {

	/**
	 * Settings manager instance
	 *
	 * @var EVAS_Settings
	 */
	private $settings_manager;

	/**
	 * Constructor
	 *
	 * @param EVAS_Settings $settings_manager Settings manager instance.
	 */
	public function __construct( $settings_manager ) {
		$this->settings_manager = $settings_manager;
		add_action( 'rest_api_init', array( $this, 'register_routes' ) );
	}

	/**
	 * Register REST API routes
	 */
	public function register_routes() {
		// Get all settings
		register_rest_route(
			'evas/v1',
			'/settings',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_settings' ),
				'permission_callback' => array( $this, 'check_permissions' ),
			)
		);

		// Update settings
		register_rest_route(
			'evas/v1',
			'/settings',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'update_settings' ),
				'permission_callback' => array( $this, 'check_permissions' ),
			)
		);

		// Get accessibility settings
		register_rest_route(
			'evas/v1',
			'/accessibility/settings',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_accessibility_settings' ),
				'permission_callback' => array( $this, 'check_permissions' ),
			)
		);

		// Update accessibility settings
		register_rest_route(
			'evas/v1',
			'/accessibility/settings',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'update_accessibility_settings' ),
				'permission_callback' => array( $this, 'check_permissions' ),
			)
		);
	}

	/**
	 * Get all settings
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response
	 */
	public function get_settings( $request ) {
		$settings = $this->settings_manager->get_all_settings();

		return rest_ensure_response(
			array(
				'success'  => true,
				'settings' => $settings,
			)
		);
	}

	/**
	 * Update settings
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response
	 */
	public function update_settings( $request ) {
		$new_settings = $request->get_json_params();

		if ( empty( $new_settings ) ) {
			return rest_ensure_response(
				array(
					'success' => false,
					'message' => __( 'No settings provided', 'everyone-accessibility-suite' ),
				)
			);
		}

		// Sanitize settings before saving
		$sanitized_settings = $this->sanitize_settings( $new_settings );

		// Update settings
		$updated = $this->settings_manager->update_all_settings( $sanitized_settings );

		if ( $updated ) {
			return rest_ensure_response(
				array(
					'success'  => true,
					'settings' => $sanitized_settings,
					'message'  => __( 'Settings updated successfully', 'everyone-accessibility-suite' ),
				)
			);
		}

		return rest_ensure_response(
			array(
				'success' => false,
				'message' => __( 'Failed to update settings', 'everyone-accessibility-suite' ),
			)
		);
	}

	/**
	 * Get accessibility settings
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response
	 */
	public function get_accessibility_settings( $request ) {
		$settings = $this->settings_manager->get_all_settings();
		
		// Set default values (all 'on' by default)
		$defaults = $this->get_accessibility_defaults();
		$settings = wp_parse_args( $settings, $defaults );

		return rest_ensure_response(
			array(
				'success'  => true,
				'settings' => $settings,
			)
		);
	}

	/**
	 * Get default accessibility settings (all enabled by default)
	 *
	 * @return array Default settings.
	 */
	private function get_accessibility_defaults() {
		$defaults = array(
			'enabled'       => 'on',
			'panel_position' => 'bottom-right',
			
			// Accessibility Profiles
			'accessibility_profiles_heading' => 'on',
			'profile_epilepsy' => 'on',
			'profile_visually_impaired' => 'on',
			'profile_cognitive_disability' => 'on',
			'profile_adhd_friendly' => 'on',
			'profile_blind_users' => 'on',
			
			// Readable Experience
			'readable_experience_heading' => 'on',
			'content_scaling' => 'on',
			'readable_font' => 'on',
			'dyslexia_font' => 'on',
			'text_magnifier' => 'on',
			'highlight_titles' => 'on',
			'highlight_links' => 'on',
			'font_sizing' => 'on',
			'line_height' => 'on',
			'letter_spacing' => 'on',
			'align_left' => 'on',
			'align_center' => 'on',
			'align_right' => 'on',
			
			// Visually Pleasing Experience
			'visually_pleasing_heading' => 'on',
			'dark_contrast' => 'on',
			'light_contrast' => 'on',
			'monochrome' => 'on',
			'high_saturation' => 'on',
			'high_contrast' => 'on',
			'low_saturation' => 'on',
			'text_colors' => 'on',
			'title_colors' => 'on',
			'background_colors' => 'on',
			
			// Easy Orientation
			'easy_orientation_heading' => 'on',
			'mute_sounds' => 'on',
			'hide_images' => 'on',
			'hide_emoji' => 'on',
			'reading_guide' => 'on',
			'stop_animations' => 'on',
			'reading_mask' => 'on',
			'highlight_hover' => 'on',
			'highlight_focus' => 'on',
			'big_black_cursor' => 'on',
			'big_white_cursor' => 'on',
			'cognitive_reading' => 'on',
			'keyboard_navigation' => 'on',
			'useful_links' => 'on',
			
			// Module Features
			'text_to_speech' => 'off', // off by default (requires API key)
			'voice_navigation' => 'on',
			'virtual_keyboard' => 'on',
			'online_dictionary' => 'on',
			
			// Online Dictionary
			'online_dictionary_heading' => 'on',
		);
		
		return $defaults;
	}

	/**
	 * Update accessibility settings
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response
	 */
	public function update_accessibility_settings( $request ) {
		$new_settings = $request->get_json_params();

		if ( empty( $new_settings ) ) {
			return rest_ensure_response(
				array(
					'success' => false,
					'message' => __( 'No settings provided', 'everyone-accessibility-suite' ),
				)
			);
		}

		// Sanitize settings before saving
		$sanitized_settings = $this->sanitize_accessibility_settings( $new_settings );

		// Merge with existing settings
		$existing_settings = $this->settings_manager->get_all_settings();
		$merged_settings   = array_merge( $existing_settings, $sanitized_settings );

		// Update settings
		$updated = $this->settings_manager->update_all_settings( $merged_settings );

		if ( $updated ) {
			return rest_ensure_response(
				array(
					'success'  => true,
					'settings' => $merged_settings,
					'message'  => __( 'Settings updated successfully', 'everyone-accessibility-suite' ),
				)
			);
		}

		return rest_ensure_response(
			array(
				'success' => false,
				'message' => __( 'Failed to update settings', 'everyone-accessibility-suite' ),
			)
		);
	}

	/**
	 * Sanitize accessibility settings (Readabler-style)
	 *
	 * @param array $settings Settings to sanitize.
	 * @return array Sanitized settings.
	 */
	private function sanitize_accessibility_settings( $settings ) {
		$sanitized = array();

		// Boolean/enabled field keys (on/off values)
		$on_off_fields = array(
			'enabled',
			// Accessibility Profiles
			'profile_epilepsy',
			'profile_visually_impaired',
			'profile_cognitive_disability',
			'profile_adhd_friendly',
			'profile_blind_users',
			// Readable Experience
			'content_scaling',
			'readable_font',
			'dyslexia_font',
			'highlight_titles',
			'highlight_links',
			'text_magnifier',
			'font_sizing',
			'line_height',
			'letter_spacing',
			'align_left',
			'align_center',
			'align_right',
			// Visually Pleasing
			'dark_contrast',
			'light_contrast',
			'monochrome',
			'high_saturation',
			'high_contrast',
			'low_saturation',
			'text_colors',
			'title_colors',
			'background_colors',
			// Easy Orientation
			'mute_sounds',
			'hide_images',
			'hide_emoji',
			'reading_guide',
			'stop_animations',
			'reading_mask',
			'highlight_hover',
			'highlight_focus',
			'big_black_cursor',
			'big_white_cursor',
			'cognitive_reading',
			'text_to_speech',
			'virtual_keyboard',
			'keyboard_navigation',
			'voice_navigation',
			'useful_links',
			'online_dictionary',
			// Section Headings
			'accessibility_profiles_heading',
			'readable_experience_heading',
			'visually_pleasing_heading',
			'easy_orientation_heading',
			'online_dictionary_heading',
		);

		foreach ( $on_off_fields as $field ) {
			if ( isset( $settings[ $field ] ) ) {
				// Handle both boolean and 'on'/'off' values
				if ( is_bool( $settings[ $field ] ) ) {
					$sanitized[ $field ] = $settings[ $field ] ? 'on' : 'off';
				} elseif ( $settings[ $field ] === true || $settings[ $field ] === 'on' || $settings[ $field ] === '1' || $settings[ $field ] === 1 ) {
					$sanitized[ $field ] = 'on';
				} else {
					$sanitized[ $field ] = 'off';
				}
			}
		}

		// Panel position
		if ( isset( $settings['panel_position'] ) ) {
			$allowed_positions = array( 'top-left', 'top-right', 'bottom-left', 'bottom-right', 'left', 'right' );
			$sanitized['panel_position'] = in_array( $settings['panel_position'], $allowed_positions, true )
				? sanitize_text_field( $settings['panel_position'] )
				: 'bottom-right';
		}

		return $sanitized;
	}

	/**
	 * Sanitize settings
	 *
	 * @param array $settings Settings to sanitize.
	 * @return array Sanitized settings.
	 */
	private function sanitize_settings( $settings ) {
		$sanitized = array();

		// Sanitize enabled checkbox
		if ( isset( $settings['enabled'] ) ) {
			$sanitized['enabled'] = (bool) $settings['enabled'];
		}

		// Sanitize panel position
		if ( isset( $settings['panel_position'] ) ) {
			$sanitized['panel_position'] = in_array( $settings['panel_position'], array( 'left', 'right' ), true )
				? $settings['panel_position']
				: 'left';
		}

		// Sanitize font sizes
		if ( isset( $settings['min_font_size'] ) ) {
			$sanitized['min_font_size'] = absint( $settings['min_font_size'] );
			$sanitized['min_font_size'] = max( 50, min( 100, $sanitized['min_font_size'] ) );
		}

		if ( isset( $settings['max_font_size'] ) ) {
			$sanitized['max_font_size'] = absint( $settings['max_font_size'] );
			$sanitized['max_font_size'] = max( 100, min( 300, $sanitized['max_font_size'] ) );
		}

		if ( isset( $settings['default_font_size'] ) ) {
			$sanitized['default_font_size'] = absint( $settings['default_font_size'] );
			$sanitized['default_font_size'] = max( 50, min( 300, $sanitized['default_font_size'] ) );
		}

		// Sanitize enabled themes
		if ( isset( $settings['enabled_themes'] ) && is_array( $settings['enabled_themes'] ) ) {
			$allowed_themes               = array( 'light', 'dark', 'high-contrast' );
			$sanitized['enabled_themes'] = array_intersect( $settings['enabled_themes'], $allowed_themes );
		}

		// Sanitize default theme
		if ( isset( $settings['default_theme'] ) ) {
			$allowed_themes              = array( 'light', 'dark', 'high-contrast' );
			$sanitized['default_theme'] = in_array( $settings['default_theme'], $allowed_themes, true )
				? $settings['default_theme']
				: 'light';
		}

		// Sanitize modules settings
		if ( isset( $settings['modules'] ) && is_array( $settings['modules'] ) ) {
			$sanitized['modules'] = array();
			foreach ( $settings['modules'] as $module_id => $module_settings ) {
				if ( is_array( $module_settings ) ) {
					$sanitized['modules'][ $module_id ] = array_map( 'sanitize_text_field', $module_settings );
					// Preserve boolean values
					if ( isset( $module_settings['enabled'] ) ) {
						$sanitized['modules'][ $module_id ]['enabled'] = (bool) $module_settings['enabled'];
					}
				} else {
					$sanitized['modules'][ $module_id ] = (bool) $module_settings;
				}
			}
		}

		return $sanitized;
	}

	/**
	 * Check permissions
	 *
	 * @return bool
	 */
	public function check_permissions() {
		return current_user_can( 'manage_options' );
	}
}

