<?php
/**
 * Components Page Class
 *
 * REST API for modules/components management
 *
 * @package Everyone_Accessibility_Suite
 * @since 1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Components page REST API class
 */
class EVAS_Components_Page {

	/**
	 * Module manager instance
	 *
	 * @var EVAS_Module_Manager
	 */
	private $module_manager;

	/**
	 * Constructor
	 *
	 * @param EVAS_Module_Manager $module_manager Module manager instance.
	 */
	public function __construct( $module_manager ) {
		$this->module_manager = $module_manager;
		add_action( 'rest_api_init', array( $this, 'register_routes' ) );
	}

	/**
	 * Register REST API routes
	 */
	public function register_routes() {
		// Get all modules
		register_rest_route(
			'evas/v1',
			'/modules',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_modules' ),
				'permission_callback' => array( $this, 'check_permissions' ),
			)
		);

		// Toggle module status
		register_rest_route(
			'evas/v1',
			'/modules/toggle',
			array(
				'methods'             => 'POST',
				'callback'            => array( $this, 'toggle_module' ),
				'permission_callback' => array( $this, 'check_permissions' ),
				'args'                => array(
					'module_id' => array(
						'required'          => true,
						'type'              => 'string',
						'sanitize_callback' => 'sanitize_text_field',
					),
					'enabled'   => array(
						'required' => true,
						'type'     => 'boolean',
					),
				),
			)
		);

		// Get active modules
		register_rest_route(
			'evas/v1',
			'/modules/active',
			array(
				'methods'             => 'GET',
				'callback'            => array( $this, 'get_active_modules' ),
				'permission_callback' => array( $this, 'check_permissions' ),
			)
		);
	}

	/**
	 * Get all modules
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response
	 */
	public function get_modules( $request ) {
		$modules = $this->module_manager->get_available_modules_info();

		// Filter out accessibility_features (now core functionality)
		$modules = array_filter(
			$modules,
			function ( $module ) {
				return $module['id'] !== 'accessibility_features';
			}
		);

		// Re-index array
		$modules = array_values( $modules );

		// Sort modules: usage_analytics first, then others
		usort( $modules, function( $a, $b ) {
			if ( $a['id'] === 'usage_analytics' ) {
				return -1;
			}
			if ( $b['id'] === 'usage_analytics' ) {
				return 1;
			}
			return 0;
		});

		return rest_ensure_response(
			array(
				'success' => true,
				'modules' => $modules,
			)
		);
	}

	/**
	 * Get active modules
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response
	 */
	public function get_active_modules( $request ) {
		$active_modules = $this->module_manager->get_active_modules();

		return rest_ensure_response(
			array(
				'success' => true,
				'modules' => $active_modules,
			)
		);
	}

	/**
	 * Toggle module status
	 *
	 * @param WP_REST_Request $request Request object.
	 * @return WP_REST_Response
	 */
	public function toggle_module( $request ) {
		$module_id = $request->get_param( 'module_id' );
		$enabled   = $request->get_param( 'enabled' );

		if ( empty( $module_id ) ) {
			return rest_ensure_response(
				array(
					'success' => false,
					'message' => __( 'Module ID is required', 'everyone-accessibility-suite' ),
				)
			);
		}

		// Get settings manager from global plugin instance
		global $evas_plugin;
		if ( ! $evas_plugin ) {
			return rest_ensure_response(
				array(
					'success' => false,
					'message' => __( 'Plugin not initialized', 'everyone-accessibility-suite' ),
				)
			);
		}

		$settings_manager = $evas_plugin->get_settings_manager();

		// Check if module is locked (only in production, usage_analytics is always available)
		if ( $enabled && $module_id !== 'usage_analytics' ) {
			if ( ! defined( 'EVAS_DEBUG' ) || ! EVAS_DEBUG ) {
				return rest_ensure_response(
					array(
						'success' => false,
						'message' => __( 'This module is coming soon and is not available yet.', 'everyone-accessibility-suite' ),
					)
				);
			}
		}

		// Usage Analytics cannot be disabled
		if ( ! $enabled && $module_id === 'usage_analytics' ) {
			return rest_ensure_response(
				array(
					'success' => false,
					'message' => __( 'Usage Analytics module cannot be disabled. It is always active.', 'everyone-accessibility-suite' ),
				)
			);
		}

		// Toggle module
		if ( $enabled ) {
			$result = $settings_manager->enable_module( $module_id );
		} else {
			$result = $settings_manager->disable_module( $module_id );
		}

		if ( $result ) {
			return rest_ensure_response(
				array(
					'success' => true,
					'message' => $enabled
						? __( 'Module enabled successfully', 'everyone-accessibility-suite' )
						: __( 'Module disabled successfully', 'everyone-accessibility-suite' ),
					'modules' => $this->module_manager->get_available_modules_info(),
				)
			);
		}

		return rest_ensure_response(
			array(
				'success' => false,
				'message' => __( 'Failed to toggle module', 'everyone-accessibility-suite' ),
			)
		);
	}

	/**
	 * Check permissions
	 *
	 * @return bool
	 */
	public function check_permissions() {
		return current_user_can( 'manage_options' );
	}
}

