<?php
/**
 * Admin App Class
 *
 * Manages the main admin page with React SPA application
 *
 * @package Everyone_Accessibility_Suite
 * @since 1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Admin App class
 */
class EVAS_Admin_App {

	/**
	 * Page slug
	 *
	 * @var string
	 */
	private $page_slug = 'evas-admin';

	/**
	 * Module manager instance
	 *
	 * @var EVAS_Module_Manager
	 */
	private $module_manager;

	/**
	 * Settings manager instance
	 *
	 * @var EVAS_Settings
	 */
	private $settings_manager;

	/**
	 * Constructor
	 *
	 * @param EVAS_Module_Manager $module_manager Module manager instance.
	 * @param EVAS_Settings       $settings_manager Settings manager instance.
	 */
	public function __construct( $module_manager, $settings_manager ) {
		$this->module_manager   = $module_manager;
		$this->settings_manager = $settings_manager;

		add_action( 'admin_menu', array( $this, 'add_menu_page' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
	}

	/**
	 * Add menu page
	 */
	public function add_menu_page() {
		// Main menu page
		add_menu_page(
			__( 'Accessibility Suite', 'everyone-accessibility-suite' ),
			__( 'Accessibility', 'everyone-accessibility-suite' ),
			'manage_options',
			$this->page_slug,
			array( $this, 'render_page' ),
			'dashicons-universal-access',
			100
		);

		// Submenu: Dashboard (first item, same as main menu)
		add_submenu_page(
			$this->page_slug,
			__( 'Dashboard', 'everyone-accessibility-suite' ),
			__( 'Dashboard', 'everyone-accessibility-suite' ),
			'manage_options',
			$this->page_slug,
			array( $this, 'render_page' )
		);

		// Submenu: Components
		add_submenu_page(
			$this->page_slug,
			__( 'Components', 'everyone-accessibility-suite' ),
			__( 'Components', 'everyone-accessibility-suite' ),
			'manage_options',
			$this->page_slug . '#/components',
			array( $this, 'render_page' )
		);

		// Submenu: Settings
		add_submenu_page(
			$this->page_slug,
			__( 'Settings', 'everyone-accessibility-suite' ),
			__( 'Settings', 'everyone-accessibility-suite' ),
			'manage_options',
			$this->page_slug . '#/settings',
			array( $this, 'render_page' )
		);

		// Submenu: Accessibility Panel (only if module is enabled)
		if ( $this->module_manager && $this->module_manager->is_module_enabled( 'accessibility_panel' ) ) {
			add_submenu_page(
				$this->page_slug,
				__( 'Accessibility Panel', 'everyone-accessibility-suite' ),
				__( 'Accessibility Panel', 'everyone-accessibility-suite' ),
				'manage_options',
				$this->page_slug . '#/accessibility',
				array( $this, 'render_page' )
			);
		}

		// Submenu: Customizer (only if module is enabled)
		if ( $this->module_manager && $this->module_manager->is_module_enabled( 'customizer' ) ) {
			add_submenu_page(
				$this->page_slug,
				__( 'Customizer', 'everyone-accessibility-suite' ),
				__( 'Customizer', 'everyone-accessibility-suite' ),
				'manage_options',
				$this->page_slug . '#/customizer',
				array( $this, 'render_page' )
			);
		}

		// Submenu: Usage Analytics (only if module is enabled)
		if ( $this->module_manager && $this->module_manager->is_module_enabled( 'usage_analytics' ) ) {
			add_submenu_page(
				$this->page_slug,
				__( 'Usage Analytics', 'everyone-accessibility-suite' ),
				__( 'Usage Analytics', 'everyone-accessibility-suite' ),
				'manage_options',
				$this->page_slug . '#/usage-analytics',
				array( $this, 'render_page' )
			);
		}
	}

	/**
	 * Enqueue scripts
	 *
	 * @param string $hook Current admin page hook.
	 */
	public function enqueue_scripts( $hook ) {
		// Only load on our admin page
		if ( strpos( $hook, $this->page_slug ) === false ) {
			return;
		}

		// Get registered routes from modules
		$routes = apply_filters( 'evas_admin_routes', array() );

		// Get modules info
		$modules_info = $this->module_manager ? $this->module_manager->get_available_modules_info() : array();

		// Get settings
		$settings = $this->settings_manager ? $this->settings_manager->get_all_settings() : array();

		// Enqueue compiled React app
		$asset_file = EVAS_PLUGIN_DIR . 'includes/admin/assets/dist/admin-app.asset.php';
		$js_file    = EVAS_PLUGIN_DIR . 'includes/admin/assets/dist/admin-app.js';
		$css_file   = EVAS_PLUGIN_DIR . 'includes/admin/assets/dist/style.css';

		if ( file_exists( $asset_file ) ) {
			$asset = require $asset_file;
		} else {
			$asset = array(
				'dependencies' => array( 'react', 'react-dom', 'wp-api-fetch', 'wp-components', 'wp-element', 'wp-i18n' ),
				'version'      => EVAS_VERSION,
			);
		}

		// Use plugin version for cache busting
		$js_version = EVAS_VERSION;

		wp_enqueue_script(
			'evas-admin-app',
			EVAS_PLUGIN_URL . 'includes/admin/assets/dist/admin-app.js',
			$asset['dependencies'],
			$js_version,
			true
		);

		// Enqueue styles if exists
		if ( file_exists( $css_file ) ) {
			wp_enqueue_style(
				'evas-admin-app',
				EVAS_PLUGIN_URL . 'includes/admin/assets/dist/style.css',
				array( 'wp-components' ),
				EVAS_VERSION
			);
		}

		// Localize script
		wp_localize_script(
			'evas-admin-app',
			'evasAdmin',
			array(
				'apiUrl'        => 'evas/v1/', // Use namespace only, apiFetch will handle the rest
				'nonce'         => wp_create_nonce( 'wp_rest' ),
				'pluginUrl'     => EVAS_PLUGIN_URL,
				'pluginName'    => __( 'Everyone Accessibility Suite', 'everyone-accessibility-suite' ),
				'pluginVersion' => EVAS_VERSION,
				'routes'        => $routes,
				'modules'       => $modules_info,
				'settings'      => $settings,
				'adminUrl'      => admin_url( 'admin.php?page=' . $this->page_slug ),
				'i18n'          => array(
					'dashboard'      => __( 'Dashboard', 'everyone-accessibility-suite' ),
					'components'     => __( 'Components', 'everyone-accessibility-suite' ),
					'settings'       => __( 'Settings', 'everyone-accessibility-suite' ),
					'accessibility'  => __( 'Accessibility', 'everyone-accessibility-suite' ),
					'save'           => __( 'Save Changes', 'everyone-accessibility-suite' ),
					'saved'          => __( 'Settings saved successfully', 'everyone-accessibility-suite' ),
					'error'          => __( 'Error saving settings', 'everyone-accessibility-suite' ),
					'loading'        => __( 'Loading...', 'everyone-accessibility-suite' ),
					'enableModule'   => __( 'Enable Module', 'everyone-accessibility-suite' ),
					'disableModule'  => __( 'Disable Module', 'everyone-accessibility-suite' ),
					'moduleEnabled'  => __( 'Module enabled successfully', 'everyone-accessibility-suite' ),
					'moduleDisabled' => __( 'Module disabled successfully', 'everyone-accessibility-suite' ),
				),
			)
		);
	}

	/**
	 * Render admin page
	 */
	public function render_page() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}
		?>
		<div class="wrap">
			<div id="evas-admin-root"></div>
		</div>
		<?php
	}
}

