<?php
/**
 * Plugin Name: Everyone Accessibility Suite
 * Plugin URI: https://wpaifactory.com/everyone-accessibility-suite/
 * Description: A comprehensive accessibility toolkit that provides a floating panel with font size adjustment and theme selection for improved website accessibility.
 * Version: 1.1.1
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * Author: WP Ai Factory
 * Author URI: https://wpaifactory.com/
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: everyone-accessibility-suite
 * Domain Path: /languages
 *
 * @package Everyone_Accessibility_Suite
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Current plugin version.
 */
define( 'EVAS_VERSION', '1.1.1' );

/**
 * Plugin file path.
 */
define( 'EVAS_PLUGIN_FILE', __FILE__ );

/**
 * Plugin directory path.
 */
define( 'EVAS_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );

/**
 * Plugin directory URL.
 */
define( 'EVAS_PLUGIN_URL', plugin_dir_url( __FILE__ ) );

/**
 * Plugin basename.
 */
define( 'EVAS_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );

/**
 * Debug mode - enables all modules for development/testing
 * Set to true to unlock all modules, false for production
 */
if ( ! defined( 'EVAS_DEBUG' ) ) {
	define( 'EVAS_DEBUG', false );
}

// Check PHP version compatibility.
if ( version_compare( PHP_VERSION, '7.4', '<' ) ) {
	add_action( 'admin_notices', function() {
		echo '<div class="error"><p>';
		echo esc_html__( 'Everyone Accessibility Suite requires PHP 7.4 or higher. You are running PHP ', 'everyone-accessibility-suite' );
		echo esc_html( PHP_VERSION );
		echo '.</p></div>';
	});
	return;
}

/**
 * Autoloader for plugin classes.
 *
 * @param string $class_name The class name to load.
 */
function evas_autoloader( $class_name ) {
	// Check if class belongs to our plugin.
	if ( strpos( $class_name, 'EVAS_' ) !== 0 ) {
		return;
	}

	// Convert class name to file name.
	$class_file = str_replace( 'EVAS_', '', $class_name );
	$class_file = strtolower( str_replace( '_', '-', $class_file ) );
	$class_file = 'class-' . $class_file . '.php';

	// Check in includes/core directory.
	$core_path = EVAS_PLUGIN_DIR . 'includes/core/' . $class_file;
	if ( file_exists( $core_path ) ) {
		require_once $core_path;
		return;
	}

	// Check in includes/admin directory.
	$admin_path = EVAS_PLUGIN_DIR . 'includes/admin/' . $class_file;
	if ( file_exists( $admin_path ) ) {
		require_once $admin_path;
		return;
	}

	// Check in old includes directory (backwards compatibility).
	$includes_path = EVAS_PLUGIN_DIR . 'includes/' . 'class-evas-' . strtolower( str_replace( '_', '-', str_replace( 'EVAS_', '', $class_name ) ) ) . '.php';
	if ( file_exists( $includes_path ) ) {
		require_once $includes_path;
		return;
	}

	// Check in includes/modules/{module_name}/ directory.
	$modules_dir = EVAS_PLUGIN_DIR . 'includes/modules/';
	if ( is_dir( $modules_dir ) ) {
		$modules = glob( $modules_dir . '*/', GLOB_ONLYDIR );
		foreach ( $modules as $module_dir ) {
			$module_path = $module_dir . $class_file;
			if ( file_exists( $module_path ) ) {
				require_once $module_path;
				return;
			}
		}
	}
}
spl_autoload_register( 'evas_autoloader' );

// Load Composer autoloader for vendor dependencies
if ( file_exists( EVAS_PLUGIN_DIR . 'vendor/autoload.php' ) ) {
	require_once EVAS_PLUGIN_DIR . 'vendor/autoload.php';
}

/**
 * Global plugin instance
 *
 * @var EVAS_Plugin_Loader
 */
global $evas_plugin;

/**
 * Available modules configuration
 * Note: accessibility_features is now core functionality (always active)
 */
$evas_modules = array(
	'usage_analytics'      => array(
		'autoload'      => 'includes/modules/usage-analytics/evas-usage-analytics.php',
		'module_prefix' => 'EVAS_UA_',
		'name'          => __( 'Usage Analytics', 'everyone-accessibility-suite' ),
		'description'   => __( 'Track accessibility panel usage with detailed statistics and charts.', 'everyone-accessibility-suite' ),
		'icon'          => 'dashicons-chart-area',
		'type'          => 'internal',
		'build'         => 'pro',
	),
	'customizer' => array(
		'autoload'      => 'includes/modules/customizer/evas-customizer.php',
		'module_prefix' => 'EVAS_CUSTOMIZER_',
		'name'          => __( 'Customizer', 'everyone-accessibility-suite' ),
		'description'   => __( 'Customize the appearance and behavior of the accessibility panel and button.', 'everyone-accessibility-suite' ),
		'icon'          => 'dashicons-admin-customizer',
		'type'          => 'internal',
		'build'         => 'pro',
	),
	'text_to_speech' => array(
		'autoload'      => 'includes/modules/text-to-speech/evas-text-to-speech.php',
		'module_prefix' => 'EVAS_TTS_',
		'name'          => __( 'Text-to-Speech', 'everyone-accessibility-suite' ),
		'description'   => __( 'Convert text to speech using Google Cloud Text-to-Speech API.', 'everyone-accessibility-suite' ),
		'icon'          => 'dashicons-controls-volumeon',
		'type'          => 'internal',
		'build'         => 'pro',
	),
	'voice_navigation' => array(
		'autoload'      => 'includes/modules/voice-navigation/evas-voice-navigation.php',
		'module_prefix' => 'EVAS_VN_',
		'name'          => __( 'Voice Navigation', 'everyone-accessibility-suite' ),
		'description'   => __( 'Navigate your website using voice commands with Web Speech API.', 'everyone-accessibility-suite' ),
		'icon'          => 'dashicons-microphone',
		'type'          => 'internal',
		'build'         => 'pro',
	),
	'virtual_keyboard' => array(
		'autoload'      => 'includes/modules/virtual-keyboard/evas-virtual-keyboard.php',
		'module_prefix' => 'EVAS_VK_',
		'name'          => __( 'Virtual Keyboard', 'everyone-accessibility-suite' ),
		'description'   => __( 'On-screen keyboard with multiple layouts and language support.', 'everyone-accessibility-suite' ),
		'icon'          => 'dashicons-keyboard-hide',
		'type'          => 'internal',
		'build'         => 'pro',
	),
	'accessibility_analyzer' => array(
		'autoload'      => 'includes/modules/accessibility-analyzer/evas-accessibility-analyzer.php',
		'module_prefix' => 'EVAS_AA_',
		'name'          => __( 'Accessibility Analyzer', 'everyone-accessibility-suite' ),
		'description'   => __( 'Scan your website for WCAG 2.1 compliance and accessibility issues.', 'everyone-accessibility-suite' ),
		'icon'          => 'dashicons-chart-bar',
		'type'          => 'internal',
		'build'         => 'pro',
	),
	'accessibility_extras' => array(
		'autoload'      => 'includes/modules/accessibility-extras/evas-accessibility-extras.php',
		'module_prefix' => 'EVAS_AE_',
		'name'          => __( 'Accessibility Statement', 'everyone-accessibility-suite' ),
		'description'   => __( 'Generate and display an accessibility statement for your website.', 'everyone-accessibility-suite' ),
		'icon'          => 'dashicons-media-document',
		'type'          => 'internal',
		'build'         => 'pro',
	),
);

/**
 * Initialize the plugin.
 */
function evas_init() {
	global $evas_plugin, $evas_modules;

	// Initialize plugin loader.
	if ( class_exists( 'EVAS_Plugin_Loader' ) ) {
		$evas_plugin = new EVAS_Plugin_Loader( $evas_modules );
		$evas_plugin->run();
	}

	// Initialize core accessibility panel (always active)
	if ( class_exists( 'EVAS_Accessibility_Panel' ) ) {
		$settings = new EVAS_Settings();
		$panel    = new EVAS_Accessibility_Panel( $settings );
		$panel->init();
	}
}
add_action( 'plugins_loaded', 'evas_init' );

/**
 * Activation hook.
 */
function evas_activate() {
	// Set default options matching Readabler structure.
	$default_options = array(
		'enabled'           => true,
		'panel_position'    => 'bottom-right',
		'open_button_icon'  => 'accessibility',
		
		// Accessibility Profiles
		'profile_epilepsy'             => 'on',
		'profile_visually_impaired'    => 'on',
		'profile_cognitive_disability' => 'on',
		'profile_adhd_friendly'        => 'on',
		'profile_blind_users'          => 'on',
		
		// Readable Experience
		'content_scaling'    => 'on',
		'readable_font'      => 'on',
		'dyslexia_font'      => 'on',
		'highlight_titles'   => 'on',
		'highlight_links'    => 'on',
		'text_magnifier'     => 'on',
		'font_sizing'        => 'on',
		'line_height'        => 'on',
		'letter_spacing'     => 'on',
		'align_left'         => 'on',
		'align_center'       => 'on',
		'align_right'        => 'on',
		
		// Visually Pleasing Experience
		'dark_contrast'      => 'on',
		'light_contrast'     => 'on',
		'monochrome'         => 'on',
		'high_saturation'    => 'on',
		'high_contrast'      => 'on',
		'low_saturation'     => 'on',
		'text_colors'        => 'on',
		'title_colors'       => 'on',
		'background_colors'  => 'on',
		
		// Easy Orientation
		'mute_sounds'         => 'on',
		'hide_images'         => 'on',
		'hide_emoji'          => 'on',
		'reading_guide'       => 'on',
		'stop_animations'     => 'on',
		'reading_mask'        => 'on',
		'highlight_hover'     => 'on',
		'highlight_focus'     => 'on',
		'big_black_cursor'    => 'on',
		'big_white_cursor'    => 'on',
		'cognitive_reading'   => 'on',
		'text_to_speech'      => 'on',
		'virtual_keyboard'    => 'on',
		'keyboard_navigation' => 'on',
		'voice_navigation'    => 'on',
		'useful_links'        => 'on',
		'online_dictionary'   => 'on',
		
		// Section Headings
		'accessibility_profiles_heading' => 'on',
		'readable_experience_heading'    => 'on',
		'visually_pleasing_heading'      => 'on',
		'easy_orientation_heading'       => 'on',
		'online_dictionary_heading'      => 'on',
		
		// Modules
		'modules' => array(
			'usage_analytics'        => array( 'enabled' => true ), // Always enabled
			'text_to_speech'         => array( 'enabled' => false ),
			'voice_navigation'       => array( 'enabled' => false ),
			'virtual_keyboard'       => array( 'enabled' => false ),
			'accessibility_analyzer' => array( 'enabled' => false ),
			'accessibility_extras'   => array( 'enabled' => false ),
			'customizer'             => array( 'enabled' => false ),
		),
	);

	$existing_settings = get_option( 'evas_settings' );

	if ( ! $existing_settings ) {
		add_option( 'evas_settings', $default_options );
	} else {
		// Merge with new defaults, preserving existing values.
		$merged = array_merge( $default_options, $existing_settings );
		
		// Ensure modules array exists and has all modules.
		if ( ! isset( $merged['modules'] ) ) {
			$merged['modules'] = $default_options['modules'];
		} else {
			$merged['modules'] = array_merge( $default_options['modules'], $merged['modules'] );
		}
		
		// Remove old accessibility_panel module if exists.
		unset( $merged['modules']['accessibility_panel'] );
		
		// Ensure usage_analytics is always enabled
		if ( ! isset( $merged['modules']['usage_analytics'] ) ) {
			$merged['modules']['usage_analytics'] = array( 'enabled' => true );
		} else {
			$merged['modules']['usage_analytics']['enabled'] = true;
		}
		
		update_option( 'evas_settings', $merged );
	}

	// Flush rewrite rules if needed.
	flush_rewrite_rules();
}
register_activation_hook( __FILE__, 'evas_activate' );

/**
 * Deactivation hook.
 */
function evas_deactivate() {
	// Flush rewrite rules.
	flush_rewrite_rules();
}
register_deactivation_hook( __FILE__, 'evas_deactivate' );

