/**
 * EVAS Color Palettes Module
 *
 * Handles color palette selection and application
 *
 * @package Everyone_Accessibility_Suite
 * @version 1.0.0
 */

export const ColorPalettes = {
    /**
     * Handle color palette selection
     * @param {HTMLElement} element Color button element
     * @param {Object} stateManager State manager instance
     * @param {Function} triggerEvent Event trigger callback
     */
    handleSelect(element, stateManager, triggerEvent) {
        const $color = $(element);
        const $palette = $color.closest('.evas-palette-box');
        const feature = $palette.data('feature');
        const color = $color.data('color');
        const isActive = $color.hasClass('evas-active');

        if (isActive) {
            // Deactivate - remove color
            $color.removeClass('evas-active');
            this.removeStyles(feature);
            stateManager.delete(feature);
        } else {
            // Deactivate other colors in same palette
            $palette.find('.evas-color').removeClass('evas-active');
            // Activate selected color
            $color.addClass('evas-active');
            this.applyStyles(feature, color);
            stateManager.set(feature, color);
        }

        triggerEvent(isActive ? 'color_removed' : 'color_applied', feature, color);
    },

    /**
     * Apply color styles to page elements
     * @param {string} feature Feature ID (text_colors, title_colors, background_colors)
     * @param {string} color Color name
     */
    applyStyles(feature, color) {
        this.removeStyles(feature); // Remove existing styles

        const styleId = `evas-${feature}-style`;
        let css = '';

        switch(feature) {
            case 'text_colors':
                css = `
                    body.evas-text-colors,
                    body.evas-text-colors p,
                    body.evas-text-colors li,
                    body.evas-text-colors label,
                    body.evas-text-colors input,
                    body.evas-text-colors select,
                    body.evas-text-colors textarea,
                    body.evas-text-colors legend,
                    body.evas-text-colors code,
                    body.evas-text-colors pre,
                    body.evas-text-colors dd,
                    body.evas-text-colors dt,
                    body.evas-text-colors span,
                    body.evas-text-colors blockquote {
                        color: ${color} !important;
                    }
                `;
                $('body').addClass('evas-text-colors');
                break;

            case 'title_colors':
                css = `
                    body.evas-title-colors h1,
                    body.evas-title-colors h1 span,
                    body.evas-title-colors h2,
                    body.evas-title-colors h2 span,
                    body.evas-title-colors h3,
                    body.evas-title-colors h3 span,
                    body.evas-title-colors h4,
                    body.evas-title-colors h4 span,
                    body.evas-title-colors h5,
                    body.evas-title-colors h5 span,
                    body.evas-title-colors h6,
                    body.evas-title-colors h6 span {
                        color: ${color} !important;
                    }
                `;
                $('body').addClass('evas-title-colors');
                break;

            case 'background_colors':
                css = `
                    body.evas-background-colors,
                    body.evas-background-colors p,
                    body.evas-background-colors li,
                    body.evas-background-colors label,
                    body.evas-background-colors input,
                    body.evas-background-colors select,
                    body.evas-background-colors textarea,
                    body.evas-background-colors legend,
                    body.evas-background-colors code,
                    body.evas-background-colors pre,
                    body.evas-background-colors dd,
                    body.evas-background-colors dt,
                    body.evas-background-colors span,
                    body.evas-background-colors blockquote,
                    body.evas-background-colors h1,
                    body.evas-background-colors h2,
                    body.evas-background-colors h3,
                    body.evas-background-colors h4,
                    body.evas-background-colors h5,
                    body.evas-background-colors h6 {
                        background-color: ${color} !important;
                    }
                `;
                $('body').addClass('evas-background-colors');
                break;
        }

        // Append style to head
        if (css) {
            $('<style>')
                .attr('id', styleId)
                .html(css)
                .appendTo('head');
        }
    },

    /**
     * Remove color styles from page
     * @param {string} feature Feature ID
     */
    removeStyles(feature) {
        const styleId = `evas-${feature}-style`;
        $(`#${styleId}`).remove();
        $('body').removeClass(`evas-${feature}`);
    },

    /**
     * Reset all color palettes
     */
    resetAll() {
        $('.evas-palette-box .evas-color').removeClass('evas-active');
        ['text_colors', 'title_colors', 'background_colors'].forEach(feature => {
            this.removeStyles(feature);
        });
    },

    /**
     * Restore color palette state
     * @param {Object} state Saved state
     */
    restore(state) {
        ['text_colors', 'title_colors', 'background_colors'].forEach(feature => {
            if (state[feature]) {
                const color = state[feature];
                $(`.evas-palette-box[data-feature="${feature}"] .evas-color[data-color="${color}"]`)
                    .addClass('evas-active');
                this.applyStyles(feature, color);
            }
        });
    }
};

