/**
 * EVAS Accessibility Features
 *
 * Main JavaScript file for all 40+ accessibility features.
 * Handles feature activation/deactivation, state management, and UI updates.
 *
 * @package Everyone_Accessibility_Suite
 * @version 1.0.0
 */

(function($) {
    'use strict';

    /**
     * EVAS Accessibility Features Controller
     */
    const EVAS_AF = {
        settings: window.evasAF?.settings || {},
        features: window.evasAF?.features || {},
        profiles: window.evasAF?.profiles || {},
        i18n: window.evasAF?.i18n || {},
        state: {},
        storageKey: 'evas_af_state',

        /**
         * Initialize the module
         */
        init() {
            this.loadSavedState();
            this.bindEvents();
            this.applyInitialState();
            this.createHelperElements();
        },

        /**
         * Bind event handlers
         */
        bindEvents() {
            // Toggle buttons
            $(document).on('click keypress', '.evas-toggle-box', (e) => {
                if (e.type === 'keypress' && e.which !== 13 && e.which !== 32) return;
                e.preventDefault();
                this.handleToggle(e.currentTarget);
            });

            // Spinner controls - Plus
            $(document).on('click', '.evas-plus', (e) => {
                e.stopPropagation();
                this.handleSpinnerIncrease(e.currentTarget);
            });

            // Spinner controls - Minus
            $(document).on('click', '.evas-minus', (e) => {
                e.stopPropagation();
                this.handleSpinnerDecrease(e.currentTarget);
            });

            // Profile buttons
            $(document).on('click keypress', '.evas-profile-box', (e) => {
                if (e.type === 'keypress' && e.which !== 13 && e.which !== 32) return;
                e.preventDefault();
                this.handleProfile(e.currentTarget);
            });

            // Reset button
            $(document).on('click', '#evas-reset-all', () => {
                this.resetAll();
            });

            // Color palette selection
            $(document).on('click keypress', '.evas-palette-box .evas-color', (e) => {
                if (e.type === 'keypress' && e.which !== 13 && e.which !== 32) return;
                e.preventDefault();
                e.stopPropagation();
                this.handleColorSelect(e.currentTarget);
            });

            // Keyboard navigation for spinners
            $(document).on('keydown', '.evas-spinner-box', (e) => {
                if (e.which === 38) { // Arrow up
                    e.preventDefault();
                    this.handleSpinnerIncrease($(e.currentTarget).find('.evas-plus')[0]);
                } else if (e.which === 40) { // Arrow down
                    e.preventDefault();
                    this.handleSpinnerDecrease($(e.currentTarget).find('.evas-minus')[0]);
                }
            });
        },

        /**
         * Handle toggle button click
         * @param {HTMLElement} element Toggle button element
         */
        handleToggle(element) {
            const $box = $(element);
            const feature = $box.data('feature');
            const isActive = $box.hasClass('evas-active');

            if (isActive) {
                this.deactivateFeature(feature, $box);
            } else {
                this.activateFeature(feature, $box);
            }

            this.saveState();
            this.triggerEvent(isActive ? 'deactivated' : 'activated', feature);
        },

        /**
         * Handle color palette selection
         * @param {HTMLElement} element Color button element
         */
        handleColorSelect(element) {
            const $color = $(element);
            const $palette = $color.closest('.evas-palette-box');
            const feature = $palette.data('feature');
            const color = $color.data('color');
            const isActive = $color.hasClass('evas-active');

            if (isActive) {
                // Deactivate - remove color
                $color.removeClass('evas-active');
                this.removeColorStyles(feature);
                delete this.state[feature];
            } else {
                // Deactivate other colors in same palette
                $palette.find('.evas-color').removeClass('evas-active');
                // Activate selected color
                $color.addClass('evas-active');
                this.applyColorStyles(feature, color);
                this.state[feature] = color;
            }

            this.saveState();
            this.triggerEvent(isActive ? 'color_removed' : 'color_applied', feature, color);
        },

        /**
         * Apply color styles to page elements
         * @param {string} feature Feature ID (text_colors, title_colors, background_colors)
         * @param {string} color Color name
         */
        applyColorStyles(feature, color) {
            this.removeColorStyles(feature); // Remove existing styles

            const styleId = `evas-${feature}-style`;
            let css = '';

            switch(feature) {
                case 'text_colors':
                    css = `
                        body.evas-text-colors,
                        body.evas-text-colors p,
                        body.evas-text-colors li,
                        body.evas-text-colors label,
                        body.evas-text-colors input,
                        body.evas-text-colors select,
                        body.evas-text-colors textarea,
                        body.evas-text-colors legend,
                        body.evas-text-colors code,
                        body.evas-text-colors pre,
                        body.evas-text-colors dd,
                        body.evas-text-colors dt,
                        body.evas-text-colors span,
                        body.evas-text-colors blockquote {
                            color: ${color} !important;
                        }
                    `;
                    $('body').addClass('evas-text-colors');
                    break;

                case 'title_colors':
                    css = `
                        body.evas-title-colors h1,
                        body.evas-title-colors h1 span,
                        body.evas-title-colors h2,
                        body.evas-title-colors h2 span,
                        body.evas-title-colors h3,
                        body.evas-title-colors h3 span,
                        body.evas-title-colors h4,
                        body.evas-title-colors h4 span,
                        body.evas-title-colors h5,
                        body.evas-title-colors h5 span,
                        body.evas-title-colors h6,
                        body.evas-title-colors h6 span {
                            color: ${color} !important;
                        }
                    `;
                    $('body').addClass('evas-title-colors');
                    break;

                case 'background_colors':
                    css = `
                        body.evas-background-colors,
                        body.evas-background-colors p,
                        body.evas-background-colors li,
                        body.evas-background-colors label,
                        body.evas-background-colors input,
                        body.evas-background-colors select,
                        body.evas-background-colors textarea,
                        body.evas-background-colors legend,
                        body.evas-background-colors code,
                        body.evas-background-colors pre,
                        body.evas-background-colors dd,
                        body.evas-background-colors dt,
                        body.evas-background-colors span,
                        body.evas-background-colors blockquote,
                        body.evas-background-colors h1,
                        body.evas-background-colors h2,
                        body.evas-background-colors h3,
                        body.evas-background-colors h4,
                        body.evas-background-colors h5,
                        body.evas-background-colors h6 {
                            background-color: ${color} !important;
                        }
                    `;
                    $('body').addClass('evas-background-colors');
                    break;
            }

            // Append style to head
            if (css) {
                $('<style>')
                    .attr('id', styleId)
                    .html(css)
                    .appendTo('head');
            }
        },

        /**
         * Remove color styles from page
         * @param {string} feature Feature ID
         */
        removeColorStyles(feature) {
            const styleId = `evas-${feature}-style`;
            $(`#${styleId}`).remove();
            $('body').removeClass(`evas-${feature}`);
        },

        /**
         * Activate a feature
         * @param {string} feature Feature ID
         * @param {jQuery} $box Feature box element
         */
        activateFeature(feature, $box) {
            const config = this.features[feature];
            if (!config) return;

            // Handle mutually exclusive groups
            const group = config.group;
            if (group) {
                this.deactivateGroup(group, feature);
            }

            // Update UI
            $box.addClass('evas-active').attr('aria-pressed', 'true');
            this.state[feature] = true;

            // Apply feature
            this.applyFeature(feature, true);
        },

        /**
         * Deactivate a feature
         * @param {string} feature Feature ID
         * @param {jQuery} $box Feature box element
         */
        deactivateFeature(feature, $box) {
            $box.removeClass('evas-active').attr('aria-pressed', 'false');
            this.state[feature] = false;

            // Remove feature
            this.applyFeature(feature, false);
        },

        /**
         * Apply or remove a feature
         * @param {string} feature Feature ID
         * @param {boolean} activate Whether to activate or deactivate
         */
        applyFeature(feature, activate) {
            const config = this.features[feature];
            if (!config) return;

            const bodyClass = config.class;

            if (activate) {
                if (bodyClass) {
                    $('body').addClass(bodyClass);
                }

                // Feature-specific logic
                switch (feature) {
                    case 'mute_sounds':
                        this.muteSounds(true);
                        break;
                    case 'hide_images':
                        this.hideImages(true);
                        break;
                    case 'text_magnifier':
                        this.enableTextMagnifier();
                        break;
                    case 'reading_guide':
                        this.enableReadingGuide();
                        break;
                    case 'reading_mask':
                        this.enableReadingMask();
                        break;
                    case 'cognitive_reading':
                        this.enableCognitiveReading();
                        break;
                    case 'useful_links':
                        this.showUsefulLinks();
                        break;
                    case 'stop_animations':
                        this.stopAnimations(true);
                        break;
                    case 'highlight_titles':
                        this.highlightTitles(true);
                        break;
                    case 'highlight_links':
                        this.highlightLinks(true);
                        break;
                }
            } else {
                if (bodyClass) {
                    $('body').removeClass(bodyClass);
                }

                // Feature-specific cleanup
                switch (feature) {
                    case 'mute_sounds':
                        this.muteSounds(false);
                        break;
                    case 'hide_images':
                        this.hideImages(false);
                        break;
                    case 'text_magnifier':
                        this.disableTextMagnifier();
                        break;
                    case 'reading_guide':
                        this.disableReadingGuide();
                        break;
                    case 'reading_mask':
                        this.disableReadingMask();
                        break;
                    case 'cognitive_reading':
                        this.disableCognitiveReading();
                        break;
                    case 'useful_links':
                        this.hideUsefulLinks();
                        break;
                    case 'stop_animations':
                        this.stopAnimations(false);
                        break;
                    case 'highlight_titles':
                        this.highlightTitles(false);
                        break;
                    case 'highlight_links':
                        this.highlightLinks(false);
                        break;
                }
            }
        },

        /**
         * Deactivate all features in a group except the specified one
         * @param {string} group Group ID
         * @param {string} except Feature to keep active
         */
        deactivateGroup(group, except) {
            Object.entries(this.features).forEach(([id, config]) => {
                if (config.group === group && id !== except && this.state[id]) {
                    const $box = $(`#evas-action-${id.replace(/_/g, '-')}`);
                    this.deactivateFeature(id, $box);
                }
            });
        },

        /**
         * Handle spinner increase
         * @param {HTMLElement} button Plus button element
         */
        handleSpinnerIncrease(button) {
            const $spinner = $(button).closest('.evas-spinner-box');
            const $value = $spinner.find('.evas-value');
            const step = parseInt($spinner.data('step')) || 5;
            const max = parseInt($spinner.data('max')) || 100;
            const current = parseInt($value.data('value')) || 0;
            const unit = $spinner.data('unit') || '%';
            const newValue = Math.min(current + step, max);

            this.updateSpinnerValue($spinner, newValue, unit);
        },

        /**
         * Handle spinner decrease
         * @param {HTMLElement} button Minus button element
         */
        handleSpinnerDecrease(button) {
            const $spinner = $(button).closest('.evas-spinner-box');
            const $value = $spinner.find('.evas-value');
            const step = parseInt($spinner.data('step')) || 5;
            const min = parseInt($spinner.data('min')) || -100;
            const current = parseInt($value.data('value')) || 0;
            const unit = $spinner.data('unit') || '%';
            const newValue = Math.max(current - step, min);

            this.updateSpinnerValue($spinner, newValue, unit);
        },

        /**
         * Update spinner value
         * @param {jQuery} $spinner Spinner element
         * @param {number} value New value
         * @param {string} unit Value unit
         */
        updateSpinnerValue($spinner, value, unit) {
            const $value = $spinner.find('.evas-value');
            const feature = $spinner.data('feature');
            const defaultVal = parseInt($spinner.data('default')) || 0;

            $value.data('value', value);
            
            if (value === defaultVal) {
                $value.text(this.i18n.default || 'Default');
            } else {
                $value.text(value + unit);
            }

            this.state[feature] = value;
            this.applySpinnerFeature(feature, value, unit);
            this.saveState();
        },

        /**
         * Apply spinner feature value
         * @param {string} feature Feature ID
         * @param {number} value Current value
         * @param {string} unit Value unit
         */
        applySpinnerFeature(feature, value, unit) {
            const config = this.features[feature];
            if (!config || !config.css_var) return;

            const cssVar = config.css_var;
            let cssValue;

            switch (feature) {
                case 'content_scaling':
                    cssValue = (100 + value) / 100;
                    document.documentElement.style.setProperty(cssVar, cssValue);
                    $('body').css('zoom', cssValue);
                    break;
                case 'font_sizing':
                    cssValue = `${100 + value}%`;
                    document.documentElement.style.setProperty(cssVar, cssValue);
                    $('body').css('font-size', cssValue);
                    break;
                case 'line_height':
                    cssValue = 1 + (value / 100);
                    document.documentElement.style.setProperty(cssVar, cssValue);
                    $('body').css('line-height', cssValue);
                    break;
                case 'letter_spacing':
                    cssValue = `${value}px`;
                    document.documentElement.style.setProperty(cssVar, cssValue);
                    $('body').css('letter-spacing', cssValue);
                    break;
            }
        },

        /**
         * Handle profile activation
         * @param {HTMLElement} element Profile button element
         */
        handleProfile(element) {
            const $box = $(element);
            const profile = $box.data('profile');
            let features = [];
            
            // Parse features from data attribute
            try {
                const featuresAttr = $box.attr('data-features');
                features = featuresAttr ? JSON.parse(featuresAttr) : [];
            } catch (e) {
                console.warn('EVAS: Could not parse profile features', e);
                features = [];
            }
            
            const isActive = $box.hasClass('evas-active');

            // Deactivate other profiles
            $('.evas-profile-box.evas-active').not($box).each((i, el) => {
                const otherProfile = $(el).data('profile');
                $(el).removeClass('evas-active').attr('aria-pressed', 'false');
                // Uncheck switch
                $(el).find('.evas-switch input').prop('checked', false);
                delete this.state['profile_' + otherProfile];
            });

            if (isActive) {
                // Deactivate profile
                $box.removeClass('evas-active').attr('aria-pressed', 'false');
                // Uncheck switch
                $box.find('.evas-switch input').prop('checked', false);
                delete this.state['profile_' + profile];
                
                // Deactivate profile features
                features.forEach(featureId => {
                    const $featureBox = $(`#evas-action-${featureId.replace(/_/g, '-')}`);
                    if ($featureBox.length && $featureBox.hasClass('evas-active')) {
                        this.deactivateFeature(featureId, $featureBox);
                    }
                });
            } else {
                // Reset first, then activate profile
                this.resetFeaturesOnly();
                
                // Activate profile
                $box.addClass('evas-active').attr('aria-pressed', 'true');
                // Check switch
                $box.find('.evas-switch input').prop('checked', true);
                this.state['profile_' + profile] = true;

                // Apply profile features
                features.forEach(featureId => {
                    const $featureBox = $(`#evas-action-${featureId.replace(/_/g, '-')}`);
                    if ($featureBox.length) {
                        this.activateFeature(featureId, $featureBox);
                    }
                });
            }

            this.saveState();
            this.triggerEvent('profile_applied', profile);
        },

        // =====================================
        // Feature-specific implementations
        // =====================================

        muteSounds(enable) {
            $('audio, video').each(function() {
                this.muted = enable;
            });
        },

        hideImages(enable) {
            if (enable) {
                $('img').not('.evas-keep-image, .evas-icon, .evas-panel img').addClass('evas-hidden-image');
            } else {
                $('.evas-hidden-image').removeClass('evas-hidden-image');
            }
        },

        stopAnimations(enable) {
            if (enable) {
                $('<style id="evas-stop-animations-style">')
                    .text('*, *::before, *::after { animation-duration: 0s !important; animation-delay: 0s !important; transition-duration: 0s !important; }')
                    .appendTo('head');
            } else {
                $('#evas-stop-animations-style').remove();
            }
        },

        highlightTitles(enable) {
            if (enable) {
                $('h1, h2, h3, h4, h5, h6').addClass('evas-highlighted-title');
            } else {
                $('.evas-highlighted-title').removeClass('evas-highlighted-title');
            }
        },

        highlightLinks(enable) {
            if (enable) {
                $('a').not('.evas-panel a').addClass('evas-highlighted-link');
            } else {
                $('.evas-highlighted-link').removeClass('evas-highlighted-link');
            }
        },

        enableTextMagnifier() {
            if ($('#evas-text-magnifier').length === 0) {
                $('body').append('<div id="evas-text-magnifier" aria-hidden="true"></div>');
            }
            
            $(document).on('mousemove.evas-magnifier', (e) => {
                const $el = document.elementFromPoint(e.clientX, e.clientY);
                if (!$el || $el.closest('.evas-panel, #evas-text-magnifier')) return;
                
                let tooltipContent = '';
                
                switch ($el.nodeName) {
                    case 'DIV':
                        if ($el.children.length === 0) {
                            tooltipContent = $el.innerText || '';
                        }
                        break;
                    case 'A':
                        tooltipContent = $el.innerText || $el.title || '';
                        break;
                    case 'IMG':
                        tooltipContent = $el.alt || '';
                        break;
                    case 'INPUT':
                        if ($el.type === 'button' || $el.type === 'submit') {
                            tooltipContent = $el.value || '';
                        } else if ($el.type === 'radio' || $el.type === 'checkbox') {
                            tooltipContent = $el.labels && $el.labels[0] ? $el.labels[0].innerText : '';
                        } else if ($el.type === 'text') {
                            tooltipContent = $el.value || '';
                        }
                        break;
                    case 'BUTTON':
                        tooltipContent = $el.innerText || $el.title || '';
                        break;
                    case 'SELECT':
                        tooltipContent = $el.options && $el.selectedIndex >= 0 ? $el.options[$el.selectedIndex].text : '';
                        break;
                    case 'TEXTAREA':
                        tooltipContent = $el.value || '';
                        break;
                    case 'P':
                    case 'SPAN':
                    case 'LI':
                    case 'TD':
                    case 'H1':
                    case 'H2':
                    case 'H3':
                    case 'H4':
                    case 'H5':
                    case 'H6':
                        tooltipContent = $el.innerText || '';
                        break;
                    default:
                        tooltipContent = $el.innerText || '';
                        break;
                }
                
                // Hide if no content
                tooltipContent = tooltipContent.trim().substring(0, 150);
                if (tooltipContent === '') {
                    $('#evas-text-magnifier').css('display', 'none');
                    return;
                }
                
                // Show and position tooltip
                const shift = 15;
                const maxWidth = window.innerWidth;
                const $magnifier = $('#evas-text-magnifier');
                
                $magnifier.html(tooltipContent).css({
                    top: (e.clientY + shift) + 'px',
                    display: 'block'
                });
                
                // Position: stick to right if cursor on right 50%
                if (e.clientX > maxWidth / 2) {
                    $magnifier.css({
                        left: 'auto',
                        right: (maxWidth - e.clientX + shift) + 'px'
                    });
                } else {
                    $magnifier.css({
                        left: (e.clientX + shift) + 'px',
                        right: 'auto'
                    });
                }
            });
        },

        disableTextMagnifier() {
            $(document).off('mousemove.evas-magnifier');
            $('#evas-text-magnifier').remove();
        },

        enableReadingGuide() {
            if ($('#evas-reading-guide').length === 0) {
                $('body').append('<div id="evas-reading-guide" aria-hidden="true"></div>');
            }
            
            $(document).on('mousemove.evas-guide', (e) => {
                $('#evas-reading-guide').css('top', e.pageY + 'px');
            });
        },

        disableReadingGuide() {
            $(document).off('mousemove.evas-guide');
            $('#evas-reading-guide').remove();
        },

        enableReadingMask() {
            if ($('#evas-reading-mask-top').length === 0) {
                $('body').append(`
                    <div id="evas-reading-mask-top" aria-hidden="true"></div>
                    <div id="evas-reading-mask-bottom" aria-hidden="true"></div>
                `);
            }
            
            const maskHeight = 100;
            
            $(document).on('mousemove.evas-mask', (e) => {
                const y = e.pageY;
                $('#evas-reading-mask-top').css('height', Math.max(0, y - maskHeight) + 'px');
                $('#evas-reading-mask-bottom').css({
                    top: (y + maskHeight) + 'px',
                    height: 'calc(100% - ' + (y + maskHeight) + 'px)'
                });
            });
        },

        disableReadingMask() {
            $(document).off('mousemove.evas-mask');
            $('#evas-reading-mask-top, #evas-reading-mask-bottom').remove();
        },

        enableCognitiveReading() {
            $('p, li, td, span, div').not('.evas-panel *').each(function() {
                const $el = $(this);
                if ($el.children().length === 0 && $el.text().trim()) {
                    const text = $el.text();
                    const highlighted = text.replace(/\b(\w)(\w*)/g, '<strong class="evas-cognitive-bold">$1</strong>$2');
                    $el.data('original-text', text).html(highlighted);
                }
            });
        },

        disableCognitiveReading() {
            $('[data-original-text]').each(function() {
                $(this).text($(this).data('original-text')).removeData('original-text');
            });
        },

        showUsefulLinks() {
            if ($('#evas-useful-links').length) return;

            const links = [];
            $('a[href]').each(function() {
                const $a = $(this);
                const href = $a.attr('href');
                const text = $a.text().trim();
                
                if (text && href && !href.startsWith('#') && !href.startsWith('javascript')) {
                    links.push({ href, text: text.substring(0, 50) });
                }
            });

            // Limit to 20 links
            const displayLinks = links.slice(0, 20);

            const $panel = $(`
                <div id="evas-useful-links" role="dialog" aria-label="${this.i18n.useful_links || 'Useful Links'}">
                    <div class="evas-useful-links-header">
                        <h3>${this.i18n.useful_links || 'Useful Links'}</h3>
                        <button class="evas-close-useful-links" aria-label="Close">&times;</button>
                    </div>
                    <ul class="evas-useful-links-list">
                        ${displayLinks.map(l => `<li><a href="${l.href}">${l.text}</a></li>`).join('')}
                    </ul>
                </div>
            `);

            $('body').append($panel);

            $panel.find('.evas-close-useful-links').on('click', () => {
                this.hideUsefulLinks();
                const $box = $('#evas-action-useful-links');
                this.deactivateFeature('useful_links', $box);
                this.saveState();
            });
        },

        hideUsefulLinks() {
            $('#evas-useful-links').remove();
        },

        // =====================================
        // Helper elements
        // =====================================

        createHelperElements() {
            // Create CSS variables container
            if (!document.getElementById('evas-dynamic-styles')) {
                $('<style id="evas-dynamic-styles"></style>').appendTo('head');
            }
        },

        // =====================================
        // State management
        // =====================================

        saveState() {
            try {
                localStorage.setItem(this.storageKey, JSON.stringify(this.state));
            } catch (e) {
                console.warn('EVAS: Could not save state to localStorage', e);
            }
        },

        loadSavedState() {
            try {
                const saved = localStorage.getItem(this.storageKey);
                if (saved) {
                    this.state = JSON.parse(saved);
                }
            } catch (e) {
                console.warn('EVAS: Could not load state from localStorage', e);
                this.state = {};
            }
        },

        applyInitialState() {
            Object.entries(this.state).forEach(([key, value]) => {
                if (!value) return;

                // Check if it's a profile
                if (key.startsWith('profile_')) {
                    const profileId = key.replace('profile_', '');
                    const $box = $(`#evas-profile-${profileId.replace(/_/g, '-')}`);
                    if ($box.length) {
                        $box.addClass('evas-active').attr('aria-pressed', 'true');
                        
                        // Apply profile features
                        try {
                            const featuresAttr = $box.attr('data-features');
                            const features = featuresAttr ? JSON.parse(featuresAttr) : [];
                            features.forEach(featureId => {
                                const $featureBox = $(`#evas-action-${featureId.replace(/_/g, '-')}`);
                                if ($featureBox.length) {
                                    this.activateFeature(featureId, $featureBox);
                                }
                            });
                        } catch (e) {
                            console.warn('EVAS: Could not restore profile features', e);
                        }
                    }
                    return;
                }

                // Check if it's a spinner value
                if (typeof value === 'number') {
                    const $spinner = $(`#evas-action-${key.replace(/_/g, '-')}`);
                    if ($spinner.length && $spinner.hasClass('evas-spinner-box')) {
                        const unit = $spinner.data('unit') || '%';
                        this.updateSpinnerValue($spinner, value, unit);
                    }
                    return;
                }

                // Check if it's a color palette
                if (['text_colors', 'title_colors', 'background_colors'].includes(key)) {
                    const $colorBox = $(`.evas-palette-box[data-feature="${key}"] .evas-color[data-color="${value}"]`);
                    if ($colorBox.length) {
                        $colorBox.addClass('evas-active');
                        this.applyColorStyles(key, value);
                    }
                    return;
                }

                // Toggle feature
                const $box = $(`#evas-action-${key.replace(/_/g, '-')}`);
                if ($box.length) {
                    this.activateFeature(key, $box);
                }
            });
        },

        resetFeaturesOnly() {
            // Reset toggles (not spinners)
            $('.evas-toggle-box.evas-active').each((i, el) => {
                const feature = $(el).data('feature');
                this.deactivateFeature(feature, $(el));
                // Clear state for this feature
                delete this.state[feature];
            });
        },

        resetAll() {
            if (!confirm(this.i18n.reset_confirm || 'Reset all settings?')) {
                return;
            }

            // Reset toggles
            $('.evas-toggle-box.evas-active, .evas-profile-box.evas-active').each((i, el) => {
                const feature = $(el).data('feature') || $(el).data('profile');
                const $box = $(el);
                
                $box.removeClass('evas-active').attr('aria-pressed', 'false');
                
                if (feature) {
                    this.applyFeature(feature, false);
                }
            });

            // Reset spinners
            $('.evas-spinner-box').each((i, el) => {
                const $spinner = $(el);
                const defaultVal = parseInt($spinner.data('default')) || 0;
                const unit = $spinner.data('unit') || '%';
                this.updateSpinnerValue($spinner, defaultVal, unit);
            });

            // Reset color palettes
            $('.evas-palette-box .evas-color').removeClass('evas-active');
            ['text_colors', 'title_colors', 'background_colors'].forEach(feature => {
                this.removeColorStyles(feature);
            });

            // Clear state
            this.state = {};
            this.saveState();

            // Remove body classes
            $('body').removeClass(function(index, className) {
                return (className.match(/\bevas-[^\s]+/g) || []).join(' ');
            });

            // Remove dynamic elements
            $('#evas-text-magnifier, #evas-reading-guide, .evas-reading-mask, #evas-useful-links, #evas-stop-animations-style').remove();
            $('.evas-highlighted-title, .evas-highlighted-link, .evas-hidden-image').removeClass('evas-highlighted-title evas-highlighted-link evas-hidden-image');

            this.triggerEvent('reset', 'all');
        },

        /**
         * Trigger custom event
         * @param {string} action Action name
         * @param {string} feature Feature ID
         */
        triggerEvent(action, feature) {
            $(document).trigger('evas:feature:' + action, { feature });
            $(document).trigger('evas:feature', { action, feature });
        }
    };

    // Initialize when DOM is ready
    $(document).ready(() => EVAS_AF.init());

    // Expose for external access
    window.EVAS_AF = EVAS_AF;

})(jQuery);

