/**
 * EVAS Accessibility Features - Modular Version
 *
 * Main entry point that imports and initializes all modules
 *
 * @package Everyone_Accessibility_Suite
 * @version 1.0.0
 */

import { StateManager } from './modules/state.js';
import { ColorPalettes } from './modules/colors.js';

(function($) {
    'use strict';

    /**
     * EVAS Accessibility Features Controller
     */
    const EVAS_AF = {
        settings: window.evasAF?.settings || {},
        features: window.evasAF?.features || {},
        profiles: window.evasAF?.profiles || {},
        i18n: window.evasAF?.i18n || {},
        stateManager: StateManager,

        /**
         * Initialize the module
         */
        init() {
            this.stateManager.load();
            this.bindEvents();
            this.applyInitialState();
            this.createHelperElements();
        },

        /**
         * Bind event handlers
         */
        bindEvents() {
            // Toggle buttons
            $(document).on('click keypress', '.evas-toggle-box', (e) => {
                if (e.type === 'keypress' && e.which !== 13 && e.which !== 32) return;
                e.preventDefault();
                this.handleToggle(e.currentTarget);
            });

            // Spinner controls - Plus
            $(document).on('click', '.evas-plus', (e) => {
                e.stopPropagation();
                this.handleSpinnerIncrease(e.currentTarget);
            });

            // Spinner controls - Minus
            $(document).on('click', '.evas-minus', (e) => {
                e.stopPropagation();
                this.handleSpinnerDecrease(e.currentTarget);
            });

            // Profile buttons
            $(document).on('click keypress', '.evas-profile-box', (e) => {
                if (e.type === 'keypress' && e.which !== 13 && e.which !== 32) return;
                e.preventDefault();
                this.handleProfile(e.currentTarget);
            });

            // Reset button
            $(document).on('click', '#evas-reset-all', () => {
                this.resetAll();
            });

            // Color palette selection
            $(document).on('click keypress', '.evas-palette-box .evas-color', (e) => {
                if (e.type === 'keypress' && e.which !== 13 && e.which !== 32) return;
                e.preventDefault();
                e.stopPropagation();
                ColorPalettes.handleSelect(e.currentTarget, this.stateManager, this.triggerEvent.bind(this));
            });

            // Keyboard navigation for spinners
            $(document).on('keydown', '.evas-spinner-box', (e) => {
                if (e.which === 38) { // Arrow up
                    e.preventDefault();
                    this.handleSpinnerIncrease($(e.currentTarget).find('.evas-plus')[0]);
                } else if (e.which === 40) { // Arrow down
                    e.preventDefault();
                    this.handleSpinnerDecrease($(e.currentTarget).find('.evas-minus')[0]);
                }
            });
        },

        // ... (остальные методы из оригинального файла)
        // Для краткости показываю только структуру

        triggerEvent(eventName, ...args) {
            $(document).trigger(`evas:${eventName}`, args);
        }
    };

    // Initialize when DOM is ready
    $(document).ready(() => {
        EVAS_AF.init();
    });

    // Expose to global scope
    window.EVAS_AF = EVAS_AF;

})(jQuery);

