<?php
/*
 * Plugin Name: Evergreen Payments Northwest Gateway For WooCommerce
 * Plugin URI: https://www.evergreenpaymentsnorthwest.com/evergreen-for-woocommerce
 * Description: A payment gateway for Evergreen Payments Northwest. An Evergreen Payments Northwest account and a server with cURL, SSL support, and a valid SSL certificate is required (for security reasons) for this gateway to function. Requires WC 3.3+
 * Version: 1.1.0
 * Author: Evergreen Payments Northwest
 * Author URI: https://www.evergreenpaymentsnorthwest.com
 * Text Domain: evergreen-payments-northwest-gateway-wc
 * Domain Path: /languages
 * WC requires at least: 3.3
 * WC tested up to: 10.3
 * Requires PHP: 7.4
 * License: GPLv3
 * License URI: https://www.gnu.org/licenses/gpl-3.0.html
 * Requires Plugins: woocommerce
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

define( 'WCEG_VERSION', '1.1.0' );
define( 'WCEG_TEMPLATE_PATH', untrailingslashit( plugin_dir_path( __FILE__ ) ) . '/templates/' );
define( 'WCEG_PLUGIN_URL', untrailingslashit( plugins_url( basename( plugin_dir_path( __FILE__ ) ), basename( __FILE__ ) ) ) );
define( 'WCEG_MAIN_FILE', __FILE__ );

/**
 * Main Evergreen Payments Northwest class which sets the gateway up for us
 */
class WCEG_Main {

	/**
	 * @var WCEG_Main Singleton The reference the *Singleton* instance of this class
	 */
	private static $instance;

	/**
	 * Returns the *Singleton* instance of this class.
	 *
	 * @return WCEG_Main Singleton The *Singleton* instance.
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Flag to indicate whether or not we need to load code for / support subscriptions.
	 *
	 * @var bool
	 */
	private $subscription_support_enabled = false;

	/**
	 * Flag to indicate whether or not we need to load support for pre-orders.
	 *
	 * @since 3.0.3
	 *
	 * @var bool
	 */
	private $pre_order_enabled = false;

	/**
	 * Notices (array)
	 * @var array
	 */
	public $notices = array();

	/**
	 * Constructor
	 */
	public function __construct() {

		add_action( 'before_woocommerce_init', function() {
			// Declaring HPOS feature compatibility
			if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
				\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', __FILE__, true );
			}
			// Declaring cart and checkout blocks incompatibility
			if ( class_exists( '\Automattic\WooCommerce\Utilities\FeaturesUtil' ) ) {
				\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'cart_checkout_blocks', __FILE__, false );
			}
		} );

		add_action( 'admin_init', array( $this, 'check_environment' ) );
		add_action( 'admin_notices', array( $this, 'admin_notices' ), 15 );
		add_action( 'plugins_loaded', array( $this, 'init_environment' ) );
	}

	public function settings_url() {
		return admin_url( 'admin.php?page=wc-settings&tab=checkout&section=evergreen' );
	}

	/**
	 * Add relevant links to plugins page
	 * @param  array $links
	 * @return array
	 */
	public function plugin_action_links( $links ) {
		$plugin_links = array(
			'<a href="' . esc_url( $this->settings_url() ) . '">' . __( 'Settings', 'evergreen-payments-northwest-gateway-wc' ) . '</a>',
			'<a href="https://wpgateways.com/support/">' . __( 'Support', 'evergreen-payments-northwest-gateway-wc' ) . '</a>',
		);
		return array_merge( $plugin_links, $links );
	}

	/**
	 * Initialize the gateway. Called very early - in the context of the plugins_loaded action
	 *
	 * @since 1.0.0
	 */
	public function init_gateways() {
		if ( class_exists( 'WC_Subscriptions' ) && function_exists( 'wcs_create_renewal_order' ) ) {
			$this->subscription_support_enabled = true;
		}

		if ( class_exists( 'WC_Pre_Orders_Order' ) ) {
			$this->pre_order_enabled = true;
		}

		if ( ! class_exists( 'WC_Payment_Gateway' ) ) {
			return;
		}

		// Includes
		if ( is_admin() ) {
			require_once( dirname( __FILE__ ) . '/includes/class-wceg-privacy.php' );
		}

		include_once( dirname( __FILE__ ) . '/includes/class-wceg-gateway.php' );

		add_filter( 'woocommerce_payment_gateways', array( $this, 'add_gateways' ) );

		$load_addons = (
			$this->subscription_support_enabled
			||
			$this->pre_order_enabled
		);

		if ( $load_addons ) {
			require_once( dirname( __FILE__ ) . '/includes/class-wceg-gateway-addons.php' );
		}

	}

	/**
	 * Add the gateways to WooCommerce
	 *
	 * @since 1.0.0
	 */
	public function add_gateways( $methods ) {

		if ( $this->subscription_support_enabled || $this->pre_order_enabled ) {
			$methods[] = 'WCEG_Gateway_Addons';
		} else {
			$methods[] = 'WCEG_Gateway';
		}

		return $methods;
	}

	/**
	 * Init localisations and files
	 */
	public function init_environment() {

		// Init the gateway itself
		$this->init_gateways();

		// required files
		require_once( 'includes/class-wceg-gateway-logger.php' );

		add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), array( $this, 'plugin_action_links' ), 11 );
		add_action( 'init', array( $this, 'load_plugin_textdomain' ) );

		if( version_compare( WC_VERSION, '8.4.0', '<' ) ) {
			add_action( 'woocommerce_order_status_processing', array( $this, 'capture_payment' ), 10, 2 );
			add_action( 'woocommerce_order_status_completed', array( $this, 'capture_payment' ), 10, 2 );
		} else {
			add_action( 'woocommerce_order_status_processing', array( $this, 'capture_payment' ), 10, 3 );
			add_action( 'woocommerce_order_status_completed', array( $this, 'capture_payment' ), 10, 3 );
		}

		add_action( 'woocommerce_order_status_cancelled', array( $this, 'cancel_payment' ) );
		add_action( 'woocommerce_order_status_refunded', array( $this, 'cancel_payment' ) );

	}

	/**
	 * Allow this class and other classes to add slug keyed notices (to avoid duplication)
	 */
	public function add_admin_notice( $slug, $class, $message ) {
		$this->notices[ $slug ] = array(
			'class'   => $class,
			'message' => $message,
		);
	}

	/**
	 * The backup sanity check, in case the plugin is activated in a weird way,
	 * or the environment changes after activation. Also handles upgrade routines.
	 */
	public function check_environment() {
		if ( ! defined( 'IFRAME_REQUEST' ) && ( WCEG_VERSION !== get_option( 'wceg_version', '1.0.0' ) ) ) {
			$this->install();

			do_action( 'wceg_updated' );
		}

		$environment_warning = self::get_environment_warning();

		if ( $environment_warning && is_plugin_active( plugin_basename( __FILE__ ) ) ) {
			$this->add_admin_notice( 'bad_environment', 'error', $environment_warning );
		}

		if ( ! class_exists( 'WCEG_Gateway' ) ) {
			return;
		}

		$gateway = new WCEG_Gateway();

		if ( ! $gateway->private_key && ! ( isset( $_GET['page'], $_GET['section'] ) && 'wc-settings' === $_GET['page'] && 'evergreen' === $_GET['section'] ) ) {
			$this->add_admin_notice( 'prompt_connect', 'notice notice-warning', sprintf( __( 'Evergreen Payments Northwest is almost ready. To get started, <a href="%s">set your Evergreen Payments Northwest account keys</a>.', 'evergreen-payments-northwest-gateway-wc' ), esc_url( $this->settings_url() ) ) );
		}
	}

	/**
	 * Updates the plugin version in db
	 *
	 * @since 3.1.0
	 * @version 3.1.0
	 * @return bool
	 */
	private static function _update_plugin_version() {
		delete_option( 'wceg_version' );
		update_option( 'wceg_version', WCEG_VERSION );

		return true;
	}

	/**
	 * Handles upgrade routines.
	 *
	 * @since 3.1.0
	 * @version 3.1.0
	 */
	public function install() {
		if ( ! defined( 'WCEG_INSTALLING' ) ) {
			define( 'WCEG_INSTALLING', true );
		}

		$this->_update_plugin_version();
	}

	/**
	 * Checks the environment for compatibility problems. Returns a string with the first incompatibility
	 * found or false if the environment has no problems.
	 */
	static function get_environment_warning() {

		if ( ! defined( 'WC_VERSION' ) ) {
			return __( 'WooCommerce Evergreen Payments Northwest extension requires WooCommerce to be activated to work.', 'evergreen-payments-northwest-gateway-wc' );
		}

		if ( ! function_exists( 'curl_init' ) ) {
			return __( 'WooCommerce Evergreen Payments Northwest - cURL is not installed.', 'evergreen-payments-northwest-gateway-wc' );
		}

		return false;
	}

	/**
	 * Display any notices we've collected thus far (e.g. for connection, disconnection)
	 */
	public function admin_notices() {

		foreach ( (array) $this->notices as $notice ) {
			echo "<div class='" . esc_attr( $notice['class'] ) . "'><p>";
			echo wp_kses( $notice['message'], array( 'a' => array( 'href' => array() ), 'strong' => array(), 'br' => array() ) );
			echo '</p></div>';
		}
	}

	public function load_plugin_textdomain() {
		load_plugin_textdomain( 'evergreen-payments-northwest-gateway-wc', false, plugin_basename( dirname( __FILE__ ) ) . '/languages' );
	}

	/**
	 * Capture payment when the order is changed from on-hold to complete or processing
	 *
	 * @param int $order_id
	 * @param $order
	 * @param $status_transition
	 */
	public function capture_payment( $order_id, $order, $status_transition = array() ) {

		if ( $order->get_payment_method() == 'evergreen' && apply_filters( 'wceg_capture_payment', true, $order ) ) {
			$charge   = $order->get_meta( '_evergreen_charge_id' );
			$captured = $order->get_meta( '_evergreen_charge_captured' );

			$gateway = new WCEG_Gateway();

			if ( apply_filters( 'wceg_capture_on_status_change', $gateway->capture_on_status_change, $order, $status_transition ) && $charge && $captured == 'no' ) {

				$gateway->log( "Info: Beginning capture payment for order $order_id for the amount of {$order->get_total()}" );

				$order_total = $order->get_total();

				if ( 0 < $order->get_total_refunded() ) {
					$order_total = $order_total - $order->get_total_refunded();
				}

				$args = array(
					'amount'		=> $order_total,
					'transactionid'	=> $order->get_transaction_id(),
					'type' 			=> 'capture',
					'email' 		=> $order->get_billing_email(),
					'currency'		=> $gateway->get_payment_currency( $order_id ),
				);
				$args = apply_filters( 'wceg_capture_request_args', $args, $order );

				$response = $gateway->evergreen_request( $args );

				if ( is_wp_error( $response ) ) {
					if( $order->get_meta( '_evergreen_capture_failed' ) == 'yes' ) {
						$order->add_order_note( sprintf( __( "<strong>Unable to capture charge!</strong> Please <strong>DO NOT FULFIL THE ORDER</strong> if the amount cannot be captured in the gateway account manually or by changing the status. In that case, set status to Failed manually and do not fulfil. \n\nEvergreen Payments Northwest failure reason: %s \n\n", 'evergreen-payments-northwest-gateway-wc' ), $response->get_error_message() ) );
					} else {
						$order->update_status( 'failed', sprintf( __( "<strong>Unable to capture charge!</strong> The order status is set to <strong>Failed</strong> the first time to draw your attention. If the next attempt fails, your intended order status will still take place. \n\nPlease double-check that the amount is captured in the gateway account before fulfilling the order. \n\nEvergreen Payments Northwest failure reason: %s \n\n", 'evergreen-payments-northwest-gateway-wc' ), $response->get_error_message() ) );
						$order->update_meta_data( '_evergreen_capture_failed', 'yes' );
						$order->save();
					}
				} else {
					$complete_message = sprintf( __( 'Evergreen Payments Northwest charge captured for %s (Charge ID: %s).', 'wc-evergreen-payments-northwest-gateway-wc' ), wc_price( $args['amount'], array( 'currency' => $order->get_currency() ) ), $response['transactionid'] );
					$order->add_order_note( $complete_message );
					$gateway->log( 'Success: ' . wp_strip_all_tags( $complete_message ) );

					$order->update_meta_data( '_evergreen_charge_captured', 'yes' );
					$order->update_meta_data( 'Evergreen Payments Northwest Payment ID', $response['transactionid'] );

					$order->set_transaction_id( $response['transactionid'] );
					$order->save();
				}
			}
		}
	}

	/**
	 * Cancel pre-auth on refund/cancellation
	 *
	 * @param  int $order_id
	 */
	public function cancel_payment( $order_id ) {
		$order = wc_get_order( $order_id );

		if ( $order->get_payment_method() == 'evergreen' ) {
			$charge	  = $order->get_meta( '_evergreen_charge_id' ) ? $order->get_meta( '_evergreen_charge_id' ) : $order->get_transaction_id();
			$captured = $order->get_meta( '_evergreen_charge_captured' );

			if ( $charge && $captured != 'yes' ) {

				$gateway = new WCEG_Gateway();

				$gateway->log( "Info: Beginning cancel payment for order $order_id for the amount of {$order->get_total()}" );

				$args = array(
					'amount'		=> $order->get_total(),
					'transactionid'	=> $order->get_transaction_id(),
					'type' 			=> 'void',
					'email' 		=> $order->get_billing_email(),
					'currency'		=> $gateway->get_payment_currency( $order_id ),
				);
				$args = apply_filters( 'wceg_cancel_request_args', $args, $order );

				$response = $gateway->evergreen_request( $args );

				if ( is_wp_error( $response ) ) {
					$order->add_order_note( __( 'Unable to refund charge!', 'evergreen-payments-northwest-gateway-wc' ) . ' ' . $response->get_error_message() );
				} else {
					$cancel_message = sprintf( __( 'Evergreen Payments Northwest charge refunded (Charge ID: %s).', 'evergreen-payments-northwest-gateway-wc' ), $response['transactionid'] );
					$order->add_order_note( $cancel_message );
					$gateway->log( "Success: $cancel_message" );

					$order->delete_meta_data( '_evergreen_charge_captured' );
					$order->delete_meta_data( '_evergreen_charge_id' );
					$order->save();
				}
			}
		}
	}

}
$GLOBALS['wceg'] = WCEG_Main::get_instance();