<?php
namespace Etn_Surecart_Addon\Integrations\SureCart;

defined( 'ABSPATH' ) || exit;

use Etn\Utils\Helper;
use Eventin\Order\OrderModel;
use Stripe\Charge;
use SureCart\Models\ApiToken;
use WP_REST_Request;

/**
 * SureCart Integration Handler (moved to addon)
 *
 * Listens to surecart/checkout_confirmed and completes Eventin orders.
 * 
 * @package Etn_Surecart_Addon/Integrations/SureCart
 *
 * @since 1.0.0
 */
class SureCartWebhook {
    /**
     * Register WordPress action hooks
     *
     * @return void
     *
     * @since 1.0.0
     */
    public function register_hooks(): void {
        add_action('surecart/checkout_confirmed', [$this, 'handle_checkout_confirmed'], 10, 2);

        add_action('rest_api_init', function(){
            register_rest_route('surecart/v1', '/webhook', array(
                'methods' => 'POST',
                'callback' => [$this, 'handle_order_status_change'],
                'permission_callback' => [$this, 'permission_handle_order_status_change']
            ));
        });
    }

    /**
     * Permission callback for webhook endpoint
     *
     * @param WP_REST_Request $request
     * @return bool
     */
    public function permission_handle_order_status_change(WP_REST_Request $request) {
        $timestamp  = $request->get_header('x_webhook_timestamp');
        $signature  = $request->get_header('x_webhook_signature');
        $user_agent = $request->get_header('user_agent');

        // 1. Basic sanity check
        if (empty($timestamp) || empty($signature)) {
            return false;
        }

        // 2. Protection against replay attacks (5 min window)
        if (abs(time() - intval($timestamp)) > 300) {
            return false;
        }

        // 3. Check signature
        if (!$signature) {
            return false;
        }

        // 4. Check user agent
        if (!$user_agent || stripos($user_agent, 'SureCart') === false) {
            return false;
        }

        return true;
    }

    /**
     * Extract Eventin order ID from metadata
     *
     * @param mixed $metadata Metadata object or array from SureCart
     * @return int Order ID or 0 if not found
     */
    private function extract_eventin_order_id($metadata) {
        if (is_object($metadata)) {
            return isset($metadata->eventin_order_id) ? intval($metadata->eventin_order_id) : 0;
        } elseif (is_array($metadata)) {
            return isset($metadata['eventin_order_id']) ? intval($metadata['eventin_order_id']) : 0;
        }

        return 0;
    }

    /**
     * Complete an Eventin order
     *
     * @param int $eventin_order_id
     */
    private function complete_eventin_order($eventin_order_id) {
        $order = new OrderModel($eventin_order_id);

        if (!$order->id) {
            return;
        }

        if ('completed' === $order->status) {
            return;
        }

        $order->update([
            'status' => 'completed'
        ]);

        do_action('eventin_order_completed', $order);

        $order->send_email();
    }

    /**
     * Handle checkout_confirmed event
     *
     * @param object $checkout SureCart checkout object
     * @param mixed $request
     * @return void
     *
     * @since 1.0.0
     */
    public function handle_checkout_confirmed($checkout, $request = null) {
        $metadata = (is_object($checkout) && isset($checkout->metadata)) ? $checkout->metadata : null;
        $eventin_order_id = $this->extract_eventin_order_id($metadata);

        if (empty($eventin_order_id)) {
            return;
        }

        if( isset($checkout->status) && $checkout->status !== 'paid'){
            return;
        }

        $this->complete_eventin_order($eventin_order_id);
    }

    /**
     * Handle order status change
     *
     * @param object $request
     * @return void
     *
     * @since 1.0.0
     */
    public function handle_order_status_change($request){
        // Get JSON data
        $order = $request->get_json_params();

        $charge_id = !empty($order['data']['object']['charge']) ? sanitize_text_field($order['data']['object']['charge']) : null;

        $checkout_id = !empty($order['data']['object']['checkout']) ? sanitize_text_field($order['data']['object']['checkout']) : null;

        if(empty($checkout_id) && !empty($charge_id)){
            $checkout_id = $this->get_checkout_id_by_order_id($charge_id);
        }

        $status = '';
        $attendee_status = '';
        $order_type = !empty($order['type']) ? sanitize_text_field($order['type']) : '';
        $order_status = !empty($order['data']['object']['status']) ? sanitize_text_field($order['data']['object']['status']) : '';

        if($order_type == 'order.paid' && $order_status == 'paid'){
            $status = 'completed';
            $attendee_status = 'success';
        }
        elseif($order_type == 'refund.created' && $order_status == 'succeeded'){
            $status = 'refunded';
            $attendee_status = 'failed';
        }
        elseif($order_type == 'purchase.revoked' || $order_type == 'order.voided'){
            $status = 'failed';
            $attendee_status = 'failed';
        }

        if($status){
            $this->update_eventin_order_status($checkout_id,$status);
            $this->change_attendee_payment_status_on_order_status_update($checkout_id,$attendee_status);
        }

        return ['success' => true, 'message' => esc_html__( 'Webhook received', 'eventin-addon-for-surecart' )];
    }

    /**
     * Get checkout ID by order ID
     *
     * @param string $order_id
     * @return string
     *
     * @since 1.0.0
     */
    public function get_checkout_id_by_order_id($charge_id){
        // Sanitize charge_id for safe URL usage
        $charge_id = sanitize_text_field($charge_id);

        $authToken = ApiToken::get();

        $response = wp_remote_get(
            "https://api.surecart.com/v1/charges/{$charge_id}",
            [
                'headers' => [
                    'Authorization' => "Bearer {$authToken}"
                ]
            ]
        );

        $checkout_id = null;

        if ( is_array($response) && ! is_wp_error($response) ) {
            $body = !empty($response['body'])?(array)json_decode( $response['body'] ):[];

            $checkout_id = !empty($body['checkout']) ? sanitize_text_field($body['checkout']) : null;
        }

        return $checkout_id;
    }

    /**
     * Update Eventin order status
     *
     * @param string $checkout_id
     * @param string $status
     * @return void
     *
     * @since 1.0.0
     */
    public function update_eventin_order_status($checkout_id,$status){
		// Sanitize inputs to prevent SQL injection
		$checkout_id = sanitize_text_field($checkout_id);

		// Whitelist allowed statuses
		$allowed_statuses = ['completed', 'refunded', 'failed'];
		if (!in_array($status, $allowed_statuses, true)) {
			return;
		}

		// phpcs:disable WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Required to find order by payment_id
		$post_id = get_posts([
            'post_type'      => 'etn-order',
            'posts_per_page' => 1,
            'post_status'    => 'any',
            'fields'         => 'ids',
            'meta_query'     => [
                [
                    'key'   => 'payment_id',
                    'value' => $checkout_id,
                ],
            ],
        ]);
		// phpcs:enable WordPress.DB.SlowDBQuery.slow_db_query_meta_query

        $eventin_order_id = !empty($post_id) ? $post_id[0] : null;

		if ( ! $eventin_order_id ) {
			return;
		}

		$event_order      = new OrderModel( $eventin_order_id );

		if ( ! $event_order ) {
			return;
		}
        
		switch ($status) {
			case 'completed':
				if ( 'completed' === $event_order->status ) {
				    return;
				}
				$event_order->update([
					'status' => 'completed'
				]);

				do_action( 'eventin_order_completed', $event_order );

				$event_order->send_email();
				break;

			case "refunded":
				$event_order->update([
					'status' => 'refunded',
				]);

				do_action( 'eventin_order_refund', $event_order );
				break;
			default:
                
                if('refunded' === $event_order->status){
                    return;
                }
				$event_order->update([
					'status' => 'failed',
				]);

				do_action( 'eventin_order_failed', $event_order );
		}
    }

    /**
     * Change attendee payment status
     *
     * @param [type] $checkout_id
     * @param [type] $new_order_status
     * @return void
     *
     * @since 1.0.0
     */
    public function change_attendee_payment_status_on_order_status_update(  $checkout_id, $new_order_status ) {
        // phpcs:disable WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- Required to find order by payment_id
        $post_id = get_posts([
            'post_type'      => 'etn-order',
            'posts_per_page' => 1,
            'post_status'    => 'any',
            'fields'         => 'ids',
            'meta_query'     => [
                [
                    'key'   => 'payment_id',
                    'value' => $checkout_id,
                ],
            ],
        ]);
        // phpcs:enable WordPress.DB.SlowDBQuery.slow_db_query_meta_query

        $eventin_order_id = !empty($post_id) ? $post_id[0] : null;

		if ( ! $eventin_order_id ) {
			return;
		}

        $order_attendees = Helper::get_attendee_by_order_id( $eventin_order_id );

        if( is_array( $order_attendees ) && !empty( $order_attendees )){
            foreach($order_attendees as $attendee_id){
                Helper::update_attendee_payment_status($attendee_id, $new_order_status);
            }
        }
    }
}
