<?php
namespace Etn_Surecart_Addon\Integrations\SureCart;

defined( 'ABSPATH' ) || exit;

use Eventin\Order\OrderModel;
use SureCart\Models\ApiToken;
use WP_Error;

/**
 * SureCart class
 *
 * @package Etn_Surecart_Addon/Integrations/SureCart
 *
 * @since 1.0.0
 */
class SureCart {
    /**
     * Base URL for the SureCart API
     *
     * @var string
     *
     * @since 1.0.0
     */
    private   $base_url = 'https://api.surecart.com/v1';
    /**
     * API key for the SureCart API
     *
     * @var string
     *
     * @since 1.0.0
     */
    protected $api_key;

    public function __construct() {
        $this->api_key = ApiToken::get();
    }

    /**
     * Create a payment checkout in SureCart
     *
     * @param  OrderModel $order The order object
     * @return array|false|WP_Error The response body on success, false on failure, or WP_Error on error
     *
     * @since 1.0.0
     */
    public function create_payment( OrderModel $order ) {
        $validation = $this->validate_api_key();
        if ( is_wp_error( $validation ) ) {
            return $validation;
        }

        $customer   = $order->get_customer();
        $line_items = $this->get_line_items( $order );

        if ( empty( $line_items ) ) {
            return new WP_Error('no_line_items', esc_html__('No line items found for this order', 'eventin-addon-for-surecart'));
        }

        $checkout_data = [
            'checkout' => [
                'email'      => $order->customer_email,
                'first_name' => !empty($customer->first_name) ? $customer->first_name : '',
                'last_name'  => !empty($customer->last_name) ? $customer->last_name : '',
                'live_mode'  => $this->is_live_mode(),
                'line_items' => $line_items,
                'metadata'   => [
                    'eventin_order_id' => $order->id,
                ],
            ],
        ];

        $checkout_response = $this->make_api_request('POST', '/checkouts', $checkout_data);
        return $checkout_response;
    }

    /**
     * Retrieve a checkout by ID and check its status
     *
     * @param string $checkout_id The checkout ID
     * @return array|WP_Error Checkout data or error
     *
     * @since 1.0.0
     */
    public function create_refund( $checkout_id ) {
        $api_key = etn_get_option( 'surecart_secret_key' );
        $url = "https://api.surecart.com/v1/checkouts/" . urlencode( $checkout_id );
        $response = wp_remote_get( $url, [
            'headers' => [
                'Authorization' => 'Bearer ' . $api_key,
                'Accept'        => 'application/json',
            ],
        ] );

        if ( is_wp_error( $response ) ) {
            return false;
        }

        $code = wp_remote_retrieve_response_code( $response );
        $body = wp_remote_retrieve_body( $response );
        $data = json_decode( $body );

        if($data->manual_payment){
            return false;
        }

        if ( $code !== 200 ) {
            return false;
        }

        if(isset($data->order) && isset($data->charge) && isset($data->currency) && isset($data->net_paid_amount)){
            // Sanitize JSON decoded values before use
            return $this->sc_create_refund(
                sanitize_text_field($data->order),
                sanitize_text_field($data->charge),
                sanitize_text_field($data->currency),
                floatval($data->net_paid_amount)
            );
        }

        return false;
    }

    /**
     * Create a refund for a charge
     *
     * @param string $order_id The order ID
     * @param string $charge_id The charge ID
     * @param string $currency The currency
     * @param float $amount The amount to refund
     * @return array|WP_Error Refund data or error
     *
     * @since 1.0.0
     */
    public function sc_create_refund( $order_id, $charge_id, $currency, $amount ) {
        $api_key = etn_get_option( 'surecart_secret_key' );
        $url = "https://api.surecart.com/v1/refunds";
        $body = [
            'refund'=>[
                'charge'   => $charge_id,
                'order_id' => $order_id,
                'amount'   => $amount,
                'currency' => $currency
            ]
        ];
        $response = wp_remote_post( $url, [
            'headers' => [
                'Authorization' => 'Bearer ' . $api_key,
                'Content-Type'  => 'application/json',
            ],
            'body'    => wp_json_encode( $body ),
        ] );

        if ( is_wp_error( $response ) ) {
            return false;
        }
        
        $code = wp_remote_retrieve_response_code( $response );

        if ( $code !== 200 ) {
            return false;
        }
        
        return true;
    }

    /**
     * Get line items for the SureCart API by creating products/prices dynamically
     *
     * @param OrderModel $order The order object
     * @return array Line items for the API request
     *
     * @since 1.0.0
     */
    public function get_line_items( $order ) {
        $tickets = $order->get_tickets();
        $line_items = [];

        if (empty($tickets)) {
            return [];
        }

        foreach( $tickets as $ticket) {
            if ( ! isset( $ticket['etn_ticket_qty'] ) || ! isset( $ticket['etn_ticket_price'] ) || ! isset( $ticket['etn_ticket_name'] ) ) {
                continue;
            }

            $quantity     = (int) $ticket['etn_ticket_qty'];
            $price_amount = (int) ( floatval($ticket['etn_ticket_price']) * 100 ); // Convert to cents

            if ( $quantity <= 0 || $price_amount <= 0 ) {
                continue;
            }

            $price_id = $this->create_product_and_price($ticket);

            if ( ! $price_id ) {
                continue;
            }

            $line_items[] = [
                'quantity' => $quantity,
                'price'    => $price_id,
            ];
        }

        return $line_items;
    }

    /**
     * Create a product and price for a ticket, or return false/WP_Error on failure.
     *
     * @param array $ticket
     * @return string|WP_Error Price ID on success, WP_Error on failure.
     *
     * @since 1.0.0
     */
    private function create_product_and_price( $ticket ) {
        $name_raw  = $ticket['etn_ticket_name'] ?? '';
        $price_raw = $ticket['etn_ticket_price'] ?? 0;

        $name  = trim( sanitize_text_field( $name_raw ) );
        $price = floatval( $price_raw );

        if ( empty( $name ) || $price <= 0 ) {
            return new WP_Error( 'invalid_ticket', esc_html__('Invalid ticket data for SureCart product/price creation.', 'eventin-addon-for-surecart') );
        }

        $product_data = [
            'name'        => $name,
            /* translators: %s: Ticket name */
            'description' => sprintf( __( 'Event ticket: %s', 'eventin-addon-for-surecart' ), $name ),
            'status'      => 'draft',
            'metadata'    => [
                'created_by'  => 'eventin_integration',
                'ticket_type' => 'event_ticket',
            ],
        ];

        $product_response = $this->make_api_request( 'POST', '/products', [ 'product' => $product_data ] );

        if ( is_wp_error( $product_response ) ) {
            return $product_response;
        }

        $product_id = $this->extract_id_from_response($product_response, 'product');

        if ( empty( $product_id ) ) {
            return new WP_Error( 'product_no_id', esc_html__('Product created but no product id returned from SureCart.', 'eventin-addon-for-surecart') );
        }
        $currency     = etn_currency();
        $amount_cents = (int) round( $price * 100 );

        $price_payload = [
            'product'  => $product_id,
            'amount'   => $amount_cents,
            'currency' => $currency,
            'type'     => 'one_time',
            'metadata' => [
                'created_by'     => 'eventin',
                'original_price' => number_format( $price, 2, '.', '' ),
            ],
        ];

        $price_request = [ 'price' => $price_payload ];
        $price_response = $this->make_api_request( 'POST', '/prices', $price_request );

        if ( is_wp_error( $price_response ) ) {
            try {
                $this->make_api_request( 'DELETE', '/products/' . rawurlencode( $product_id ) );
            } catch ( \Throwable $e ) {
                // ignore cleanup failures
            }
            return $price_response;
        }

        $price_id = $this->extract_id_from_response($price_response, 'price');

        if ( empty( $price_id ) ) {
            return new WP_Error( 'price_no_id', esc_html__('Price created but no price id returned from SureCart.', 'eventin-addon-for-surecart') );
        }

        return $price_id;
    }

    /**
     * Make API request to SureCart
     *
     * @param string $method HTTP method
     * @param string $endpoint API endpoint
     * @param array $data Request data
     * @return array|false Response data on success, false on failure
     *
     * @since 1.0.0
     */
    private function make_api_request($method, $endpoint, $data = []) {
        $url = $this->base_url . $endpoint;

        $args = [
            'method'  => $method,
            'headers' => [
                'Authorization' => 'Bearer ' . $this->api_key,
                'Content-Type'  => 'application/json',
            ],
            'timeout'   => 30,
            'sslverify' => true,
        ];

        if (!empty($data)) {
            $args['body'] = json_encode($data);
        }

        $response = wp_remote_request( $url, $args );

        if ( is_wp_error( $response ) ) {
            return new WP_Error( 'surecart_api_error', esc_html__('Unable to process API request', 'eventin-addon-for-surecart') );
        }

        $response_code = wp_remote_retrieve_response_code( $response );
        $body = wp_remote_retrieve_body( $response );

        if ( $response_code >= 200 && $response_code < 300 ) {
            return json_decode($body, true);
        }

        return false;
    }

    /**
     * Validate API key is configured
     *
     * @return bool|WP_Error True if valid, WP_Error if missing
     */
    private function validate_api_key() {
        if (empty($this->api_key)) {
            return new WP_Error('missing_api_key', esc_html__('SureCart API key is not configured', 'eventin-addon-for-surecart'));
        }
        return true;
    }

    /**
     * Extract ID from API response
     *
     * @param array $response API response array
     * @param string $key The key to extract from (default: 'id')
     * @return string The extracted ID or empty string if not found
     *
     * @since 1.0.0
     */
    private function extract_id_from_response($response, $key = 'id') {
        if (isset($response['id'])) {
            return (string) $response['id'];
        }

        if (isset($response[$key]['id'])) {
            return (string) $response[$key]['id'];
        }

        return '';
    }

    /**
     * Check if we're in live mode
     *
     * @return bool
     */
    public function is_live_mode() {
        if ( function_exists('etn_get_option') && etn_get_option('surecart_sandbox', true) ) {
            return !etn_get_option('surecart_sandbox');
        }

        return false;
    }

    /**
     * Retrieve a checkout by ID and check its status
     *
     * @param string $checkout_id The checkout ID
     * @return array|WP_Error Checkout data or error
     *
     * @since 1.0.0
     */
    public function get_checkout_status( $checkout_id ) {
        $validation = $this->validate_api_key();
        if ( is_wp_error( $validation ) ) {
            return $validation;
        }

        $response = $this->make_api_request('GET', '/checkouts/' . $checkout_id);

        if ( is_wp_error( $response ) ) {
            return $response;
        }

        if ( !$response ) {
            return new \WP_Error('empty_response', esc_html__('Empty response from SureCart API', 'eventin-addon-for-surecart'));
        }

        return $response;
    }
}
