<?php
namespace Etn_Surecart_Addon;

defined( 'ABSPATH' ) || exit;

use Etn_Surecart_Addon\Integrations\SureCart\SureCartPaymentTimer;
/**
 * Init Main Class
 *
 * @package Etn_Surecart_Addon/Init
 *
 * @since 1.0.0
 */
class Init {

    /**
     * Plugin version.
     *
     * @var string
     *
     * @since 1.0.0
     */
    public $version;

    /**
     * Plugin file.
     *
     * @var string
     *
     * @since 1.0.0
     */
    public $plugin_file;

    /**
     * Plugin directory.
     *
     * @var string
     *
     * @since 1.0.0
     */
    public $plugin_directory;

    /**
     * @var string
     *
     * @since 1.0.0
     */
    public $build_url;

    /**
     * Plugin base name.
     *
     * @var string
     *
     * @since 1.0.0
     */
    public $basename;

    /**
     * Plugin text directory path.
     *
     * @var string
     *
     * @since 1.0.0
     */
    public $text_domain_directory;

    /**
     * Plugin text directory path.
     *
     * @var string
     *
     * @since 1.0.0
     */
    public $template_directory;

    /**
     * Plugin url.
     *
     * @var string
     */
    public $plugin_url;

    /**
     * Container that holds all the services.
     *
     * @var Container
     *
     * @since 1.0.0
     */
    public $container;

    /**
     * Eventin Surecart Addon Constructor.
     *
     * @since 1.0.0
     */
    public function __construct() {
        $this->init();
        $this->init_classes();
        $this->add_necessary_filters();

        // Prevent deactivation of Eventin free plugin when surecart addon is active
        add_action( 'admin_init', [$this, 'prevent_eventin_deactivation'] );
        add_action( 'deactivate_wp-event-solution/eventin.php', [$this, 'block_eventin_deactivation'], 1 );
    }

    /**
     * Prevent deactivation of Eventin free plugin when surecart addon is active
     *
     * @return void
     *
     * @since 1.0.0´
     */
    public function prevent_eventin_deactivation() {
        // Only run on plugins page
        $page = isset($_GET['page']) ? sanitize_text_field(wp_unslash($_GET['page'])) : '';
        if ($page !== 'plugins.php') {
            return;
        }

        // Verify nonce for plugin deactivation
        if (isset($_GET['_wpnonce']) && !wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'deactivate-plugin_wp-event-solution/eventin.php')) {
            return;
        }

        // Check if user is trying to deactivate Eventin free plugin
        $action = isset($_GET['action']) ? sanitize_text_field(wp_unslash($_GET['action'])) : '';
        $plugin = isset($_GET['plugin']) ? sanitize_text_field(wp_unslash($_GET['plugin'])) : '';

        if ($action === 'deactivate' && $plugin === 'wp-event-solution/eventin.php') {

            // Check if Eventin surecart addon is active
            if (class_exists('Etn_Surecart_Addon\Init')) {
                // Redirect back to plugins page with error message
                wp_safe_redirect(admin_url('plugins.php?Etn_Surecart_Addon_deactivation_blocked=1'));
                exit;
            }
        }

        // Show error message if deactivation was blocked
        $deactivation_blocked = isset($_GET['Etn_Surecart_Addon_deactivation_blocked']) ? sanitize_text_field(wp_unslash($_GET['Etn_Surecart_Addon_deactivation_blocked'])) : '';

        if ($deactivation_blocked === '1' &&
            isset($_GET['_wpnonce']) &&
            wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'deactivate-plugin_wp-event-solution/eventin.php')) {
            add_action('admin_notices', function() {
                echo '<div class="notice notice-error is-dismissible"><p>' .
                     esc_html__('Eventin Free Plugin cannot be deactivated. Eventin surecart addon requires the free Eventin plugin to be active. Please deactivate Eventin surecart addon first if you want to deactivate the free plugin.', 'eventin-addon-for-surecart') .
                     '</p></div>';
            });
        }
    }

    /**
     * Block Eventin free plugin deactivation using WordPress hook
     *
     * @return void
     *
     * @since 1.0.0
     */
    public function block_eventin_deactivation() {
        // Check if Eventin surecart addon is active
        if (class_exists('Etn_Surecart_Addon\Init')) {
            // Prevent deactivation by throwing an error
            wp_die(
                esc_html__('Eventin Free Plugin cannot be deactivated because Eventin surecart addon requires it to be active. Please deactivate Eventin surecart addon first if you want to deactivate the free plugin.', 'eventin-addon-for-surecart'),
                esc_html__('Plugin Deactivation Blocked', 'eventin-addon-for-surecart'),
                ['response' => 403, 'back_link' => true]
            );
        }
    }

    /**
     * Initialize the plugin.
     *
     * @return void
     *
     * @since 1.0.0
     */
    protected function init(): void {
        $this->version               = '1.0.0';
        $this->plugin_file           = EVENTIN_SURECART_ADDON_FILE;
        $this->plugin_directory      = EVENTIN_SURECART_ADDON_DIR;
        $this->basename              = plugin_basename( $this->plugin_file );
        $this->text_domain_directory = $this->plugin_directory . '/languages';
        $this->template_directory    = $this->plugin_directory . '/templates';
        $this->plugin_url            = plugins_url( '', $this->plugin_file );
        $this->build_url             = $this->plugin_url . '/build';
    }

    /**
     * Init classes
     *
     * @return  void
     *
     * @since 1.0.0
     */
    protected function init_classes() {
        $this->get_container()->get('global');
    }

    /**
     * Add necessary filters
     *
     * @return void
     *
     * @since 1.0.0
     */
    protected function add_necessary_filters() {
        // Ensure SureCart payment method is mapped early
        add_filter('eventin_payment_methods', function ($methods) {
            $methods['sure_cart'] = \Etn_Surecart_Addon\Integrations\SureCart\SureCartPaymentMethod::class;
            return $methods;
        }, 5);

        // Expose SureCart enabled flag in Eventin localized vars
        add_filter('etn_locale_vars', function ($data) {
            if (function_exists('etn_get_option')) {
                $data['payment_option_surecart'] = etn_get_option('surecart_status');
            }
            return $data;
        }, 10);

        add_filter('surecart-components/scData', function ($data) {
            $base_url  = get_site_url();
            $success_url = trailingslashit( $base_url ) . 'eventin-purchase/checkout/#/success';
            $data['pages']['dashboard'] = $success_url;
            return $data;
        });

        add_action('plugins_loaded', function () {
            new SureCartPaymentTimer();
        });

    }

    /**
     * Get container
     *
     * @return  Container
     *
     * @since 1.0.0
     */
    public function get_container() {
        return eventin_surecart_addon_container();
    }

    /**
     * Initializes the Eventin Surecart Addon class.
     *
     * Checks for an existing Eventin Surecart Addons instance
     * and if it doesn't find one, creates it.
     *
     * @return Init
     *
     * @since 1.0.0
     */
    public static function instance(): Init {
        static $instance = false;

        if ( ! $instance ) {
            $instance = new self();
        }

        return $instance;
    }

    /**
	 * Plugin File
	 *
	 * @since 1.0.0
	 *
	 * @return string
	 */
	public static function plugin_file() {
		return __FILE__;
	}

    /**
	 * Plugin Directory Path
	 *
	 * @since 1.0.0
	 *
	 * @return string
	 */
	public static function plugin_dir() {
		return trailingslashit( plugin_dir_path( self::plugin_file() ) );
	}
}
