<?php

namespace Eventflow\Interlink;

use Eventflow\Interlink\Client\Request_Builder;
use Eventflow\Interlink\Client\WP_Http_Invoker;

class Plugin {

	const CONFIDENCE_THRESHOLD = 0.3;
	const MAX_KEYWORDS_IN_RESULT = 10;
	const KEYWORD_TAXONOMY_SLUG = 'keyword';

	private static $instance = null;

	protected $version;

	private function __construct() {
		$this->version = EVENTFLOW_INTERLINK_PLUGIN_VERSION;
	}

	public static function getInstance() {
		if ( static::$instance == null ) {
			static::$instance = new Plugin();
		}

		return static::$instance;
	}

	public function register() {
		( new Activator() )->register();

		// i18n
		add_action(
			'plugins_loaded',
			function () {
				load_plugin_textdomain(
					'eventflow-interlink',
					false,
					dirname( dirname( plugin_basename( __FILE__ ) ) ) . '/languages/'
				);
			}
		);

		// taxonomy
		add_action(
			'init',
			function () {
				register_taxonomy(
					self::KEYWORD_TAXONOMY_SLUG,
					'post',  // only for posts (not pages, etc.)
					array(
						'label'       => __( 'Keywords', 'eventflow-interlink' ),
						'meta_box_cb' => [ $this, 'renderMetaBox' ],
						'rewrite'     => array( 'slug' => self::KEYWORD_TAXONOMY_SLUG )
					)
				);
			}
		);

		// Ajax handler for keywords metabox
		add_action(
			'admin_enqueue_scripts',
			function () {
				$screen = get_current_screen();
				if ( $screen->id == 'post' ) {
					global $post;
					$js_handle = 'eventflow-metabox-handler';
					wp_enqueue_script( $js_handle, EVENTFLOW_INTERLINK_PLUGIN_URL . 'dist/assets/js/eventflow-metabox-handler.js' );
					wp_localize_script( $js_handle, 'evfl_l10n', array(
						'loading'        => __( 'loading', 'eventflow-interlink' ),
						'admin_ajax_url' => get_admin_url( null, '', 'rest' ) . 'admin-ajax.php',
						'post_id'        => $post->ID,
						'tag_delimiter'  => _x( ',', 'tag delimiter', 'default' )
					) );
				}
			}
		);

		add_action( 'wp_ajax_evfl_extract', [ $this, 'handleAjaxCall' ] );
		add_filter( 'the_content', [ $this, 'renderTagCloud' ] );
	}

	/**
	 * meta-box
	 *
	 * @param $post
	 * @param $box
	 */
	public function renderMetaBox( $post, $box ) {
		$defaults = array( 'taxonomy' => self::KEYWORD_TAXONOMY_SLUG );
		if ( ! isset( $box['args'] ) || ! is_array( $box['args'] ) ) {
			$args = array();
		} else {
			$args = $box['args'];
		}
		extract( wp_parse_args( $args, $defaults ), EXTR_SKIP );
		$tax_name              = self::KEYWORD_TAXONOMY_SLUG;
		$taxonomy              = get_taxonomy( self::KEYWORD_TAXONOMY_SLUG );
		$user_can_assign_terms = current_user_can( $taxonomy->cap->assign_terms );
		$comma                 = _x( ',', 'tag delimiter', 'default' );
		?>
        <div class="tagsdiv" id="<?php echo $tax_name; ?>">
            <div class="jaxtag">
                <div class="nojs-tags hide-if-js">
                    <p><?php echo $taxonomy->labels->add_or_remove_items; ?></p>
                    <textarea name="<?php echo "tax_input[$tax_name]"; ?>" rows="3" cols="20" class="the-tags"
                              id="tax-input-<?php echo $tax_name; ?>" <?php disabled( ! $user_can_assign_terms ); ?>><?php echo str_replace( ',', $comma . ' ', get_terms_to_edit( $post->ID, $tax_name ) ); // textarea_escaped by esc_attr()
						?></textarea>
                </div>
				<?php if ( $user_can_assign_terms ) : ?>
                    <div class="ajaxtag hide-if-no-js">
                        <label class="screen-reader-text"
                               for="new-tag-<?php echo $tax_name; ?>"><?php echo $box['title']; ?></label>
                        <p><input type="hidden" id="new-tag-<?php echo $tax_name; ?>"
                                  name="newtag[<?php echo $tax_name; ?>]" class="newtag form-input-tip" size="16"
                                  autocomplete="off" value=""/>
                            <input id="evfl_keywords_btn" type="button" class="button"
                                   value="<?php esc_attr_e( 'Extract', 'eventflow-interlink' ); ?>"/>
                            <span id="evfl_message" style="color: #666; font-style: italic; margin-left: 5px;"></span>
                        </p>
                    </div>
				<?php endif; ?>
                <p class="howto"
                   id="evfl_keywords_desc"><?php echo __( 'Save draft, extract, remove unwanted', 'eventflow-interlink' ); ?></p>
            </div>
            <ul class="tagchecklist" role="list"></ul>
        </div>
		<?php
	}

	/**
	 * display keywords in a tag cloud
	 *
	 * @param $content
	 */
	public function renderTagCloud( $content ) {
		global $post;
		echo $content;
		echo the_terms( $post->ID, self::KEYWORD_TAXONOMY_SLUG, ' ', ' &nbsp;&nbsp; ' );
	}

	public function handleAjaxCall() {
		$aResponse = array( 'message' => '', 'data' => array() );

		if ( isset( $_POST['post_id'] ) ) {
			$iPostId         = (int) $_POST['post_id'];
			$arrAjaxResponse = $this->fetchKeywords( $iPostId );

			if ( $arrAjaxResponse != null ) {
				$iStatus   = $arrAjaxResponse['status'];
				$aKeywords = $arrAjaxResponse['keywords'];

				if ( $iStatus == 200 && ! empty( $aKeywords ) ) {
					wp_set_object_terms( $iPostId, array_map( 'esc_html', $aKeywords ), self::KEYWORD_TAXONOMY_SLUG, false );
					$aResponse['data'] = $aKeywords;
				} else if ( $iStatus == 200 && empty( $aKeywords ) ) {
					$aResponse['message'] = __( 'no result', 'eventflow-interlink' );
				} else if ( $iStatus == 401 ) {
					$aResponse['message'] = __( 'unauthorized', 'eventflow-interlink' );
				} else if ( $iStatus == 403 ) {
					$aResponse['message'] = __( 'quota exceeded', 'eventflow-interlink' );
				} else {
					$aResponse['message'] = __( 'request failed', 'eventflow-interlink' ) . '&nbsp;(' . $iStatus . ')';
				}
			} else {
				$aResponse['message'] = __( 'request failed', 'eventflow-interlink' );
			}
		} else {
			$aResponse['message'] = __( 'request failed', 'eventflow-interlink' );
		}

		// the script dies after calling 'wp_send_json'
		wp_send_json( $aResponse );
	}

	/**
	 * API call wrapper
	 */
	private function fetchKeywords( $post_id ) {
		$post = get_post( $post_id );

		if ( $post == null ) {
			error_log( "Request to Eventflow API failed (post not found)" );

			return null;
		}

		$oRequestBuilder = new Request_Builder( EVENTFLOW_INTERLINK_CONSUMER_KEY, EVENTFLOW_INTERLINK_CONSUMER_SECRET );
		$oRequestBuilder->setCredentials( get_option( 'eventflow_access_token' ), get_option( 'eventflow_access_token_secret' ) );
		$oRequestBuilder->setEndpoint( EVENTFLOW_API_KEYWORD_EXTRACTION_ENDPOINT );
		$oRequestBuilder->setHTTPMethod( 'POST' );

		$aContent = array( "text" => sanitize_textarea_field( $post->post_content ) );
		$oRequestBuilder->setContent( json_encode( $aContent ) );
		$oRequest = $oRequestBuilder->build();

		$oInvoker  = new WP_Http_Invoker( $oRequest );
		$oResponse = $oInvoker->request();

		\Eventflow\Interlink\Plugin::DEBUG( $oResponse );

		if ( ! is_null( $oResponse ) ) {
			if ( $oResponse->getStatusCode() == 200 ) {
				$keywords     = json_decode( $oResponse->getBody() );
				$aNewKeywords = array();
				if ( sizeof( $keywords ) ) {
					$iKeywordCount = 0;
					foreach ( $keywords as $keyword ) {
						if ( $iKeywordCount > self::MAX_KEYWORDS_IN_RESULT ) {
							break;
						}
						if ( $keyword->score >= self::CONFIDENCE_THRESHOLD ) {
							$aNewKeywords[] = $keyword->orthography;
							$iKeywordCount ++;
						}
					}
				}

				return array( 'status' => $oResponse->getStatusCode(), 'keywords' => $aNewKeywords );
			} else {
				error_log( "Request to Eventflow API failed with status " . $oResponse->getStatusCode() );

				return array( 'status' => $oResponse->getStatusCode(), 'keywords' => null );
			}
		} else {
			error_log( "Request to Eventflow API failed (WP error)" );
		}

		return null;
	}

	public static function DEBUG( $anything ) {
		ob_start();
		var_dump( $anything );
		$dump = ob_get_clean();
		error_log( $dump );
	}

}
