<?php

namespace Eventflow\Interlink\OAuth;


class Request {
	public static $PHP_STDIN = 'php://input'; // this is kind of strange, but it's set in tests directly

	public static $version = '1.0';

	private $_aParameters;
	private $_sHTTPMethod;
	private $_sURL;
	private $_sContent;

	/**
	 * @var string $_sBaseString added for debugging
	 */
	private $_sBaseString;

	function __construct( $sHTTPMethod, $sURL, array $aParameters = array() ) {
		$this->_aParameters = array_merge( Util::parse_parameters( parse_url( $sURL, PHP_URL_QUERY ) ),
			$aParameters );
		$this->_sHTTPMethod = $sHTTPMethod;
		$this->_sURL        = $sURL;
	}

	/**
	 * @static
	 *
	 * @param  $http_method
	 * @param  $http_url
	 * @param  $parameters
	 *
	 * @return Request
	 */
	public static function fromRequest( $http_method = null, $http_url = null, $parameters = null ) {
		$scheme      = ( ! isset( $_SERVER['HTTPS'] ) || $_SERVER['HTTPS'] != "on" ) ? 'http' : 'https';
		$http_url    = ( $http_url ) ? $http_url :
			sprintf( "%s://%s:%s%s", $scheme, $_SERVER['HTTP_HOST'], $_SERVER['SERVER_PORT'], $_SERVER['REQUEST_URI'] );
		$http_method = ( $http_method ) ? $http_method : $_SERVER['REQUEST_METHOD'];

		// We weren't handed any parameters, so let's find the ones relevant to this request.
		// If you run XML-RPC or similar you should use this to provide your own parsed parameter-list
		if ( ! $parameters ) {
			// Find request headers
			$request_headers = Util::get_headers();

			// Parse the query-string to find GET parameters
			$parameters = Util::parse_parameters( $_SERVER['QUERY_STRING'] );

			// It's a POST request of the proper content-type, so parse POST
			// parameters and add those overriding any duplicates from GET
			if ( $http_method == "POST" &&
			     isset( $request_headers['Content-Type'] ) &&
			     strstr( $request_headers['Content-Type'], 'application/x-www-form-urlencoded' ) ) {
				$post_data  = Util::parse_parameters( file_get_contents( self::$PHP_STDIN ) );
				$parameters = array_merge( $parameters, $post_data );
			}

			// We have a Authorization-header with OAuth data. Parse the header
			// and add those overriding any duplicates from GET or POST
			if ( isset( $request_headers['Authorization'] ) &&
			     substr( $request_headers['Authorization'], 0, 6 ) == 'OAuth ' ) {
				$header_parameters = Util::split_header( $request_headers['Authorization'] );
				$parameters        = array_merge( $parameters, $header_parameters );
			}

		}

		return new Request( $http_method, $http_url, $parameters );
	}

	/**
	 * pretty much a helper function to set up the request
	 *
	 * @static
	 *
	 * @param Consumer $oConsumer
	 * @param Token $oToken
	 * @param  $sHTTPMethod
	 * @param  $http_url
	 * @param  $parameters
	 *
	 * @return Request
	 */
	public static function fromConsumerAndToken(
		Consumer $oConsumer, Token $oToken, $sHTTPMethod,
		$http_url, $parameters = null
	) {
		$parameters = ( $parameters ) ? $parameters : array();
		$defaults   = array(
			"oauth_version"      => Request::$version,
			"oauth_nonce"        => Request::_generateNonce(),
			"oauth_timestamp"    => Request::_generateTimestamp(),
			"oauth_consumer_key" => $oConsumer->getKey()
		);
		if ( $oToken ) {
			$defaults['oauth_token'] = $oToken->getKey();
		}

		$parameters = array_merge( $defaults, $parameters );

		return new Request( $sHTTPMethod, $http_url, $parameters );
	}

	public function setContent( $sContent ) {
		$this->_sContent = $sContent;
	}

	/**
	 * @return mixed
	 */
	public function getContent() {
		return $this->_sContent;
	}

	public function setParameter( $name, $value, $allow_duplicates = true ) {
		if ( $allow_duplicates && isset( $this->_aParameters[ $name ] ) ) {
			// We have already added parameter(s) with this name, so add to the list
			if ( is_scalar( $this->_aParameters[ $name ] ) ) {
				// This is the first duplicate, so transform scalar (string)
				// into an array so we can add the duplicates
				$this->_aParameters[ $name ] = array( $this->_aParameters[ $name ] );
			}
			$this->_aParameters[ $name ][] = $value;
		} else {
			$this->_aParameters[ $name ] = $value;
		}
	}

	public function getParameter( $name ) {
		return isset( $this->_aParameters[ $name ] ) ? $this->_aParameters[ $name ] : null;
	}

	/**
	 * @return array
	 */
	public function getParameters() {
		return $this->_aParameters;
	}

	public function unsetParameter( $name ) {
		unset( $this->_aParameters[ $name ] );
	}

	/**
	 * The request parameters, sorted and concatenated into a normalized string.
	 * @return string
	 */
	public function getSignableParameters() {
		// Grab all parameters
		$params = $this->_aParameters;

		// Remove oauth_signature if present
		// Ref: Spec: 9.1.1 ("The oauth_signature parameter MUST be excluded.")
		if ( isset( $params['oauth_signature'] ) ) {
			unset( $params['oauth_signature'] );
		}

		return Util::build_http_query( $params );
	}

	/**
	 * Returns the base string of this request
	 *
	 * The base string defined as the method, the url
	 * and the parameters (normalized), each urlencoded
	 * and the concated with &.
	 */
	public function getSignatureBaseString() {
		$parts = array(
			$this->getNormalizedHTTPMethod(),
			$this->getNormalizedURL(),
			$this->getSignableParameters()
		);
		$parts = Util::urlencode_rfc3986( $parts );

		return implode( '&', $parts );
	}

	/**
	 * just uppercases the http method
	 */
	public function getNormalizedHTTPMethod() {
		return strtoupper( $this->_sHTTPMethod );
	}

	/**
	 * parses the url and rebuilds it to be ``scheme://host/path''
	 */
	public function getNormalizedURL() {
		$parts = parse_url( $this->_sURL );

		$scheme = ( isset( $parts['scheme'] ) ) ? $parts['scheme'] : 'http';
		$port   = ( isset( $parts['port'] ) ) ? $parts['port'] : ( ( $scheme == 'https' ) ? '443' : '80' );
		$host   = ( isset( $parts['host'] ) ) ? $parts['host'] : '';
		$path   = ( isset( $parts['path'] ) ) ? $parts['path'] : '';

		if ( ( $scheme == 'https' && $port != '443' ) || ( $scheme == 'http' && $port != '80' ) ) {
			$host = "$host:$port";
		}

		return sprintf( "%s://%s%s", $scheme, $host, $path );
	}

	/**
	 * builds a url usable for a GET request
	 */
	public function toURL() {
		$post_data = $this->toPOST();
		$out       = $this->getNormalizedURL();
		if ( $post_data ) {
			$out .= '?' . $post_data;
		}

		return $out;
	}

	/**
	 * builds the data one would send in a POST request
	 */
	public function toPOST() {
		return Util::build_http_query( $this->_aParameters );
	}

	/**
	 * builds the Authorization: header
	 */
	public function toHeader( $realm = null ) {
		$first = true;
		if ( $realm ) {
			$out   = 'Authorization: OAuth realm="' . Util::urlencode_rfc3986( $realm ) . '"';
			$first = false;
		} else {
			$out = 'Authorization: OAuth';
		}

		foreach ( $this->_aParameters as $k => $v ) {
			if ( substr( $k, 0, 5 ) != "oauth" ) {
				continue;
			}
			if ( is_array( $v ) ) {
				throw new OAuth_Exception( 'Arrays not supported in headers' );
			}
			$out   .= ( $first ) ? ' ' : ',';
			$out   .= Util::urlencode_rfc3986( $k ) . '="' . Util::urlencode_rfc3986( $v ) . '"';
			$first = false;
		}

		return $out;
	}

	public function __toString() {
		return $this->toURL();
	}

	public function signRequest( Signature_Method $oSignatureMethod, Consumer $oConsumer, Token $oToken ) {
		$this->setParameter( "oauth_signature_method", $oSignatureMethod->getMethod(), false );
		$signature = $this->buildSignature( $oSignatureMethod, $oConsumer, $oToken );
		$this->setParameter( "oauth_signature", $signature, false );
	}

	public function buildSignature( Signature_Method $oSignatureMethod, Consumer $oConsumer, Token $oToken ) {
		$signature = $oSignatureMethod->buildSignature( $this, $oConsumer, $oToken );

		return $signature;
	}

	/**
	 * This is added for debugging, we never read this value but it's gonna be there when we dump the entire object
	 *
	 * @param  $sBaseString
	 *
	 * @return void
	 */
	public function setBaseString( $sBaseString ) {
		$this->_sBaseString = $sBaseString;
	}

	/**
	 * util function: current timestamp
	 */
	private static function _generateTimestamp() {
		return time();
	}

	/**
	 * util function: current nonce
	 */
	private static function _generateNonce() {
		$mt   = microtime();
		$rand = mt_rand();

		return md5( $mt . $rand ); // md5s look nicer than numbers
	}
}
