<?php

namespace Eventflow\Interlink\Client;

use Eventflow\Interlink\OAuth\Consumer;
use Eventflow\Interlink\OAuth\Request;
use Eventflow\Interlink\OAuth\Signature_Method_HMAC_SHA1;
use Eventflow\Interlink\OAuth\Token;


/**
 * Build and sign requests
 */
class Request_Builder {
	/**
	 * Note: Content-Type does not make sense for GET
	 */
	const APPLICATION_JSON_TYPE = "application/json";

	const X_WWW_FORM_URLENCODED_TYPE = "application/x-www-form-urlencoded";

	private $_sConsumerKey;
	private $_sConsumerSecret;

	private $_sEndpoint;

	private $_sAccessToken;
	private $_sAccessSecret;

	/**
	 * @var Signature_Method_HMAC_SHA1
	 */
	private $_oSignatureMethod;

	private $_sHTTPMethod;

	private $_sContent = '';

	/**
	 * @var array key-value pairs
	 */
	private $_aHeaders = array();

	public function __construct( $sConsumerKey, $sConsumerSecret ) {
		$this->_sConsumerKey     = $sConsumerKey;
		$this->_sConsumerSecret  = $sConsumerSecret;
		$this->_oSignatureMethod = new Signature_Method_HMAC_SHA1();
	}

	public function setEndpoint( $sEndpoint ) {
		$this->_sEndpoint = $sEndpoint;
	}

	public function setCredentials( $sAccessToken = null, $sAccessSecret = null ) {
		$this->_sAccessToken  = $sAccessToken;
		$this->_sAccessSecret = $sAccessSecret;
	}

	public function setHTTPMethod( $sHTTPMethod ) {
		$this->_sHTTPMethod = $sHTTPMethod;
	}

	/**
	 * Вы всегда можете установить контент, но он будет добавлен в запрос и подписан только если он имеет смысл,
	 * т.е. если ваш запрос - POST или PUT
	 *
	 * @param  $sContent
	 *
	 * @return void
	 */
	public function setContent( $sContent ) {
		$this->_sContent = $sContent;
	}

	/**
	 * Это мы отправим внутрь Eventflow\Interlink\Client\DecoratedRequest, когда он у нас появится
	 *
	 * @param  $sKey
	 * @param  $sValue
	 *
	 * @return void
	 */
	public function addHeader( $sKey, $sValue ) {
		$this->_aHeaders[ $sKey ] = $sValue;
	}

	/**
	 * @return Decorated_Request
	 */
	public function build() {
		$oConsumer = new Consumer( $this->_sConsumerKey, $this->_sConsumerSecret, null );
		$oToken    = new Token( $this->_sAccessToken, $this->_sAccessSecret );

		$parsed      = parse_url( $this->_sEndpoint );
		$aParameters = array();
		@parse_str( $parsed['query'], $aParameters );

		if ( $this->_sContent && ( $this->_sHTTPMethod === 'PUT' || $this->_sHTTPMethod === 'POST' ) ) {
			//   $arrParameters['content'] = $this->_strContent;
			$this->addHeader( 'Content-Type', self::APPLICATION_JSON_TYPE . ';charset=UTF-8' );
			$this->addHeader( 'Accept', self::APPLICATION_JSON_TYPE );
		}

		// POST с пустым контентом используется для OAuth авторизации и DELETE, ставим правильный Content-Type
		if ( ! $this->_sContent && $this->_sHTTPMethod === 'POST' ) {
			$this->addHeader( 'Content-Type', self::X_WWW_FORM_URLENCODED_TYPE . ';charset=UTF-8' );
			$this->addHeader( 'Accept', 'text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8' );
		}

		// Все отправляем POSTом с заголовком X-Http-Method-Override
		if ( $this->_sHTTPMethod === 'PUT' || $this->_sHTTPMethod === 'DELETE' ) {
			$sHTTPMethodOverride = 'POST';
		} else {
			$sHTTPMethodOverride = $this->_sHTTPMethod;
		}

		// запрос подписывается так, как будто это GET или POST (так и есть)
		$oRequest = Request::fromConsumerAndToken( $oConsumer, $oToken, $sHTTPMethodOverride,
			$this->_sEndpoint, $aParameters );
		$oRequest->signRequest( $this->_oSignatureMethod, $oConsumer, $oToken );

		// NOT the Twitter method (content is not used for signature calculation)
		$oRequest->setContent( $this->_sContent );

		// Создаем финальный объект запроса и украшаем его для продажи
		$oDecoratedRequest = new Decorated_Request( $oRequest );
		if ( $this->_sHTTPMethod === 'PUT' || $this->_sHTTPMethod === 'DELETE' ) {
			$oDecoratedRequest->setXHttpMethodOverride( $this->_sHTTPMethod );
		}
		foreach ( $this->_aHeaders as $key => $value ) {
			$oDecoratedRequest->addHeader( $key, $value );
		}

		return $oDecoratedRequest;
	}
}
