<?php

namespace Eventflow\Interlink;

use Eventflow\Interlink\Client\Request_Builder;
use Eventflow\Interlink\Client\WP_Http_Invoker;


/**
 * Access token is obtained once when the plugin is activated for the first time.
 * This process creates an anonymous user identity with Eventflow API,
 * which uniquely identifies this particular plugin installation.
 * Users will be able to upgrade their identity in the future.
 *
 * We do not remove actual registration with the service as this is supposed to be bound to
 * the user identity at Eventflow API and managed from the Eventflow API Control Panel.
 * However it is possible that we will garbage collect unused anonymous identities in the future.
 *
 * We also keep Access Token in WP options even if the user deactivates the plugin
 * so that she will not loose access to her tracked resources once she wants to reactivate the plugin.
 *
 * @since    1.0.0
 */
class Activator {
	public function __construct() {
	}

	public function register() {
		add_action( 'eventflow_interlink_activation', [ $this, 'activate' ] );
	}

	public function activate() {
		if ( null != get_option( 'eventflow_access_token' ) && null != get_option( 'eventflow_access_token_secret' ) ) {
			return true;
		}

		// obtain Request Token
		$oRequestBuilder = new Request_Builder( EVENTFLOW_INTERLINK_CONSUMER_KEY, EVENTFLOW_INTERLINK_CONSUMER_SECRET );
		$oRequestBuilder->setEndpoint( EVENTFLOW_API_REQUEST_TOKEN_ENDPOINT );
		$oRequestBuilder->setHTTPMethod( 'POST' );
		$oRequest = $oRequestBuilder->build();

		$oInvoker  = new WP_Http_Invoker( $oRequest );
		$oResponse = $oInvoker->request();

		if ( null != $oResponse && $body = $oResponse->getBody() ) {
			parse_str( $body, $rt );
		}

		if ( ! isset( $rt ) ) {
			return false;
		}

		// obtain Access Token
		$oRequestBuilder = new Request_Builder( EVENTFLOW_INTERLINK_CONSUMER_KEY, EVENTFLOW_INTERLINK_CONSUMER_SECRET );
		$oRequestBuilder->setCredentials( $rt['oauth_token'], $rt['oauth_token_secret'] );
		$oRequestBuilder->setEndpoint( EVENTFLOW_API_ACCESS_TOKEN_ENDPOINT );
		$oRequestBuilder->setHTTPMethod( 'POST' );
		$oRequest = $oRequestBuilder->build();

		$oInvoker  = new WP_Http_Invoker( $oRequest );
		$oResponse = $oInvoker->request();

		if ( null != $oResponse && $body = $oResponse->getBody() ) {
			parse_str( $body, $at );
			if ( null != $at['oauth_token'] && null != $at['oauth_token_secret'] ) {
				// store plugin credentials to WP options
				add_option( 'eventflow_access_token', $at['oauth_token'] );
				add_option( 'eventflow_access_token_secret', $at['oauth_token_secret'] );

				return true;
			}
		}

		return false;
	}

}
