<?php
/**
 * The public-facing functionality of the plugin.
 *
 * @link       https://siddiqur.com
 * @since      1.0.0
 *
 * @package    EU_Base_Price_Display
 * @subpackage EU_Base_Price_Display/includes
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and hooks for
 * enqueuing the public-facing stylesheet and JavaScript.
 *
 * @package    EU_Base_Price_Display
 * @subpackage EU_Base_Price_Display/includes
 * @author     Md Siddiqur Rahman <support@siddiqur.com>
 */
class EUBAPRDI_Frontend {

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private string $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private string $version;

    /**
     * The plugin settings.
     *
     * @since    1.0.0
     * @access   private
     * @var      array    $settings    The plugin settings.
     */
    private array $settings;

    /**
     * Cache for base price calculations.
     *
     * @since    1.0.0
     * @access   private
     * @var      array    $base_price_cache    Cached base price data.
     */
    private array $base_price_cache = [];

    /**
     * Maximum size of the cache.
     *
     * @since    1.0.0
     * @access   private
     * @var      int    $max_cache_size    Maximum number of cached items.
     */
    private int $max_cache_size = 100;

    /**
     * Unit conversion rates to base units.
     *
     * @since    1.0.0
     * @access   private
     * @var      array    $conversion_rates    Conversion rates for units.
     */
    private array $conversion_rates = [
        'weight' => [
            'kg'  => 1.0,
            'g'   => 0.001,
            'lbs' => 0.453592,
            'oz'  => 0.0283495,
        ],
        'volume' => [
            'l'   => 1.0,
            'ml'  => 0.001,
            'm3'  => 1000.0,
            'cm3' => 0.001,
        ]
    ];

    /**
     * Initialize the class and set its properties.
     *
     * @since 1.0.0
     * @param string $plugin_name The name of the plugin.
     * @param string $version     The version of the plugin.
     */
    public function __construct( string $plugin_name, string $version ) {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
        $this->settings = $this->eubaprdi_get_settings();
    }

    /**
     * Get plugin settings with defaults
     *
     * @since 1.0.0
     * @return array Plugin settings
     */
    public function eubaprdi_get_settings(): array {
        $defaults = [
            'display_position' => 'after_price',
            'price_label' => __( 'Base price:', 'eu-base-price-display' ),
            'display_unit' => 'kg',
            'decimal_places' => 2
        ];
        
        $settings = get_option( 'eubaprdi_settings', [] );
        
        return wp_parse_args( $settings, $defaults );
    }

    /**
     * Register all of the hooks related to the public-facing functionality.
     *
     * @since 1.0.0
     * @return void
     */
    public function eubaprdi_init_hooks(): void {
        // Only register frontend hooks if not in admin and WooCommerce is active
        if ( is_admin() || ! $this->eubaprdi_is_woocommerce_active() ) {
            return;
        }

        // Enqueue styles
        add_action( 'wp_enqueue_scripts', [ $this, 'eubaprdi_enqueue_styles' ] );

        // Product price display hooks
        add_filter( 'woocommerce_get_price_html', [ $this, 'eubaprdi_add_base_price' ], 100, 2 );
        add_filter( 'woocommerce_available_variation', [ $this, 'eubaprdi_add_base_price_to_variation_data' ], 10, 3 );
        
        // Cart and checkout hooks
        add_filter( 'woocommerce_cart_item_price', [ $this, 'eubaprdi_add_base_price_to_cart_item' ], 10, 3 );
        add_filter( 'woocommerce_get_item_data', [ $this, 'eubaprdi_add_base_price_to_order_summary' ], 100, 2 );
    }

    /**
     * Check if WooCommerce is active and available.
     *
     * @since 1.0.0
     * @return bool True if WooCommerce is active, false otherwise.
     */
    private function eubaprdi_is_woocommerce_active(): bool {
        return class_exists( 'WooCommerce' ) && function_exists( 'WC' );
    }

    /**
     * Register the stylesheets for the public-facing side of the site.
     *
     * @since 1.0.0
     * @return void
     */
    public function eubaprdi_enqueue_styles(): void {
        wp_enqueue_style(
            $this->plugin_name,
            EUBAPRDI_PLUGIN_URL . 'assets/css/eu-base-price-display.css',
            [],
            $this->version,
            'all'
        );
    }

    /**
     * Add base price to product price display.
     *
     * @since 1.0.0
     * @param string     $price The price HTML.
     * @param WC_Product $product The product object.
     * @return string Modified price HTML with base price.
     */
    public function eubaprdi_add_base_price( string $price, WC_Product $product ): string {
        if ( empty( $price ) ) {
            return $price;
        }

        $base_price_data = $this->eubaprdi_get_base_price( $product );
        if ( ! $base_price_data ) {
            return $price;
        }

        $formatted_base_price = $this->eubaprdi_format_base_price_display( $base_price_data, $product );
        
        return $price . ' ' . $formatted_base_price;
    }

    /**
     * Add base price to variation data.
     *
     * @since 1.0.0
     * @param array $data The variation data.
     * @param WC_Product $product The product object.
     * @param WC_Product_Variation $variation The variation object.
     * @return array Modified variation data.
     */
    public function eubaprdi_add_base_price_to_variation_data( array $data, WC_Product $product, WC_Product_Variation $variation ): array {
        $base_price_data = $this->eubaprdi_get_base_price( $variation );
        
        if ( $base_price_data ) {
            $price_label = $this->eubaprdi_get_price_label();
            $formatted_base_price = $price_label . wc_price( $base_price_data['price'] ) . ' / ' . $base_price_data['unit'];
            $data['base_price'] = $formatted_base_price;
        }
        
        return $data;
    }

    /**
     * Add base price to cart item price.
     *
     * @since 1.0.0
     * @param string $price The price HTML.
     * @param array $cart_item The cart item data.
     * @param string $cart_item_key The cart item key.
     * @return string Modified price HTML with base price.
     */
    public function eubaprdi_add_base_price_to_cart_item( string $price, array $cart_item, string $cart_item_key ): string {
        if ( empty( $price ) || ! isset( $cart_item['data'] ) || ! is_a( $cart_item['data'], 'WC_Product' ) ) {
            return $price;
        }

        $product = $cart_item['data'];
        $base_price_data = $this->eubaprdi_get_base_price( $product );
        
        if ( ! $base_price_data ) {
            return $price;
        }

        $formatted_base_price = $this->eubaprdi_format_base_price_display( $base_price_data, $product );
        
        return $price . ' ' . $formatted_base_price;
    }

    /**
     * Add base price to order summary in cart and checkout.
     *
     * @since 1.0.0
     * @param array $item_data The existing cart item data.
     * @param array $cart_item The cart item details including product and quantity.
     * @return array The modified cart item data with base price added.
     */
    public function eubaprdi_add_base_price_to_order_summary( array $item_data, array $cart_item ): array {
        if ( ! isset( $cart_item['data'] ) || ! is_a( $cart_item['data'], 'WC_Product' ) ) {
            return $item_data;
        }

        $product = $cart_item['data'];
        $base_price_data = $this->eubaprdi_get_base_price( $product );

        if ( ! $base_price_data ) {
            return $item_data;
        }

        $price_label = $this->eubaprdi_get_price_label();
        $base_price_display = wc_price( $base_price_data['price'] ) . ' / ' . $base_price_data['unit'];

        // Remove trailing colon from label to prevent double colons in display
        $clean_label = rtrim( $price_label, ':' );

        $item_data[] = [
            'key'   => esc_html( $clean_label ),
            'value' => $base_price_display,
        ];

        return $item_data;
    }

    /**
     * Get the base price for a product.
     *
     * @since 1.0.0
     * @param WC_Product $product The WooCommerce product object.
     * @return array|false Associative array with base price data or false if calculation is not possible.
     */
    public function eubaprdi_get_base_price( WC_Product $product ) {
        $product_id = $product->get_id();
        $cache_key = $this->eubaprdi_generate_cache_key( $product );
        
        // Check cache first
        if ( isset( $this->base_price_cache[ $cache_key ] ) ) {
            return $this->base_price_cache[ $cache_key ];
        }

        $this->eubaprdi_manage_cache_size();

        // Allow developers to override calculation
        $pre_base_price = apply_filters( 'eubaprdi_pre_calculate_base_price', null, $product );
        if ( $pre_base_price !== null ) {
            return $this->eubaprdi_cache_result( $cache_key, $pre_base_price );
        }

        // Check for custom base price override
        if ( $this->eubaprdi_has_custom_price_override( $product_id ) ) {
            $result = $this->eubaprdi_get_custom_base_price( $product_id );
            return $this->eubaprdi_cache_result( $cache_key, $result );
        }

        // Skip if product doesn't require shipping (unless filtered)
        if ( ! $product->needs_shipping() && ! apply_filters( 'eubaprdi_include_non_shipping', false, $product ) ) {
            return $this->eubaprdi_cache_result( $cache_key, false );
        }

        // Get product price
        $price = (float) $product->get_price();
        if ( $price <= 0 ) {
            return $this->eubaprdi_cache_result( $cache_key, false );
        }

        // Calculate base price based on weight or dimensions
        $result = $this->eubaprdi_calculate_base_price_from_product( $product, $price );
        
        return $this->eubaprdi_cache_result( $cache_key, $result );
    }

    /**
     * Generate cache key for product.
     *
     * @since 1.0.0
     * @param WC_Product $product The product object.
     * @return string Cache key.
     */
    private function eubaprdi_generate_cache_key( WC_Product $product ): string {
        $product_id = $product->get_id();
        $product_version = $product->get_date_modified() ? $product->get_date_modified()->getTimestamp() : $product_id;
        
        return $product_id . '_' . $product_version;
    }

    /**
     * Cache calculation result.
     *
     * @since 1.0.0
     * @param string $cache_key The cache key.
     * @param mixed $result The result to cache.
     * @return mixed The cached result.
     */
    private function eubaprdi_cache_result( string $cache_key, $result ) {
        $this->base_price_cache[ $cache_key ] = $result;
        return $result;
    }

    /**
     * Check if product has custom price override.
     *
     * @since 1.0.0
     * @param int $product_id The product ID.
     * @return bool True if has custom override, false otherwise.
     */
    private function eubaprdi_has_custom_price_override( int $product_id ): bool {
        $override_global = get_post_meta( $product_id, '_eubaprdi_override_global', true ) === 'yes';
        $custom_price = get_post_meta( $product_id, '_eubaprdi_custom_price', true );
        
        return $override_global && $custom_price !== '' && is_numeric( $custom_price );
    }

    /**
     * Get custom base price for product.
     *
     * @since 1.0.0
     * @param int $product_id The product ID.
     * @return array|false Custom base price data or false.
     */
    private function eubaprdi_get_custom_base_price( int $product_id ) {
        $custom_price = (float) get_post_meta( $product_id, '_eubaprdi_custom_price', true );
        
        if ( $custom_price < 0 ) {
            $custom_price = 0;
        }
        
        $display_unit = get_post_meta( $product_id, '_eubaprdi_display_unit', true );
        $valid_units = array_merge( array_keys( $this->conversion_rates['weight'] ), array_keys( $this->conversion_rates['volume'] ) );
        
        if ( empty( $display_unit ) || ! in_array( $display_unit, $valid_units, true ) ) {
            $display_unit = ! empty( $this->settings['display_unit'] ) && in_array( $this->settings['display_unit'], $valid_units, true ) 
                ? $this->settings['display_unit'] 
                : 'kg';
        }
        
        $decimal_places = isset( $this->settings['decimal_places'] ) ? absint( $this->settings['decimal_places'] ) : 2;
        
        return [
            'price' => $custom_price,
            'formatted_price' => number_format( 
                $custom_price, 
                $decimal_places, 
                wc_get_price_decimal_separator(), 
                wc_get_price_thousand_separator() 
            ),
            'unit' => apply_filters( 'eubaprdi_price_unit', $display_unit, null ),
            'raw_price' => $custom_price,
            'decimal_places' => $decimal_places,
            'is_custom' => true
        ];
    }

    /**
     * Calculate base price from product weight or dimensions.
     *
     * @since 1.0.0
     * @param WC_Product $product The product object.
     * @param float $price The product price.
     * @return array|false Base price data or false.
     */
    private function eubaprdi_calculate_base_price_from_product( WC_Product $product, float $price ) {
        $has_weight = $product->has_weight();
        $has_dimensions = $product->has_dimensions();
        
        if ( ! $has_weight && ! $has_dimensions ) {
            return false;
        }

        $display_unit = $this->settings['display_unit'] ?? '';
        
        if ( $has_weight ) {
            return $this->eubaprdi_calculate_weight_based_price( $product, $price, $display_unit );
        } elseif ( $has_dimensions ) {
            return $this->eubaprdi_calculate_volume_based_price( $product, $price, $display_unit );
        }
        
        return false;
    }

    /**
     * Calculate weight-based base price.
     *
     * @since 1.0.0
     * @param WC_Product $product The product object.
     * @param float $price The product price.
     * @param string $display_unit The display unit.
     * @return array|false Base price data or false.
     */
    private function eubaprdi_calculate_weight_based_price( WC_Product $product, float $price, string $display_unit ) {
        $weight = (float) $product->get_weight();
        $weight_unit = get_option( 'woocommerce_weight_unit', 'kg' );
        
        // Convert to base unit (kg)
        $weight_kg = $this->eubaprdi_convert_unit( $weight, $weight_unit, 'weight' );
        
        if ( $weight_kg <= 0 ) {
            return false;
        }
        
        // Set default display unit if not specified
        if ( empty( $display_unit ) || ! isset( $this->conversion_rates['weight'][ $display_unit ] ) ) {
            $display_unit = 'kg';
        }
        
        // Convert to display unit
        $weight_in_display_unit = $weight_kg / $this->conversion_rates['weight'][ $display_unit ];
        
        if ( $weight_in_display_unit <= 0 ) {
            return false;
        }
        
        $base_price = $price / $weight_in_display_unit;
        
        return $this->eubaprdi_format_base_price_result( $base_price, $display_unit );
    }

    /**
     * Calculate volume-based base price.
     *
     * @since 1.0.0
     * @param WC_Product $product The product object.
     * @param float $price The product price.
     * @param string $display_unit The display unit.
     * @return array|false Base price data or false.
     */
    private function eubaprdi_calculate_volume_based_price( WC_Product $product, float $price, string $display_unit ) {
        $length = (float) $product->get_length();
        $width = (float) $product->get_width();
        $height = (float) $product->get_height();
        $dimension_unit = get_option( 'woocommerce_dimension_unit', 'cm' );
        
        // Convert dimensions to cm
        $conversion_to_cm = $this->eubaprdi_get_dimension_conversion_rate( $dimension_unit );
        
        $length_cm = $length * $conversion_to_cm;
        $width_cm = $width * $conversion_to_cm;
        $height_cm = $height * $conversion_to_cm;
        
        // Calculate volume in liters (1L = 1000cm³)
        $volume_l = ( $length_cm * $width_cm * $height_cm ) / 1000;
        
        if ( $volume_l <= 0 ) {
            return false;
        }
        
        // Set default display unit if not specified
        if ( empty( $display_unit ) || ! isset( $this->conversion_rates['volume'][ $display_unit ] ) ) {
            $display_unit = 'l';
        }
        
        // Convert to display unit
        $volume_in_display_unit = $volume_l / $this->conversion_rates['volume'][ $display_unit ];
        
        if ( $volume_in_display_unit <= 0 ) {
            return false;
        }
        
        $base_price = $price / $volume_in_display_unit;
        
        return $this->eubaprdi_format_base_price_result( $base_price, $display_unit );
    }

    /**
     * Convert unit to base unit.
     *
     * @since 1.0.0
     * @param float $value The value to convert.
     * @param string $from_unit The unit to convert from.
     * @param string $type The type of unit (weight or volume).
     * @return float The converted value.
     */
    private function eubaprdi_convert_unit( float $value, string $from_unit, string $type ): float {
        $from_unit = strtolower( $from_unit );
        
        if ( ! isset( $this->conversion_rates[ $type ][ $from_unit ] ) ) {
            return $value;
        }
        
        return $value * $this->conversion_rates[ $type ][ $from_unit ];
    }

    /**
     * Get dimension conversion rate to cm.
     *
     * @since 1.0.0
     * @param string $dimension_unit The dimension unit.
     * @return float Conversion rate to cm.
     */
    private function eubaprdi_get_dimension_conversion_rate( string $dimension_unit ): float {
        $rates = [
            'cm' => 1,
            'm'  => 100,
            'mm' => 0.1,
            'in' => 2.54,
            'yd' => 91.44,
        ];
        
        return $rates[ $dimension_unit ] ?? 1;
    }

    /**
     * Format base price result.
     *
     * @since 1.0.0
     * @param float $base_price The calculated base price.
     * @param string $unit The unit.
     * @return array Formatted base price data.
     */
    private function eubaprdi_format_base_price_result( float $base_price, string $unit ): array {
        $decimal_places = isset( $this->settings['decimal_places'] ) ? absint( $this->settings['decimal_places'] ) : 2;
        
        $formatted_price = number_format( 
            $base_price, 
            $decimal_places, 
            wc_get_price_decimal_separator(), 
            wc_get_price_thousand_separator() 
        );

        return [
            'price' => $base_price,
            'formatted_price' => $formatted_price,
            'unit' => apply_filters( 'eubaprdi_price_unit', $unit, null ),
            'raw_price' => $base_price,
            'decimal_places' => $decimal_places,
        ];
    }

    /**
     * Get price label from settings.
     *
     * @since 1.0.0
     * @return string Price label.
     */
    private function eubaprdi_get_price_label(): string {
        $price_label = ! empty( $this->settings['price_label'] ) 
            ? $this->settings['price_label'] 
            : __( 'Base price', 'eu-base-price-display' );
            
        return $price_label;
    }

    /**
     * Format base price for display.
     *
     * @since 1.0.0
     * @param array $base_price_data The base price data.
     * @param WC_Product $product The product object.
     * @return string Formatted base price display.
     */
    private function eubaprdi_format_base_price_display( array $base_price_data, WC_Product $product ): string {
        if ( empty( $base_price_data['price'] ) ) {
            return '';
        }

        $price_label = $this->eubaprdi_get_price_label();
        $price_label = apply_filters( 'eubaprdi_price_label', $price_label, $product, $base_price_data );

        $formatted_price = wc_price( $base_price_data['price'] );
        $unit = $base_price_data['unit'] ?? '';

        if ( ! $formatted_price || ! $unit ) {
            return '';
        }

        $base_price_display = sprintf(
            '<span class="eubaprdi-base-price">%s %s / %s</span>',
            esc_html( $price_label ),
            $formatted_price,
            esc_html( $unit )
        );

        return apply_filters( 'eubaprdi_base_price_display', $base_price_display, $product, $base_price_data );
    }

    /**
     * Manage cache size to prevent memory issues.
     *
     * @since 1.0.0
     * @return void
     */
    private function eubaprdi_manage_cache_size(): void {
        if ( count( $this->base_price_cache ) > $this->max_cache_size ) {
            $this->base_price_cache = array_slice( 
                $this->base_price_cache, 
                -( (int) ( $this->max_cache_size * 0.8 ) ), // Keep 80% of max size
                null, 
                true 
            );
        }
    }

    /**
     * Clear the base price cache.
     *
     * @since 1.0.0
     * @return void
     */
    public function eubaprdi_clear_base_price_cache(): void {
        $this->base_price_cache = [];
    }

    /**
     * Get cache statistics.
     *
     * @since 1.0.0
     * @return array Cache statistics.
     */
    public function eubaprdi_get_cache_stats(): array {
        return [
            'size' => count( $this->base_price_cache ),
            'max_size' => $this->max_cache_size,
            'memory_usage' => strlen( serialize( $this->base_price_cache ) )
        ];
    }
}