<?php
/**
 * The admin-specific functionality of the plugin.
 *
 * @link       https://siddiqur.com
 * @since      1.0.0
 *
 * @package    EU_Base_Price_Display
 * @subpackage EU_Base_Price_Display/includes
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// These functions will be available by the time this class is instantiated
// as we're loading after init hook

/**
 * The admin-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and hooks for
 * enqueuing the admin-specific stylesheet and JavaScript.
 *
 * @package    EU_Base_Price_Display
 * @subpackage EU_Base_Price_Display/includes
 * @author     Md Siddiqur Rahman <support@siddiqur.com>
 */
class EUBAPRDI_Admin {

    /**
     * The ID of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private string $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0.0
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private string $version;

    /**
     * The plugin settings.
     *
     * @since    1.0.0
     * @access   private
     * @var      array    $settings    The plugin settings.
     */
    private array $settings = [];

    /**
     * Get plugin settings with defaults
     *
     * @since 1.0.0
     * @return array Plugin settings
     */
    public function eubaprdi_get_settings() {
        $defaults = array(
            'display_position' => 'after_price',
            'price_label' => __( 'Base price:', 'eu-base-price-display' ),
            'display_unit' => 'kg',
            'decimal_places' => 2
        );
        
        $settings = get_option( 'eubaprdi_settings', array() );
        
        // Ensure all settings exist with defaults
        return wp_parse_args( $settings, $defaults );
    }

    /**
     * Initialize the class and set its properties.
     *
     * @since 1.0.0
     * @param string $plugin_name The name of this plugin.
     * @param string $version     The version of this plugin.
     */
    public function __construct( string $plugin_name, string $version ) {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
        $this->settings = $this->eubaprdi_get_settings();
    }

    /**
     * Register all of the hooks related to the admin area functionality.
     *
     * @since 1.0.0
     * @return void
     */
    public function eubaprdi_init_hooks(): void {
        // Only register admin hooks if we're in admin area
        if ( ! is_admin() ) {
            return;
        }

        // Core admin functionality
        add_action( 'admin_enqueue_scripts', array( $this, 'eubaprdi_enqueue_admin_assets' ) );
        add_action( 'admin_menu', array( $this, 'eubaprdi_add_plugin_admin_menu' ) );
        add_action( 'admin_init', array( $this, 'eubaprdi_register_settings' ) );
        
        // Plugin page enhancements
        add_filter( 'plugin_action_links_' . plugin_basename( EUBAPRDI_PLUGIN_FILE ), array( $this, 'eubaprdi_add_action_links' ) );
        
        // WooCommerce product data integration (only if WooCommerce is active)
        if ( $this->eubaprdi_is_woocommerce_active() ) {
            add_filter( 'woocommerce_product_data_tabs', array( $this, 'eubaprdi_add_product_data_tab' ) );
            add_action( 'woocommerce_product_data_panels', array( $this, 'eubaprdi_add_product_data_fields' ) );
            add_action( 'woocommerce_process_product_meta', array( $this, 'eubaprdi_save_product_data_fields' ) );
        }
    }

    /**
     * Check if WooCommerce is active and available.
     *
     * @since 1.0.0
     * @return bool True if WooCommerce is active, false otherwise.
     */
    private function eubaprdi_is_woocommerce_active(): bool {
        return class_exists( 'WooCommerce' ) && function_exists( 'WC' );
    }

    /**
     * Register the stylesheets for the admin area.
     *
     * @since 1.0.0
     * @param string $hook The current admin page.
     * @return void
     */
    public function eubaprdi_enqueue_admin_assets( string $hook ): void {
        // Only load on our plugin's settings page and product edit page
        if ( 'woocommerce_page_eu-base-price-settings' !== $hook && 'product' !== get_post_type() ) {
            return;
        }

        // Enqueue admin styles
        wp_enqueue_style(
            $this->plugin_name . '-admin',
            EUBAPRDI_PLUGIN_URL . 'assets/css/admin.css',
            array(),
            $this->version,
            'all'
        );

        // Register and enqueue admin scripts
        wp_register_script(
            $this->plugin_name . '-admin',
            EUBAPRDI_PLUGIN_URL . 'assets/js/admin.js',
            array('jquery'),
            $this->version,
            true
        );
        
        // Localize script with any necessary data
        $script_data = array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce'    => wp_create_nonce($this->plugin_name . '-nonce')
        );
        
        wp_localize_script($this->plugin_name . '-admin', 'eubaprdi_admin_data', $script_data);
        wp_enqueue_script($this->plugin_name . '-admin');
    }

    /**
     * Add settings action link to the plugins page.
     *
     * @since 1.0.0
     * @param array $links Plugin action links.
     * @return array<string> Modified plugin action links.
     */
    public function eubaprdi_add_action_links( array $links ): array {
        // Build and escape the URL to the plugin's settings page
        $settings_url = add_query_arg(
            'page',
            'eu-base-price-settings',
            admin_url( 'admin.php' )
        );
        
        // Create the settings link
        $settings_link = array(
            sprintf(
                '<a href="%s">%s</a>',
                esc_url( $settings_url ),
                esc_html__( 'Settings', 'eu-base-price-display' )
            ),
            sprintf(
                '<a href="%s" target="_blank" rel="noopener noreferrer">%s</a>',
                esc_url( 'https://siddiqur.com/docs/eu-base-price-display' ),
                esc_html__( 'Documentation', 'eu-base-price-display' )
            )
        );
        
        // Add a link to the plugin review page
        if ( ! empty( $this->settings['review_link'] ) ) {
            $review_link_array = array(
                sprintf(
                    '<a href="%s" target="_blank" rel="noopener noreferrer" style="color:#ffa100;font-weight:bold;">%s</a>',
                    esc_url( $this->settings['review_link'] ),
                    esc_html__( '★ Rate this plugin', 'eu-base-price-display' )
                )
            );
            $settings_link = array_merge( $settings_link, $review_link_array );
        }
        
        return array_merge( $settings_link, $links );
    }

    /**
     * Add options page to WooCommerce menu.
     *
     * @since 1.0.0
     * @return void
     * @throws Exception If the menu page cannot be added.
     */
    public function eubaprdi_add_plugin_admin_menu(): void {
        add_submenu_page(
            'woocommerce',
            __( 'EU Base Price Display Settings', 'eu-base-price-display' ),
            __( 'EU Base Price', 'eu-base-price-display' ),
            'manage_woocommerce',
            'eu-base-price-settings',
            array( $this, 'eubaprdi_display_settings_page' )
        );
    }

    /**
     * Register the plugin settings.
     *
     * @since 1.0.0
     * @return void
     */
    public function eubaprdi_register_settings(): void {
        // Verify user capabilities
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            return;
        }

        // Register a setting and its sanitization callback
        register_setting(
            'eubaprdi_settings_group',
            'eubaprdi_settings',
            array( $this, 'eubaprdi_sanitize_settings' )
        );

        // Add settings section
        add_settings_section(
            'eubaprdi_general_section',
            __( 'General Settings', 'eu-base-price-display' ),
            array( $this, 'eubaprdi_section_callback' ),
            'eubaprdi_settings'
        );

        // Add price label field
        add_settings_field(
            'price_label',
            __( 'Price Label', 'eu-base-price-display' ),
            array( $this, 'eubaprdi_price_label_callback' ),
            'eubaprdi_settings',
            'eubaprdi_general_section'
        );

        add_settings_field(
            'display_unit',
            __( 'Display Unit', 'eu-base-price-display' ),
            array( $this, 'eubaprdi_display_unit_callback' ),
            'eubaprdi_settings',
            'eubaprdi_general_section',
            array(
                'label_for' => 'display_unit',
                'class'     => 'ebp-row',
                'description' => __( 'The unit to use when displaying the base price. If not set, the store\'s weight unit will be used.', 'eu-base-price-display' ),
            )
        );

        add_settings_field(
            'decimal_places',
            __( 'Decimal Places', 'eu-base-price-display' ),
            array( $this, 'eubaprdi_decimal_places_callback' ),
            'eubaprdi_settings',
            'eubaprdi_general_section',
            array(
                'label_for' => 'decimal_places',
                'class'     => 'ebp-row',
                'description' => __( 'Number of decimal places to show in the base price.', 'eu-base-price-display' ),
            )
        );
    }

    /**
     * Render the settings page.
     *
     * @since 1.0.0
     * @return void
     */
    public function eubaprdi_display_settings_page() {
        // Check user capabilities
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'eu-base-price-display' ) );
        }

        // Show error/update messages
        settings_errors( 'eubaprdi_messages' );
        ?>
        <div class="eubaprdi-settings-wrap">
            <div class="eubaprdi-settings-header">
                <h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
                <p><?php esc_html_e( 'Configure how the EU Base Price is displayed on your WooCommerce products.', 'eu-base-price-display' ); ?></p>
            </div>
            
            <div class="eubaprdi-settings-content">
                <form action="options.php" method="post" class="eubaprdi-settings-form">
                    <?php
                    // Output security nonce field
        wp_nonce_field( 'eubaprdi_save_settings', 'eubaprdi_settings_nonce' );
        
        // Output security fields
        settings_fields( 'eubaprdi_settings_group' );
        
        // Output setting sections and their fields with a wrapper
        echo '<div class="eubaprdi-settings-sections">';
        do_settings_sections( 'eubaprdi_settings' );
        echo '</div>';
                    
                    // Output save settings button with capability check
                    if ( current_user_can( 'manage_woocommerce' ) ) {
                        echo '<div class="eubaprdi-settings-actions">';
                        submit_button( 
                            __( 'Save Settings', 'eu-base-price-display' ),
                            'primary',
                            'submit',
                            false,
                            array( 'class' => 'eubaprdi-button eubaprdi-button-primary' )
                        );
                        echo '</div>';
                    }
                    ?>
                </form>
                
                <?php
                // Display additional help/information
                $this->eubaprdi_display_settings_help();
                ?>
            </div>
        </div>
        <?php
    }
    
    /**
     * Display additional help/information for the settings page.
     *
     * @since 1.0.0
     * @return void
     */
    private function eubaprdi_display_settings_help() {
        ?>
        <div class="card">
            <h2><?php esc_html_e( 'Need Help?', 'eu-base-price-display' ); ?></h2>
            <p><?php esc_html_e( 'Here are some tips for using the EU Base Price Display plugin:', 'eu-base-price-display' ); ?></p>
            <ul style="list-style-type: disc; margin-left: 20px;">
                <li><?php esc_html_e( 'The base price will be automatically calculated based on the product price and weight.', 'eu-base-price-display' ); ?></li>
                <li><?php esc_html_e( 'Make sure your products have weights set in the product data for the base price to display.', 'eu-base-price-display' ); ?></li>
            </ul>
            <p>
                <?php
                printf(
                    /* translators: %s: Documentation URL */
                    esc_html__( 'For more detailed documentation, please visit %s', 'eu-base-price-display' ),
                    '<a href="https://siddiqur.com/docs/eu-base-price-display" target="_blank">' . esc_html__( 'our documentation', 'eu-base-price-display' ) . '</a>'
                );
                ?>
            </p>
        </div>
        <?php
    }

    /**
     * Section callback function.
     *
     * @since 1.0.0
     * @param array $args Section arguments.
     * @return void
     */
    public function eubaprdi_section_callback( $args ) {
        // Validate input
        if ( ! is_array( $args ) || ! isset( $args['id'] ) ) {
            return;
        }

        switch ( $args['id'] ) {
            case 'eubaprdi_general_section':
                echo '<p>' . esc_html__( 'Configure how the base price is displayed on your site.', 'eu-base-price-display' ) . '</p>';
                break;
        }
    }

    /**
     * Price label field callback.
     *
     * @since 1.0.0
     * @return void
     */
    public function eubaprdi_price_label_callback() {
        $value = isset( $this->settings['price_label'] ) ? $this->settings['price_label'] : __( 'Base price:', 'eu-base-price-display' );
        ?>
        <input type="text" 
               id="eubaprdi_price_label" 
               name="eubaprdi_settings[price_label]" 
               value="<?php echo esc_attr( $value ); ?>" 
               class="regular-text">
        <p class="description"><?php esc_html_e( 'The label to display before the base price.', 'eu-base-price-display' ); ?></p>
        <?php
    }

    /**
     * Display unit field callback.
     *
     * @since 1.0.0
     * @param array $args Field arguments.
     * @return void
     */
    public function eubaprdi_display_unit_callback( $args ) {
        $value = isset( $this->settings['display_unit'] ) ? $this->settings['display_unit'] : '';
        $weight_unit = get_option( 'woocommerce_weight_unit', 'kg' );
        
        $weight_units = array(
            'g'   => __( 'Grams (g)', 'eu-base-price-display' ),
            'kg'  => __( 'Kilograms (kg)', 'eu-base-price-display' ),
            'lbs' => __( 'Pounds (lbs)', 'eu-base-price-display' ),
            'oz'  => __( 'Ounces (oz)', 'eu-base-price-display' ),
        );
        
        $volume_units = array(
            'ml'  => __( 'Milliliters (ml)', 'eu-base-price-display' ),
            'l'   => __( 'Liters (l)', 'eu-base-price-display' ),
            'm3'  => __( 'Cubic meters (m³)', 'eu-base-price-display' ),
            'cm3' => __( 'Cubic centimeters (cm³)', 'eu-base-price-display' ),
        );
        
        // Set default value to store's weight unit if not set
        if ( empty( $value ) ) {
            $value = $weight_unit;
        }
        ?>
        <select id="eubaprdi_display_unit" name="eubaprdi_settings[display_unit]" class="regular-text">
            <option value=""><?php esc_html_e( 'Auto (use store default)', 'eu-base-price-display' ); ?></option>
            
            <optgroup label="<?php esc_attr_e( 'Weight Units', 'eu-base-price-display' ); ?>">
                <?php foreach ( $weight_units as $unit => $label ) : ?>
                    <option value="<?php echo esc_attr( $unit ); ?>" <?php selected( $value, $unit ); ?>>
                        <?php echo esc_html( $label ); ?>
                    </option>
                <?php endforeach; ?>
            </optgroup>
            
            <optgroup label="<?php esc_attr_e( 'Volume Units', 'eu-base-price-display' ); ?>">
                <?php foreach ( $volume_units as $unit => $label ) : ?>
                    <option value="<?php echo esc_attr( $unit ); ?>" <?php selected( $value, $unit ); ?>>
                        <?php echo esc_html( $label ); ?>
                    </option>
                <?php endforeach; ?>
            </optgroup>
        </select>
        <p class="description"><?php echo esc_html( $args['description'] ); ?></p>
        <?php
    }

    /**
     * Decimal places field callback.
     *
     * @since 1.0.0
     * @param array $args Field arguments.
     * @return void
     */
    public function eubaprdi_decimal_places_callback( $args ) {
        $value = isset( $this->settings['decimal_places'] ) ? $this->settings['decimal_places'] : 2;
        ?>
        <input type="number" 
               id="eubaprdi_decimal_places" 
               name="eubaprdi_settings[decimal_places]" 
               value="<?php echo esc_attr( $value ); ?>" 
               min="0" 
               max="6" 
               step="1" 
               class="small-text">
        <p class="description"><?php echo esc_html( $args['description'] ); ?></p>
        <?php
    }

    /**
     * Add custom product data tab for EU Base Price settings.
     *
     * @since 1.0.0
     * @param array $tabs Existing product data tabs.
     * @return array Modified product data tabs.
     */
    public function eubaprdi_add_product_data_tab( $tabs ) {
        $tabs['eubaprdi_base_price'] = array(
            'label'    => __( 'EU Base Price', 'eu-base-price-display' ),
            'target'   => 'eubaprdi_base_price_options',
            'class'    => array( 'show_if_simple', 'show_if_variable' ),
            'priority' => 80,
        );
        return $tabs;
    }

    /**
     * Add fields to the product data panel.
     *
     * @since 1.0.0
     * @return void
     */
    public function eubaprdi_add_product_data_fields() {
        
        echo '<div id="eubaprdi_base_price_options" class="panel woocommerce_options_panel">';
        
        woocommerce_wp_checkbox( array(
            'id'          => '_eubaprdi_override_global',
            'label'       => __( 'Override Global Settings', 'eu-base-price-display' ),
            'description' => __( 'Enable to override global base price settings for this product', 'eu-base-price-display' ),
            'desc_tip'    => true,
            'wrapper_class' => 'eubaprdi-toggle-field',
        ) );
        
        echo '<div class="options_group eubaprdi-product-fields">';
        
        woocommerce_wp_select( array(
            'id'          => '_eubaprdi_display_unit',
            'label'       => __( 'Display Unit', 'eu-base-price-display' ),
            'description' => __( 'Select the unit to use for base price calculation', 'eu-base-price-display' ),
            'desc_tip'    => true,
            'options'     => array(
                ''      => __( 'Use Global Setting', 'eu-base-price-display' ),
                'kg'    => __( 'Kilogram (kg)', 'eu-base-price-display' ),
                'g'     => __( 'Gram (g)', 'eu-base-price-display' ),
                'lbs'   => __( 'Pound (lbs)', 'eu-base-price-display' ),
                'oz'    => __( 'Ounce (oz)', 'eu-base-price-display' ),
                'l'     => __( 'Liter (L)', 'eu-base-price-display' ),
                'ml'    => __( 'Milliliter (ml)', 'eu-base-price-display' ),
                'm3'    => __( 'Cubic meter (m³)', 'eu-base-price-display' ),
                'cm3'   => __( 'Cubic centimeter (cm³)', 'eu-base-price-display' ),
            ),
        ) );
        
        woocommerce_wp_text_input( array(
            'id'          => '_eubaprdi_custom_price',
            'label'       => __( 'Custom Base Price', 'eu-base-price-display' ),
            'description' => __( 'Enter a custom base price (leave empty to calculate automatically)', 'eu-base-price-display' ),
            'desc_tip'    => true,
            'type'        => 'number',
            'custom_attributes' => array(
                'step' => '0.0001',
                'min'  => '0',
            ),
        ) );
        
        echo '</div>'; // Close .options_group
        echo '</div>'; // Close #eu_base_price_options
    }
    
    /**
     * Save product data fields.
     *
     * @since 1.0.0
     * @param int $post_id The post ID.
     * @return void
     */
    public function eubaprdi_save_product_data_fields( $post_id ) {
        // Validate post ID
        if ( ! is_numeric( $post_id ) || $post_id <= 0 ) {
            return;
        }

        // Verify nonce - using WooCommerce's nonce verification
        if ( ! isset( $_POST['woocommerce_meta_nonce'] ) || 
             ! wp_verify_nonce( sanitize_key( $_POST['woocommerce_meta_nonce'] ), 'woocommerce_save_data' ) ) {
            return;
        }

        // Check user has permissions to edit products
        if ( ! current_user_can( 'edit_product', $post_id ) ) {
            return;
        }

        // Check if this is an autosave
        if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
            return;
        }

        // Validate POST data exists
        if ( ! isset( $_POST['_eubaprdi_override_global'] ) && 
             ! isset( $_POST['_eubaprdi_display_unit'] ) && 
             ! isset( $_POST['_eubaprdi_custom_price'] ) ) {
            return;
        }

        // Save override setting
        $override_global = isset( $_POST['_eubaprdi_override_global'] ) ? 'yes' : 'no';
        update_post_meta( $post_id, '_eubaprdi_override_global', $override_global );
        
        // Only save other fields if override is enabled
        if ( 'yes' === $override_global ) {
            // Save display unit
            if ( isset( $_POST['_eubaprdi_display_unit'] ) ) {
                $display_unit = sanitize_text_field( wp_unslash( $_POST['_eubaprdi_display_unit'] ) );
                // Validate display unit
                $valid_units = array( '', 'kg', 'g', 'lbs', 'oz', 'l', 'ml' );
                if ( in_array( $display_unit, $valid_units, true ) ) {
                    update_post_meta( $post_id, '_eubaprdi_display_unit', $display_unit );
                }
            }
            
            // Save custom price
            if ( isset( $_POST['_eubaprdi_custom_price'] ) && '' !== $_POST['_eubaprdi_custom_price'] ) {
                // Sanitize and validate the input
                $raw_price = sanitize_text_field( wp_unslash( $_POST['_eubaprdi_custom_price'] ) );
                $numeric_price = wc_format_decimal( $raw_price );
                
                // Additional validation to ensure it's a valid price
                if ( is_numeric( $numeric_price ) && $numeric_price >= 0 ) {
                    update_post_meta( $post_id, '_eubaprdi_custom_price', $numeric_price );
                }
            } else {
                delete_post_meta( $post_id, '_eubaprdi_custom_price' );
            }
        } else {
            // If override is disabled, clean up the meta
            delete_post_meta( $post_id, '_eubaprdi_display_unit' );
            delete_post_meta( $post_id, '_eubaprdi_custom_price' );
        }
    }
    
    /**
     * Sanitize the settings before they're saved.
     *
     * @since 1.0.0
     * @param array $input The settings to sanitize.
     * @return array The sanitized settings.
     */
    public function eubaprdi_sanitize_settings( $input ) {
        // Verify nonce for security
        if ( ! isset( $_POST['eubaprdi_settings_nonce'] ) ||
             ! wp_verify_nonce( sanitize_key( $_POST['eubaprdi_settings_nonce'] ), 'eubaprdi_save_settings' ) ) {
            wp_die( esc_html__( 'Security check failed.', 'eu-base-price-display' ) );
        }

        // Verify user capabilities
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_die( esc_html__( 'You do not have sufficient permissions to access this page.', 'eu-base-price-display' ) );
        }

        // Validate input is an array
        if ( ! is_array( $input ) ) {
            $input = array();
        }

        $sanitized = array();

        // Sanitize price label
        $sanitized['price_label'] = isset( $input['price_label'] )
            ? sanitize_text_field( $input['price_label'] )
            : __( 'Base price:', 'eu-base-price-display' );

        // Sanitize display unit
        $valid_units = array( 'g', 'kg', 'lbs', 'oz', 'ml', 'l', 'm3', 'cm3', '' );
        $sanitized['display_unit'] = isset( $input['display_unit'] ) && in_array( $input['display_unit'], $valid_units, true )
            ? sanitize_text_field( $input['display_unit'] )
            : '';

        // Sanitize decimal places
        $sanitized['decimal_places'] = isset( $input['decimal_places'] )
            ? absint( $input['decimal_places'] )
            : 2;

        // Ensure decimal places is between 0 and 6
        $sanitized['decimal_places'] = min( max( $sanitized['decimal_places'], 0 ), 6 );

        // Add settings saved message
        add_settings_error(
            'eubaprdi_messages',
            'eubaprdi_message',
            __( 'Settings Saved', 'eu-base-price-display' ),
            'updated'
        );

        // Allow other plugins to modify the sanitized settings
        return apply_filters( 'eubaprdi_sanitize_settings', $sanitized, $input );
    }
}