<?php
/**
 * Plugin Name:       EU Base Price Display for WooCommerce
 * Plugin URI:        https://wordpress.org/plugins/eu-base-price-display
 * Description:       Displays the base price (€/kg, €/L) for WooCommerce products to comply with EU price labeling regulations.
 * Version:           1.0.0
 * Requires at least: 5.6
 * Requires PHP:      7.4
 * Author:            Md Siddiqur Rahman
 * Author URI:        https://siddiqur.com
 * License:           GPL-2.0+
 * License URI:       http://www.gnu.org/licenses/gpl-2.0.txt
 * Text Domain:       eu-base-price-display
 * Domain Path:       /languages
 * WC requires at least: 5.0
 * WC tested up to:   8.0
 *
 * @package    EU_Base_Price_Display
 *
 * @wordpress-plugin
 * Copyright © 2025 Md Siddiqur Rahman (email: support@siddiqur.com)
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly
}

// Define plugin constants.
if ( ! defined( 'EUBAPRDI_VERSION' ) ) {
    define( 'EUBAPRDI_VERSION', '1.0.0' );
}

if ( ! defined( 'EUBAPRDI_PLUGIN_FILE' ) ) {
    define( 'EUBAPRDI_PLUGIN_FILE', __FILE__ );
}

if ( ! defined( 'EUBAPRDI_PLUGIN_DIR' ) ) {
    define( 'EUBAPRDI_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
}

if ( ! defined( 'EUBAPRDI_PLUGIN_URL' ) ) {
    define( 'EUBAPRDI_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
}

if ( ! defined( 'EUBAPRDI_PLUGIN_BASENAME' ) ) {
    define( 'EUBAPRDI_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );
}

if ( ! defined( 'EUBAPRDI_PLUGIN_NAME' ) ) {
    define( 'EUBAPRDI_PLUGIN_NAME', 'eu-base-price-display' );
}

// Initialize plugin after WP is fully loaded
add_action( 'wp_loaded', 'eubaprdi_run_plugin' );

/**
 * Declare HPOS compatibility.
 *
 * @since 1.0.0
 */
add_action( 'before_woocommerce_init', function() {
    if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', __FILE__, true );
    }
} );

/**
 * Check if WooCommerce is active.
 *
 * @since 1.0.0
 * @return bool True if WooCommerce is active, false otherwise.
 */
function eubaprdi_is_woocommerce_active(): bool {
    return in_array( 'woocommerce/woocommerce.php', (array) get_option( 'active_plugins', array() ), true ) || 
           ( function_exists( 'is_multisite' ) && is_multisite() && array_key_exists( 'woocommerce/woocommerce.php', (array) get_site_option( 'active_sitewide_plugins', array() ) ) );
}

/**
 * Display an admin notice if WooCommerce is not active.
 *
 * @since 1.0.0
 * @return void
 */
function eubaprdi_woocommerce_missing_notice(): void {
    ?>
    <div class="notice notice-error">
        <p><?php echo esc_html( 'EU Base Price Display requires WooCommerce to be installed and active.' ); ?></p>
    </div>
    <?php
}

/**
 * Initialize the plugin.
 *
 * @since 1.0.0
 * @return void
 */
function eubaprdi_run_plugin(): void {
    // Ensure WordPress is fully loaded
    if ( ! function_exists( 'get_locale' ) ) {
        return;
    }

    // Check if WooCommerce is active.
    if ( ! eubaprdi_is_woocommerce_active() ) {
        // Hook the notice to init to ensure translations are ready
        add_action( 'init', function() {
            add_action( 'admin_notices', 'eubaprdi_woocommerce_missing_notice' );
        });
        return;
    }
    
    try {
        // Load and run the loader class
        require_once EUBAPRDI_PLUGIN_DIR . 'includes/class-eubaprdi-loader.php';

        $plugin_loader = new EUBAPRDI_Loader();
        $plugin_loader->eubaprdi_run();
    } catch ( Exception $e ) {
        // Error is caught and not propagated to avoid white screens
        // In a production environment, we don't log or display errors to users
    }
}
// Register activation and deactivation hooks
register_activation_hook( __FILE__, 'eubaprdi_activate' );
register_uninstall_hook( __FILE__, 'eubaprdi_uninstall' );

/**
 * Plugin activation function.
 *
 * @since 1.0.0
 * @return void
 */
function eubaprdi_activate() {
    // Add any activation code here if needed in the future
    if ( ! get_option( 'eubaprdi_installed' ) ) {
        update_option( 'eubaprdi_installed', time() );
    }
    update_option( 'eubaprdi_version', EUBAPRDI_VERSION );
}

/**
 * Plugin uninstallation function.
 *
 * @since 1.0.0
 * @return void
 */
function eubaprdi_uninstall(): void {
    // Delete options
    delete_option( 'eubaprdi_settings' );
    delete_option( 'eubaprdi_installed' );
    delete_option( 'eubaprdi_version' );
}
