<?php

/**
 * Viewed Products Manager for Enhanced Ajax WooCommerce
 *
 * @package EAW
 * @since 1.0.0
 */
namespace CodeConfig\EAW\App\CartManager;

use CodeConfig\EAW\AppSettings;
use CodeConfig\EAW\Utils\Singleton;
defined( 'ABSPATH' ) || exit( 'No direct script access allowed' );
/**
 * Class ViewedProducts
 *
 * Manages the display of recently viewed products with Slick Slider integration.
 */
class ViewedProducts {
    use Singleton;
    /**
     * Settings instance.
     *
     * @var AppSettings
     */
    protected $settings;

    /**
     * Initialize hooks.
     *
     * @return void
     */
    public function doHooks() : void {
        $this->settings = AppSettings::getInstance();
        if ( wp_is_block_theme() ) {
            return;
        }
        if ( !$this->settings->getParams( 'vpEnable' ) ) {
            return;
        }
        add_action( 'woocommerce_product_after_tabs', [$this, 'renderViewedProducts'], 10 );
        add_action( 'wp_head', [$this, 'outputInlineCSS'] );
    }

    /**
     * Render the viewed products container.
     *
     * @return void
     */
    public function renderViewedProducts() : void {
        echo esc_html( $this->getViewedProductsHTML() );
    }

    /**
     * Get the viewed products wrapper HTML.
     *
     * @return string
     */
    private function getViewedProductsHTML() : string {
        ob_start();
        printf( '<div class="eaw-viewed-products-wrapper" data-current_product_id="%s">
                <div class="eaw-viewed-products-title">
                    <h2>%s</h2>
                </div>
                <div class="eaw-viewed-products" id="eaw-viewed-products"></div>
            </div>', esc_attr( get_the_ID() ), esc_html( $this->settings->getParams( 'vpTitle' ) ) );
        return ob_get_clean();
    }

    /**
     * Output inline CSS for styling.
     *
     * @return void
     */
    public function outputInlineCSS() : void {
        $settings = [
            'rvpBGColor'               => $this->settings->getParams( 'rvpBGColor' ),
            'rvpActiveBGColor'         => $this->settings->getParams( 'rvpActiveBGColor' ),
            'rvpArrowIconColor'        => $this->settings->getParams( 'rvpArrowIconColor' ),
            'rvpArrowIconHoverColor'   => $this->settings->getParams( 'rvpArrowIconHoverColor' ),
            'rvpSliderNavBorderRadius' => $this->settings->getParams( 'rvpSliderNavBorderRadius' ),
        ];
        printf(
            '<style>:root {
                --rvpBGColor: %s;
                --rvpActiveBGColor: %s;
                --rvpArrowIconColor: %s;
                --rvpArrowIconHoverColor: %s;
                --rvpSliderNavBorderRadius: %s;
            }</style>',
            esc_attr( $settings['rvpBGColor'] ),
            esc_attr( $settings['rvpActiveBGColor'] ),
            esc_attr( $settings['rvpArrowIconColor'] ),
            esc_attr( $settings['rvpArrowIconHoverColor'] ),
            esc_attr( $settings['rvpSliderNavBorderRadius'] )
        );
    }

}
